/* Tienda Básica – JS unificado y corregido */
/* Versión refactorizada - Elimina duplicaciones y soluciona problemas de inicialización */

// =============================================
// CONFIGURACIÓN GLOBAL Y UTILIDADES
// =============================================

// Función fetchJSON global
window.fetchJSON = window.fetchJSON || function(url, opts) {
    return fetch(url, opts).then(function (r) {
        if (!r.ok) throw new Error("HTTP " + r.status);
        return r.json();
    });
};

// Función para mostrar carga
function showLoading(button) {
    const originalText = button.innerHTML;
    button.innerHTML = 'Procesando...';
    button.disabled = true;
    return function() {
        button.innerHTML = originalText;
        button.disabled = false;
    };
}

// Función para manejar errores
function handleError(error, button) {
    console.error('Error:', error);
    alert('Error: ' + (error.message || 'No se pudo completar la acción'));
    if (button && button.hideLoading) button.hideLoading();
}

// =============================================
// PAYPAL SDK - Carga robusta
// =============================================

function ensurePayPalSdk() {
    return new Promise((resolve, reject) => {
        // Si ya está cargado y tiene buttons, usarlo directamente
        if (window.paypal && window.paypal.Buttons) {
            console.log('PayPal SDK ya cargado y listo');
            return resolve(true);
        }

        // Si hay SDK pero sin buttons, esperar
        if (window.paypal && !window.paypal.Buttons) {
            console.log('PayPal cargado pero sin Buttons, esperando...');
            const waitForButtons = setInterval(() => {
                if (window.paypal && window.paypal.Buttons) {
                    clearInterval(waitForButtons);
                    resolve(true);
                }
            }, 100);
            setTimeout(() => {
                clearInterval(waitForButtons);
                reject(new Error('Timeout esperando PayPal Buttons'));
            }, 5000);
            return;
        }

        // Necesitamos client-id
        const cid = window.tbAjax?.paypal_client;
        if (!cid) {
            console.warn('PayPal client-id ausente');
            return resolve(false);
        }

        // Limpiar scripts existentes de PayPal
        const paypalScripts = document.querySelectorAll('script[src*="paypal.com/sdk/js"]');
        paypalScripts.forEach(script => script.remove());

        // Construir URL
        const currency = (window.tbAjax?.currency || 'EUR').toUpperCase();
        const url = 'https://www.paypal.com/sdk/js' +
            '?client-id=' + encodeURIComponent(cid) +
            '&currency=' + encodeURIComponent(currency) +
            '&intent=capture' +
            '&components=buttons';

        // Inyectar SDK
        const tag = document.createElement('script');
        tag.src = url;
        tag.async = true;
        tag.onload = () => {
            console.log('PayPal SDK cargado exitosamente');
            // Pequeño delay para asegurar que Buttons esté disponible
            setTimeout(() => resolve(true), 100);
        };
        tag.onerror = () => {
            console.error('Error cargando PayPal SDK');
            reject(new Error('No se pudo cargar PayPal SDK'));
        };
        document.head.appendChild(tag);
    });
}


window.__shoplite_pp_rendered = window.__shoplite_pp_rendered || false;


// Flag global para evitar re-renderizar PayPal
window.__shoplite_pp_rendered = window.__shoplite_pp_rendered || false;

// =============================================
// SISTEMA DE PAGOS UNIFICADO
// =============================================

function setupPaymentButtons() {
    try {
        console.log('Inicializando botones de pago...');

        // =========================
        // PAYPAL
        // =========================
        const ppContainer = document.getElementById("tb-paypal-container");

        if (ppContainer && window.paypal && window.paypal.Buttons) {

            // ✅ Re-render controlado:
            // Si el checkout se ha re-renderizado (actualizar cantidades/cupón) y el contenedor vuelve vacío,
            // volvemos a renderizar PayPal. Si ya hay markup PayPal dentro, no hacemos nada.
            const hasPaypalMarkup =
                ppContainer.querySelector('iframe') ||
                ppContainer.querySelector('[data-funding-source]') ||
                ppContainer.querySelector('.paypal-buttons') ||
                ppContainer.children.length > 0;

            if (hasPaypalMarkup) {
                console.log('PayPal ya está presente en el contenedor, no se reinicia');
            } else {
                console.log('Configurando botón PayPal...');

                const paypalNonce = window.paypalNonce || (window.tbAjax && (window.tbAjax.paypalNonce || window.tbAjax.nonce_pp)) || "";
                const ajaxUrlPay = window.ajaxUrl || (window.tbAjax && window.tbAjax.ajax_url) || "/wp-admin/admin-ajax.php";
                const successUrl = window.successUrl || (window.tbAjax && (window.tbAjax.successUrl || window.tbAjax.success_url)) || "/";
                const cancelUrl  = window.cancelUrl  || (window.tbAjax && (window.tbAjax.cancelUrl  || window.tbAjax.cancel_url))  || "/";

                paypal.Buttons({
                    createOrder: function (data, actions) {
                        var body = new URLSearchParams();
                        body.append("action", "tb_paypal_create_order");
                        body.append("nonce", paypalNonce);

                        if (window.tbAjax && tbAjax.coupon_code) {
                            body.append("coupon_code", tbAjax.coupon_code);
                            console.log("[JS] Enviando cupón:", tbAjax.coupon_code);
                        }

                        var couponField = document.querySelector('input[name="shoplite_coupon_code"]');
                        if (couponField && couponField.value) {
                            body.append("coupon_code", couponField.value);
                            console.log('Enviando coupon_code a PayPal:', couponField.value);
                        }

                        return fetchJSON(ajaxUrlPay, {
                            method: "POST",
                            credentials: "same-origin",
                            headers: { "Content-Type": "application/x-www-form-urlencoded" },
                            body: body.toString()
                        }).then(function (json) {
                            if (json && json.success && json.data && json.data.id) {
                                console.log('Orden PayPal creada:', json.data.id);
                                return json.data.id;
                            }
                            var msg = (json && json.data && json.data.message)
                                ? json.data.message
                                : "No se pudo crear el pedido PayPal.";
                            throw new Error(msg);
                        });
                    },

                    onApprove: function (data) {
                        console.log('PayPal aprobado, capturando orden:', data.orderID);

                        var body = new URLSearchParams();
                        body.append("action", "tb_paypal_capture_order");
                        body.append("nonce", paypalNonce);
                        body.append("order_id", data.orderID);

                        return fetchJSON(ajaxUrlPay, {
                            method: "POST",
                            credentials: "same-origin",
                            headers: { "Content-Type": "application/x-www-form-urlencoded" },
                            body: body.toString()
                        }).then(function (json) {
                            if (json && json.success) {
                                console.log('Pago PayPal capturado, redirigiendo...');
                                window.location.href = successUrl;
                            } else {
                                alert((json && json.data && json.data.message)
                                    ? json.data.message
                                    : "No se pudo capturar el pago PayPal.");
                            }
                        });
                    },

                    onCancel: function () {
                        console.log('PayPal cancelado por usuario');
                        if (cancelUrl) window.location.href = cancelUrl;
                    },

                    onError: function (err) {
                        console.error("Error PayPal:", err);
                        alert("Error en PayPal: " + (err && err.message ? err.message : "indeterminado"));
                    }

                }).render("#tb-paypal-container");

                console.log('Botón PayPal renderizado');
            }

        } else if (ppContainer) {
            console.warn('Contenedor PayPal encontrado pero SDK no disponible');
        }

        // =========================
        // STRIPE
        // =========================
        const stripeBtn = document.getElementById("tb-pagar-stripe");
        const bizumBtn  = document.getElementById('tb-pagar-bizum'); // se oculta abajo (Bizum deshabilitado)
        const stripePub = (window.tbAjax && (window.tbAjax.stripe_pub || window.tbAjax.stripePub)) || "";
        const ajaxUrl   = window.ajaxUrl || (window.tbAjax && window.tbAjax.ajax_url) || "/wp-admin/admin-ajax.php";

        if (stripeBtn && typeof Stripe !== "undefined" && stripePub) {
            console.log('Configurando botón Stripe...');
            const newStripeBtn = stripeBtn.cloneNode(true);
            stripeBtn.parentNode.replaceChild(newStripeBtn, stripeBtn);

            const stripe = Stripe(stripePub);
            newStripeBtn.addEventListener("click", function (e) {
                e.preventDefault();
                const hideLoading = showLoading(this);

                var body = new URLSearchParams();
                body.append("action", "tb_stripe_create_checkout");
                body.append("nonce", newStripeBtn.getAttribute("data-nonce"));

                fetchJSON(ajaxUrl, {
                    method: "POST",
                    credentials: "same-origin",
                    headers: { "Content-Type": "application/x-www-form-urlencoded" },
                    body: body.toString()
                })
                .then(function (json) {
                    if (json && json.success && json.data && json.data.id) {
                        return stripe.redirectToCheckout({ sessionId: json.data.id });
                    }
                    throw new Error(json?.data?.message || "No se pudo iniciar el pago con Stripe.");
                })
                .catch(function () {
                    alert("No se pudo iniciar el pago con Stripe.");
                })
                .finally(hideLoading);
            });
        }

        // =========================
        // BIZUM (DESHABILITADO / OCULTO)
        // =========================
        if (bizumBtn) {
            bizumBtn.style.display = 'none';
            console.log('Botón Bizum ocultado');
        }

    } catch (e) {
        console.error('Error en setupPaymentButtons:', e);
    }
}



// Función para reinicializar pagos después de actualizar carrito
function reinitPayments() {
    console.log('Reinicializando sistema de pagos...');
    
    // Primero Stripe (más rápido, no depende de SDK externo)
    setTimeout(() => {
        try {
            setupPaymentButtons();
        } catch (e) {
            console.error('Error en setupPaymentButtons durante reinicio:', e);
        }
    }, 100);
    
    // Luego PayPal con manejo robusto
    ensurePayPalSdk()
        .then((loaded) => {
            if (loaded) {
                console.log('PayPal SDK listo después de reinicio');
                setTimeout(() => {
                    setupPaymentButtons();
                }, 300);
            } else {
                console.warn('PayPal SDK no disponible después de reinicio');
            }
        })
        .catch(err => {
            console.error('Error cargando PayPal SDK durante reinicio:', err);
        });
}

// =============================================
// SISTEMA DE CARRITO AJAX
// =============================================

function initializeCartEvents() {
    const cartForm = document.querySelector('.carrito-form');
    if (!cartForm) return;

    // 1. ACTUALIZAR CARRITO (botón "Actualizar cantidades")
    const updateBtn = cartForm.querySelector('button[name="update"]');
    if (updateBtn) {
        // Eliminar event listeners antiguos
        const newUpdateBtn = updateBtn.cloneNode(true);
        updateBtn.parentNode.replaceChild(newUpdateBtn, updateBtn);
        
        newUpdateBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            const hideLoading = showLoading(this);
            const quantities = {};
            
            // Recoger todas las cantidades
            const qtyInputs = cartForm.querySelectorAll('input[name^="qty["]');
            qtyInputs.forEach(input => {
                const match = input.name.match(/qty\[([^\]]+)\]/);
                if (match) {
                    quantities[match[1]] = parseInt(input.value) || 1;
                }
            });

            handleCartAction('update', { quantities: JSON.stringify(quantities) })
                .then(response => {
                    updateCartUI(response);
                    hideLoading();
                })
                .catch(error => handleError(error, { hideLoading }));
        });
    }

    // 2. ELIMINAR PRODUCTO (botones "Borrar")
    const removeBtns = cartForm.querySelectorAll('button[name="remove"]');
    removeBtns.forEach(btn => {
        // Clonar y reemplazar para eliminar event listeners antiguos
        const newBtn = btn.cloneNode(true);
        btn.parentNode.replaceChild(newBtn, btn);
        
        newBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            const hideLoading = showLoading(this);
            const itemKey = this.value;

            handleCartAction('remove', { item_key: itemKey })
                .then(response => {
                    updateCartUI(response);
                    hideLoading();
                })
                .catch(error => handleError(error, { hideLoading }));
        });
    });

    // 3. VACIAR CARRITO (botón "Vaciar carrito")
    const emptyBtn = cartForm.querySelector('button[name="empty"]');
    if (emptyBtn) {
        // Clonar y reemplazar para eliminar event listeners antiguos
        const newEmptyBtn = emptyBtn.cloneNode(true);
        emptyBtn.parentNode.replaceChild(newEmptyBtn, emptyBtn);
        
        newEmptyBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            if (!confirm('¿Estás seguro de que quieres vaciar el carrito?')) {
                return;
            }

            const hideLoading = showLoading(this);
            
            handleCartAction('empty')
                .then(response => {
                    updateCartUI(response);
                    hideLoading();
                })
                .catch(error => handleError(error, { hideLoading }));
        });
    }
}

// Función para manejar acciones del carrito
function handleCartAction(action, data = {}) {
    const ajaxUrl = window.ajaxUrl || (window.tbAjax && window.tbAjax.ajax_url) || '/wp-admin/admin-ajax.php';
    
    return fetch(ajaxUrl, {
        method: 'POST',
        credentials: 'same-origin',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: new URLSearchParams({
            action: 'shoplite_cart_action',
            nonce: window.tbAjax?.cartNonce || '',
            cart_action: action,
            ...data
        })
    })
    .then(async (r) => {
        const text = await r.text();
        let json = null;
        try { json = JSON.parse(text); } catch(e) {}
        if (!r.ok) {
            const msg = json?.data?.message || json?.message || text || ('HTTP ' + r.status);
            throw new Error(msg.trim());
        }
        return json || {};
    });
}



// ============================================================
// Shoplite: capture customer email early (abandoned carts + prefill)
// ============================================================
(function () {
  const ajaxUrl = window.ajaxUrl || (window.tbAjax && window.tbAjax.ajax_url) || '/wp-admin/admin-ajax.php';

  function postSaveEmail(email) {
    return fetch(ajaxUrl, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
      body: new URLSearchParams({
        action: 'shoplite_save_customer_email',
        email: email
      })
    }).catch(() => {});
  }

  function debounce(fn, wait) {
    let t = null;
    return function (...args) {
      clearTimeout(t);
      t = setTimeout(() => fn.apply(this, args), wait);
    };
  }

  function findEmailInput() {
    return (
      document.querySelector('#shoplite-email') ||
      document.querySelector('input[type="email"]') ||
      document.querySelector('input[name="email"]') ||
      document.querySelector('input[name="customer_email"]')
    );
  }

  const send = debounce(() => {
    const input = findEmailInput();
    if (!input) return;
    const email = (input.value || '').trim();
    if (!email) return;
    postSaveEmail(email);
  }, 700);

  function attach() {
    const input = findEmailInput();
    if (!input) return;
    input.addEventListener('input', send);
    input.addEventListener('change', send);
    input.addEventListener('blur', send);
  }

  document.addEventListener('DOMContentLoaded', attach);
  setTimeout(attach, 500);
  setTimeout(attach, 1500);
})();





// Función para actualizar la interfaz del carrito
function updateCartUI(response) {
    if (response.success && response.data && response.data.html) {
        console.log('Actualizando interfaz del carrito...');
        
        // Reemplazar todo el contenido del carrito
        const newCart = document.createElement('div');
        newCart.innerHTML = response.data.html;
        const currentCart = document.querySelector('.tienda-carrito');
        
        if (currentCart && newCart.querySelector('.tienda-carrito')) {
            currentCart.parentNode.replaceChild(newCart.querySelector('.tienda-carrito'), currentCart);
            
            // Re-inicializar event listeners con delay para asegurar que el DOM esté listo
            setTimeout(() => {
                initializeCartEvents();
                reinitPayments(); // ← ¡CRUCIAL! Reinicializar pagos después de actualizar carrito
            }, 200);
        }
    }
}

// =============================================
// AÑADIR AL CARRITO DESDE PÁGINA DE PRODUCTO
// =============================================

function initializeAddToCartForms() {
    const addToCartForms = document.querySelectorAll('form.tb-add-to-cart-form');
    
    addToCartForms.forEach(form => {
        const addToCartBtn = form.querySelector('.tb-add-to-cart-btn');
        if (!addToCartBtn) return;

        addToCartBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            const hideLoading = showLoading(this);
            const formData = new FormData(form);
            const productData = {};
            const ajaxUrl = window.ajaxUrl || (window.tbAjax && window.tbAjax.ajax_url) || '/wp-admin/admin-ajax.php';

            // Convertir FormData a objeto
            for (let [key, value] of formData.entries()) {
                productData[key] = value;
            }

            // Enviar por AJAX
            fetch(ajaxUrl, {
                method: 'POST',
                credentials: 'same-origin',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams({
                    action: 'shoplite_add_to_cart',
                    nonce: window.tbAjax?.cartNonce || '',
                    ...productData
                })
            })
            .then(async (r) => {
                const text = await r.text();
                let json = null; 
                try { json = JSON.parse(text); } catch(e) {}
                if (!r.ok) { 
                    throw new Error(json?.data?.message || json?.message || text || ('HTTP ' + r.status)); 
                }
                return json || {};
            })
            .then(response => {
                if (response.success) {
                    // Mostrar mensaje de éxito
                    alert('Producto añadido al carrito correctamente');
                    
                    // Actualizar contador del carrito si existe
                    const cartCount = document.querySelector('.cart-count');
                    if (cartCount && response.data?.cart_count !== undefined) {
                        cartCount.textContent = response.data.cart_count;
                    }
                    
                    // Redirigir al carrito si se especifica
                    if (response.data?.redirect_url) {
                        window.location.href = response.data.redirect_url;
                    }
                } else {
                    throw new Error(response.data?.message || 'Error al añadir al carrito');
                }
            })
            .catch(error => {
                console.error('Error añadir al carrito:', error);
                alert('Error: ' + error.message);
            })
            .finally(() => {
                hideLoading();
            });
        });
    });
}

// =============================================
// INICIALIZACIÓN PRINCIPAL
// =============================================

function initializeShop() {
    console.log('Inicializando tienda básica...');
    
    // Inicializar eventos del carrito
    initializeCartEvents();
    
    // Inicializar formularios de añadir al carrito
    initializeAddToCartForms();
    
    // Inicializar sistema de pagos
    reinitPayments();
}

// Inicializar cuando el DOM esté listo
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initializeShop);
} else {
    initializeShop();
}

// Debug inicial
console.log('Tienda Básica JS cargado correctamente');