<?php
/*
Plugin Name: Talariasoft Shoplite
Plugin URI: https://talariasoft.com/shoplite
Description: Lightweight shop plugin for WordPress with product management, cart, and payments via Stripe, Bizum, and PayPal.
Version: 1.0.1
Author: Talariasoft
Author URI: https://talariasoft.com
Text Domain: shoplite
Domain Path: /languages
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}


// AJAX hooks (PayPal) – legacy TB actions used by tienda-basica.js
add_action('wp_ajax_tb_paypal_capture_order', 'shoplite_paypal_capture_order');
add_action('wp_ajax_nopriv_tb_paypal_capture_order', 'shoplite_paypal_capture_order');

add_action('wp_ajax_tb_paypal_create_order', 'shoplite_paypal_create_order');
add_action('wp_ajax_nopriv_tb_paypal_create_order', 'shoplite_paypal_create_order');



/**
 * Plugin constants.
 */
if ( ! defined( 'SHOPLITE_PLUGIN_FILE' ) ) {
    define( 'SHOPLITE_PLUGIN_FILE', __FILE__ );
}
if ( ! defined( 'SHOPLITE_DIR' ) ) {
    define( 'SHOPLITE_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'SHOPLITE_URL' ) ) {
    define( 'SHOPLITE_URL', plugin_dir_url( __FILE__ ) );
}

/**
 * Backward-compat shims (legacy "tienda-basica"/TB names).
 * Keep for now to avoid breaking older includes/usages.
 */
if ( ! defined( 'TB_PLUGIN_FILE' ) ) {
    define( 'TB_PLUGIN_FILE', SHOPLITE_PLUGIN_FILE );
}
if ( ! defined( 'TB_PLUGIN_DIR' ) ) {
    define( 'TB_PLUGIN_DIR', SHOPLITE_DIR );
}
if ( ! defined( 'TB_PLUGIN_URL' ) ) {
    define( 'TB_PLUGIN_URL', SHOPLITE_URL );
}

/**
 * Menu / endpoints.
 */
if ( ! defined( 'TS_MENU_SLUG' ) ) {
    define( 'TS_MENU_SLUG', 'shoplite' );
}
if ( ! defined( 'SHOPLITE_LICENSE_API' ) ) {
    define( 'SHOPLITE_LICENSE_API', 'https://talariasoft.com/api/licenses/' );
}
if ( ! defined( 'SHOPLITE_UPDATE_URL' ) ) {
    define( 'SHOPLITE_UPDATE_URL', 'https://talariasoft.com/api/updates/shoplite-pro' );
}

/**
 * Optional: silence WP deprecations noise in logs (dev-only).
 */
if ( ! defined( 'SHOPLITE_SILENCE_DEPRECATIONS' ) ) {
    define( 'SHOPLITE_SILENCE_DEPRECATIONS', true );
}

/**
 * Debug helper (only logs when WP_DEBUG is true).
 */
if ( ! function_exists( 'shoplite_log' ) ) {
    function shoplite_log( $message ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG && function_exists( 'error_log' ) ) {
            error_log( (string) $message );
        }
    }
}

shoplite_log( '[shoplite] main plugin file loaded. is_admin=' . ( is_admin() ? '1' : '0' ) );

/**
 * Talariasoft brand header (admin UI helper).
 */
if ( ! function_exists( 'shoplite_render_brand_header' ) ) {
    /**
     * Render Talariasoft brand header on admin screens.
     *
     * @param string $subtitle Subtitle under the logo.
     */
    function shoplite_render_brand_header( $subtitle = '' ) {
        $logo_url = plugins_url( 'assets/img/talariasoft-brand-header.png', __FILE__ );
        ?>
        <div class="shoplite-brand-hero">
            <div class="shoplite-brand-hero__logo">
                <img
                    src="<?php echo esc_url( $logo_url ); ?>"
                    alt="<?php echo esc_attr__( 'Talariasoft - Fast, lightweight plugins', 'shoplite' ); ?>"
                >
            </div>

            <?php if ( ! empty( $subtitle ) ) : ?>
                <p class="shoplite-brand-hero__subtitle">
                    <?php echo esc_html( $subtitle ); ?>
                </p>
            <?php endif; ?>
        </div>
        <?php
    }
}





/**
 * Safe redirect wrapper.
 */
if ( ! function_exists( 'shoplite_safe_redirect' ) ) {
    function shoplite_safe_redirect( $location, $status = 302 ) {
        if ( function_exists( 'wp_safe_redirect' ) ) {
            wp_safe_redirect( $location, $status );
            exit;
        }

        // Fallback sanitization.
        $location = preg_replace( '|[^a-z0-9-~+_.?#=!&;,/:%@$\|*\'()\\x80-\\xff]|i', '', (string) $location );
        header( "Location: $location", true, (int) $status );
        exit;
    }
}

/**
 * Start session safely (frontend only, before headers).
 */
if ( ! function_exists( 'shoplite_maybe_session_start' ) ) {
    function shoplite_maybe_session_start() {
        if ( is_admin() ) {
            return;
        }

        if ( session_status() !== PHP_SESSION_NONE ) {
            return;
        }

        if ( headers_sent() ) {
            shoplite_log( '[shoplite] session_start() skipped because headers were already sent.' );
            return;
        }

        $secure   = is_ssl();
        $httponly = true;
        $path     = defined( 'COOKIEPATH' ) ? ( COOKIEPATH ?: '/' ) : '/';
        $domain   = defined( 'COOKIE_DOMAIN' ) ? COOKIE_DOMAIN : '';

        if ( function_exists( 'session_set_cookie_params' ) ) {
            @session_set_cookie_params(
                array(
                    'lifetime' => 0,
                    'path'     => $path,
                    'domain'   => $domain,
                    'secure'   => $secure,
                    'httponly' => $httponly,
                    'samesite' => 'Lax',
                )
            );
        }

        @session_start();

        shoplite_log( sprintf( '[shoplite] session start id=%s path=%s', session_id(), $path ) );
    }
}

/**
 * Detect Pro version.
 */
if ( ! function_exists( 'shoplite_is_pro' ) ) {
    function shoplite_is_pro() {
        return defined( 'SHOPLITE_PRO' ) && SHOPLITE_PRO;
    }
}

/**
 * Allow add-ons (e.g., Shoplite Pro) to hook in.
 */
do_action( 'shoplite_loaded' );

/**
 * Admin locale selector (load ONLY in admin to avoid frontend side effects).
 */
if ( is_admin() ) {
    $shoplite_admin_locale_file = SHOPLITE_DIR . 'includes/shoplite-admin-locale.php';
    if ( file_exists( $shoplite_admin_locale_file ) ) {
        require_once $shoplite_admin_locale_file;
        shoplite_log( '[shoplite] admin-locale loaded from ' . $shoplite_admin_locale_file );
    }
}

/**
 * Setup Wizard + REST endpoints.
 */
require_once SHOPLITE_DIR . 'includes/class-shoplite-setup-wizard.php';
require_once SHOPLITE_DIR . 'includes/Rest/class-shoplite-rest-setup.php';

// ✅ Inventory (stock validation + reduce on shoplite_order_paid)
require_once SHOPLITE_DIR . 'includes/inventory.php';
require_once SHOPLITE_DIR . 'includes/inventory-admin.php';

// Digital deliveries (tokens + download endpoint)
require_once SHOPLITE_DIR . 'includes/digital-delivery.php';

add_action(
    'plugins_loaded',
    function () {
        if ( class_exists( 'Shoplite_Setup_Wizard' ) ) {
            Shoplite_Setup_Wizard::init();
        }
        if ( class_exists( 'Shoplite_REST_Setup' ) ) {
            Shoplite_REST_Setup::init();
        }
    }
);

/**
 * Sessions (frontend).
 */
add_action( 'init', 'shoplite_maybe_session_start', 1 );

/**
 * Nonce constants.
 */
if ( ! defined( 'SHOPLITE_NONCE_CART' ) ) {
    define( 'SHOPLITE_NONCE_CART', 'shoplite_cart' );
}
if ( ! defined( 'SHOPLITE_NONCE_CART_ACTION' ) ) {
    define( 'SHOPLITE_NONCE_CART_ACTION', 'shoplite_cart_action' );
}
if ( ! defined( 'SHOPLITE_NONCE_PAYPAL' ) ) {
    define( 'SHOPLITE_NONCE_PAYPAL', 'shoplite_paypal_nonce' );
}
if ( ! defined( 'SHOPLITE_NONCE_STRIPE' ) ) {
    define( 'SHOPLITE_NONCE_STRIPE', 'shoplite_stripe_nonce' );
}

/**
 * Nonce helpers (aliases + verify).
 */
if ( ! function_exists( 'shoplite_nonce_aliases' ) ) {
    function shoplite_nonce_aliases( $context ) {
        $map = array(
            'cart_post'   => array( SHOPLITE_NONCE_CART, 'tienda_basica_qty' ),
            'add_to_cart' => array( SHOPLITE_NONCE_CART_ACTION, 'tienda_basica_cart_action', 'shoplite_add_to_cart' ),
            'paypal'      => array( SHOPLITE_NONCE_PAYPAL ),
            'stripe'      => array( SHOPLITE_NONCE_STRIPE ),
        );

        return isset( $map[ $context ] ) ? $map[ $context ] : array();
    }
}

if ( ! function_exists( 'shoplite_nonce_verify' ) ) {
    function shoplite_nonce_verify( $nonce, $context ) {
        if ( empty( $nonce ) ) {
            return false;
        }

        foreach ( shoplite_nonce_aliases( $context ) as $name ) {
            if ( wp_verify_nonce( $nonce, $name ) ) {
                return true;
            }
        }

        return false;
    }
}

/**
 * Ensure shortcodes render in post/page content.
 */
add_filter( 'the_content', 'do_shortcode', 11 );

/**
 * Register shortcodes early.
 */
add_action( 'init', 'shoplite_register_shortcodes_early', 1 );

function shoplite_register_shortcodes_early() {

    /**
     * Alias: [shoplite_catalogo] → [tsl_catalog]
     * Maps EN/ES attributes for backwards compatibility.
     */
    add_shortcode(
        'shoplite_catalogo',
        function ( $atts = array(), $content = null ) {
            $atts = shortcode_atts(
                array(
                    'per_page'    => '',
                    'por_pagina'  => '',
                    'cat'         => '',
                    'categoria'   => '',
                    'orderby'     => '',
                    'ordenar_por' => '',
                    'order'       => '',
                    'orden'       => '',
                ),
                $atts,
                'shoplite_catalogo'
            );

            if ( '' === $atts['por_pagina'] && '' !== $atts['per_page'] ) {
                $atts['por_pagina'] = $atts['per_page'];
            }
            if ( '' === $atts['categoria'] && '' !== $atts['cat'] ) {
                $atts['categoria'] = $atts['cat'];
            }
            if ( '' === $atts['ordenar_por'] && '' !== $atts['orderby'] ) {
                $atts['ordenar_por'] = $atts['orderby'];
            }
            if ( '' === $atts['orden'] && '' !== $atts['order'] ) {
                $atts['orden'] = $atts['order'];
            }

            $parts = array();
            foreach ( array( 'por_pagina', 'categoria', 'ordenar_por', 'orden' ) as $k ) {
                if ( '' !== $atts[ $k ] ) {
                    $parts[] = $k . '="' . esc_attr( $atts[ $k ] ) . '"';
                }
            }

            $inner = implode( ' ', $parts );

            if ( shortcode_exists( 'tsl_catalog' ) ) {
                return do_shortcode( '[tsl_catalog ' . $inner . ']' );
            }
            if ( shortcode_exists( 'tienda_catalogo' ) ) {
                return do_shortcode( '[tienda_catalogo ' . $inner . ']' );
            }

            return '<p style="color:#c00">' . esc_html__( 'Catalog is not available.', 'shoplite' ) . '</p>';
        }
    );

    /**
     * Standalone search shortcode: [shoplite_search]
     * JS-enhanced input + <noscript> fallback.
     */
    add_shortcode(
        'shoplite_search',
        function () {
            $page_id  = get_queried_object_id();
            $page_url = $page_id ? get_permalink( $page_id ) : ( is_singular() ? get_permalink() : home_url( add_query_arg( array() ) ) );
            $val      = isset( $_GET['q'] ) ? sanitize_text_field( wp_unslash( $_GET['q'] ) ) : '';
            $uid      = 'slq_' . wp_generate_password( 6, false, false );

            ob_start();
            ?>
            <div class="shoplite-search" style="margin:0 0 1rem 0;">
                <input
                    id="<?php echo esc_attr( $uid ); ?>"
                    type="search"
                    value="<?php echo esc_attr( $val ); ?>"
                    placeholder="<?php echo esc_attr__( 'Search products…', 'shoplite' ); ?>"
                />
                <button type="button" data-slq="<?php echo esc_attr( $uid ); ?>">
                    <?php echo esc_html__( 'Search', 'shoplite' ); ?>
                </button>
            </div>
            <script>
            (function(){
                var btn = document.querySelector('button[data-slq="<?php echo esc_js( $uid ); ?>"]');
                if (!btn) return;
                btn.addEventListener('click', function(){
                    var el = document.getElementById('<?php echo esc_js( $uid ); ?>');
                    var v  = (el && el.value) ? el.value : '';
                    var url = new URL('<?php echo esc_js( $page_url ); ?>', window.location.origin);
                    if (v) url.searchParams.set('q', v); else url.searchParams.delete('q');
                    url.searchParams.delete('slp');
                    window.location.href = url.toString();
                });
            })();
            </script>
            <noscript>
                <form method="get" action="<?php echo esc_url( $page_url ); ?>">
                    <input type="search" name="q" value="<?php echo esc_attr( $val ); ?>" placeholder="<?php echo esc_attr__( 'Search products…', 'shoplite' ); ?>">
                    <button type="submit"><?php echo esc_html__( 'Search', 'shoplite' ); ?></button>
                </form>
            </noscript>
            <?php
            return ob_get_clean();
        }
    );
}



/**
 * Includes (load only if readable).
 */
foreach ( array(
    'includes/security.php',
    'includes/cart-service.php',
    'includes/cart-controller.php',
    'includes/routes.php',
    'includes/deprecations.php',
) as $rel ) {
    $path = SHOPLITE_DIR . $rel;
    if ( is_readable( $path ) ) {
        require_once $path;
    } else {
        // Soft debug (won't break activation).
        // shoplite_log( '[shoplite] Missing include: ' . $rel );
    }
}

/**
 * Options migration (activation hook).
 * Maps legacy option keys to new keys.
 */
register_activation_hook( SHOPLITE_PLUGIN_FILE, 'shoplite_migrate_options' );

function shoplite_migrate_options() {
    $map = array(
        'ts_btn_text_remove'    => 'shoplite_btn_text_remove',
        'paypal_modo'           => 'shoplite_paypal_mode',
        'paypal_sandbox_client' => 'shoplite_paypal_sandbox_client',
        'paypal_sandbox_secret' => 'shoplite_paypal_sandbox_secret',
        'paypal_live_client'    => 'shoplite_paypal_live_client',
        'paypal_live_secret'    => 'shoplite_paypal_live_secret',
        // Add Stripe/Bizum mappings here if you rename keys.
    );

    foreach ( $map as $old => $new ) {
        $v = get_option( $old, null );
        if ( null !== $v && null === get_option( $new, null ) ) {
            update_option( $new, $v );
        }
    }
}

/**
 * Temporary helper to read a new option key with legacy fallback.
 * Supports 2 or 3 arguments.
 */
if ( ! function_exists( 'shoplite_get_option' ) ) {
    function shoplite_get_option( $key, $default = '', $old = null ) {
        $v = get_option( $key, null );
        if ( null !== $v && false !== $v ) {
            return $v;
        }

        if ( $old ) {
            $ov = get_option( $old, null );
            if ( null !== $ov && false !== $ov ) {
                return $ov;
            }
        }

        return $default;
    }
}

/**
 * Basic admin-ajax ping endpoint (debug).
 */
add_action( 'wp_ajax_shoplite_dbg', 'shoplite_dbg' );
add_action( 'wp_ajax_nopriv_shoplite_dbg', 'shoplite_dbg' );

function shoplite_dbg() {
    wp_send_json_success(
        array(
            'ok'      => true,
            'post'    => $_POST,
            'cookies' => array_keys( $_COOKIE ),
        )
    );
}

/**
 * Debug: verify AJAX hooks are registered (WP_DEBUG only).
 */
add_action(
    'init',
    function () {
        if ( defined( 'DOING_AJAX' ) && DOING_AJAX && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            $h1 = has_action( 'wp_ajax_shoplite_cart_action', 'shoplite_ajax_cart_action' );
            $h2 = has_action( 'wp_ajax_nopriv_shoplite_cart_action', 'shoplite_ajax_cart_action' );
            $h3 = has_action( 'wp_ajax_shoplite_add_to_cart', 'shoplite_ajax_add_to_cart' );
            $h4 = has_action( 'wp_ajax_nopriv_shoplite_add_to_cart', 'shoplite_ajax_add_to_cart' );

            if ( function_exists( 'shoplite_log' ) ) {
                shoplite_log( '[shoplite][AJAX] hooks -> cart: ' . (int) $h1 . '/' . (int) $h2 . ' add: ' . (int) $h3 . '/' . (int) $h4 );
            }
        }
    },
    1
);

/**
 * Canonical plugin menu slug.
 */
if ( ! defined( 'TS_MENU_SLUG' ) ) {
    define( 'TS_MENU_SLUG', 'shoplite' );
}

/**
 * Helpers + Orders.
 */
require_once SHOPLITE_DIR . 'includes/helpers.php';
require_once SHOPLITE_DIR . 'includes/class-shoplite-orders.php';

if ( class_exists( 'Shoplite_Orders' ) ) {
    Shoplite_Orders::init();
}

// Admin-only settings (Settings API fields)
//if ( is_admin() ) {
//    $shoplite_admin_settings_file = SHOPLITE_DIR . 'includes/admin-settings.php';
//    if ( file_exists( $shoplite_admin_settings_file ) ) {
//        require_once $shoplite_admin_settings_file;
//    }
//}

/**
 * Build order payload from session/cart.
 *
 * @param string $payment_method 'stripe' | 'paypal' | 'bizum'...
 * @return array|null
 */
if ( ! function_exists( 'shoplite_build_order_data_from_session' ) ) {
    function shoplite_build_order_data_from_session( $payment_method ) {

        // Ensure session is running (frontend only).
        if ( function_exists( 'shoplite_maybe_session_start' ) ) {
            shoplite_maybe_session_start();
        }

        $cart = isset( $_SESSION['shoplite_cart'] ) ? $_SESSION['shoplite_cart'] : ( $_SESSION['carrito'] ?? array() );
        if ( empty( $cart ) || ! is_array( $cart ) ) {
            return null;
        }

        $subtotal    = 0.0;
        $shipping    = 0.0;

        foreach ( $cart as $line_key => $item ) {

            $price = isset( $item['precio'] )
                ? (float) $item['precio']
                : ( isset( $item['price'] ) ? (float) $item['price'] : 0.0 );

            $qty = isset( $item['cantidad'] )
                ? max( 1, (int) $item['cantidad'] )
                : ( isset( $item['qty'] ) ? max( 1, (int) $item['qty'] ) : 1 );

            $subtotal += $price * $qty;

            // Per-unit shipping for physical products.
            $is_physical = ! empty( $item['fisico'] );
            $ship_unit   = 0.0;

            if ( $is_physical ) {
                $ship_unit = (float) ( $item['envio'] ?? 0 );
            } elseif ( preg_match( '/^tbp_(\d+)$/', (string) $line_key, $m ) ) {
                // Legacy compatibility (old product key format).
                $pid        = (int) $m[1];
                $is_physical = ( get_post_meta( $pid, '_tb_is_physical', true ) === '1' );
                if ( $is_physical ) {
                    $ship_unit = (float) get_post_meta( $pid, '_tb_shipping_cost', true );
                }
            }

            if ( $is_physical && $ship_unit > 0 ) {
                $shipping += $ship_unit * $qty;
            }
        }

        $discount = 0.0;

        // Coupon stored in session.
        $coupon_code = isset( $_SESSION['shoplite_coupon_code'] )
            ? trim( (string) $_SESSION['shoplite_coupon_code'] )
            : '';

        if (
            defined( 'SHOPLITE_PRO' ) &&
            class_exists( 'Shoplite_Pro_Coupons' ) &&
            method_exists( 'Shoplite_Pro_Coupons', 'apply_coupon_to_total' )
        ) {
            $coupon_result = Shoplite_Pro_Coupons::apply_coupon_to_total(
                (float) $subtotal,
                $coupon_code !== '' ? $coupon_code : null
            );

            if ( is_array( $coupon_result ) && isset( $coupon_result['total'] ) ) {
                $subtotal = (float) $coupon_result['total'];
                $discount = isset( $coupon_result['discount'] ) ? (float) $coupon_result['discount'] : 0.0;
            }
        }

        $grand_total = $subtotal + $shipping;
        if ( $grand_total <= 0 ) {
            return null;
        }

        // Currency code (not symbol).
        $currency = strtoupper( shoplite_get_currency() ?: 'EUR' );

        $customer_email = isset( $_SESSION['shoplite_customer_email'] )
            ? sanitize_email( $_SESSION['shoplite_customer_email'] )
            : '';

        $customer_name = isset( $_SESSION['shoplite_customer_name'] )
            ? sanitize_text_field( $_SESSION['shoplite_customer_name'] )
            : '';

        return array(
            'total'          => $grand_total,
            'currency'       => $currency,
            'email'          => $customer_email,
            'name'           => $customer_name,
            'items'          => $cart,
            'payment_method' => (string) $payment_method,
            'status'         => 'completed',
        );
    }
}

/**
 * CSV Importer (admin screen + import logic).
 */
$__shoplite_importer = SHOPLITE_DIR . 'includes/importer.php';
if ( file_exists( $__shoplite_importer ) ) {
    require_once $__shoplite_importer;
} else {
    // Optional admin notice if importer is missing.
    // add_action('admin_notices', function(){
    //     echo '<div class="notice notice-error"><p>' . esc_html__( 'Missing file: includes/importer.php', 'shoplite' ) . '</p></div>';
    // });
}

/**
 * Wrap shop pages with .ts-scope (frontend only).
 */
add_filter(
    'the_content',
    function ( $content ) {
        if ( is_admin() ) {
            return $content;
        }

        $shop_pages = array(
            'carrito-de-compra',
            'pagina-pagos',
            'pago',
            'pago-cancelado',
            'pago-exitoso',
        );

        if ( is_page( $shop_pages ) ) {
            return '<div class="ts-scope">' . $content . '</div>';
        }

        return $content;
    },
    9
);

/**
 * Wide, readable layout inside .ts-scope (does not affect the rest of the site).
 */
add_action(
    'wp_enqueue_scripts',
    function () {
        if ( is_admin() ) {
            return;
        }

        $css = <<<CSS
.ts-scope{width:100%;max-width:100%;}
.ts-scope .wp-block-columns{display:flex;flex-wrap:wrap;gap:var(--wp--style--block-gap,1.5rem);}
.ts-scope .wp-block-column{flex:1 1 320px;min-width:0;}
.ts-scope table{width:100%;table-layout:auto}
.ts-scope, .ts-scope *{word-break:normal;overflow-wrap:break-word;writing-mode:horizontal-tb;}
.ts-scope .ts-btn,
.ts-scope a.ts-btn,
.ts-scope button.ts-btn,
.ts-scope input.ts-btn{white-space:nowrap;}
.ts-scope a[href*="carrito"],
.ts-scope button[name*="cart"],
.ts-scope input[type=submit][name*="cart"]{white-space:nowrap;}
CSS;

        wp_register_style( 'shoplite-layout-inline', false );
        wp_enqueue_style( 'shoplite-layout-inline' );
        wp_add_inline_style( 'shoplite-layout-inline', $css );
    },
    30
);

/**
 * Admin menu + submenus.
 */
add_action( 'admin_menu', 'ts_register_admin_menu' );

function ts_register_admin_menu() {

    add_menu_page(
        __( 'Talariasoft Shoplite', 'shoplite' ),
        __( 'Talariasoft Shoplite', 'shoplite' ),
        'manage_options',
        TS_MENU_SLUG,
        'shoplite_admin_page_render',
        plugins_url( 'assets/img/talariasoft-shoe-menu.png', __FILE__ ),
        56
    );

    // Payments
    add_submenu_page(
        TS_MENU_SLUG,
        __( 'Payments', 'shoplite' ),
        __( 'Payments', 'shoplite' ),
        'manage_options',
        'shoplite-payments',
        'shoplite_admin_pagos'
    );

    // Text & Style
    add_submenu_page(
        TS_MENU_SLUG,
        __( 'Text & Style', 'shoplite' ),
        __( 'Text & Style', 'shoplite' ),
        'manage_options',
        'shoplite-appearance',
        'shoplite_admin_appearance'
    );

    // ✅ Inventory / Stock
    add_submenu_page(
        TS_MENU_SLUG,
        __( 'Inventario', 'shoplite' ), // page title
        __( 'Stock', 'shoplite' ),      // menu label (corto)
        'manage_options',
        'shoplite-inventory',
        'shoplite_render_inventory_page'
    );
}


/**
 * Adjust Shoplite menu icon size.
 */
add_action(
    'admin_head',
    function () {
        if ( ! defined( 'TS_MENU_SLUG' ) ) {
            return;
        }

        $menu_id = 'toplevel_page_' . TS_MENU_SLUG;
        ?>
        <style>
            #<?php echo esc_attr( $menu_id ); ?> .wp-menu-image img{
                width:24px;
                height:24px;
                margin-top:-5px;
                object-fit:contain;
            }
        </style>
        <?php
    }
);





/* ============================================================
 * i18n / Languages
 * ============================================================ */

/**
 * Detect whether we are on a Shoplite / Shoplite Pro admin screen.
 */
function shoplite_is_shoplite_admin_screen() {

    if ( ! is_admin() ) {
        return false;
    }

    // 1) Query string detection
    $page      = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
    $post_type = isset( $_GET['post_type'] ) ? sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) : '';

    if ( $page && false !== strpos( $page, 'shoplite' ) ) {
        return true;
    }

    if ( $post_type && 0 === strpos( $post_type, 'shoplite' ) ) {
        return true;
    }

    // 2) Full URL fallback (very permissive)
    $uri = isset( $_SERVER['REQUEST_URI'] ) ? (string) $_SERVER['REQUEST_URI'] : '';
    if ( false !== strpos( $uri, 'shoplite' ) ) {
        return true;
    }

    return false;
}

/**
 * List of supported locales.
 *
 * Key = WordPress locale.
 */
function shoplite_get_supported_locales() {
    return array(
        'es_ES' => __( 'Spanish', 'shoplite' ),
        'en_US' => __( 'English', 'shoplite' ),
        'fr_FR' => __( 'French',  'shoplite' ),
    );
}

/**
 * Detect and store preferred locale (GET → user_meta → cookie → default).
 */
function shoplite_init_locale() {
    global $shoplite_current_locale;

    $supported = shoplite_get_supported_locales();

    // 1) Explicit change via ?shoplite_lang=xx_XX
    if ( isset( $_GET['shoplite_lang'] ) ) {
        $requested = sanitize_text_field( wp_unslash( $_GET['shoplite_lang'] ) );

        if ( isset( $supported[ $requested ] ) ) {

            // Store cookie (1 year)
            setcookie(
                'shoplite_lang',
                $requested,
                time() + YEAR_IN_SECONDS,
                COOKIEPATH ? COOKIEPATH : '/',
                COOKIE_DOMAIN,
                is_ssl(),
                true
            );

            // Store user meta if logged in
            if ( is_user_logged_in() ) {
                update_user_meta( get_current_user_id(), 'shoplite_lang', $requested );
            }

            $shoplite_current_locale = $requested;
        }
    }

    // 2) User meta
    if ( empty( $shoplite_current_locale ) && is_user_logged_in() ) {
        $user_locale = get_user_meta( get_current_user_id(), 'shoplite_lang', true );
        if ( $user_locale && isset( $supported[ $user_locale ] ) ) {
            $shoplite_current_locale = $user_locale;
        }
    }

    // 3) Cookie
    if ( empty( $shoplite_current_locale ) && ! empty( $_COOKIE['shoplite_lang'] ) ) {
        $cookie_locale = sanitize_text_field( wp_unslash( $_COOKIE['shoplite_lang'] ) );
        if ( isset( $supported[ $cookie_locale ] ) ) {
            $shoplite_current_locale = $cookie_locale;
        }
    }

    // 4) WordPress default locale
    if ( empty( $shoplite_current_locale ) ) {
        $shoplite_current_locale = get_locale();
    }
}
add_action( 'init', 'shoplite_init_locale', 1 );





/**
 * Load Shoplite translations.
 * Uses 'shoplite' domain and keeps backward compatibility with 'tienda-basica'.
 */
function shoplite_load_textdomain() {

    $rel_path = dirname( plugin_basename( __FILE__ ) ) . '/languages';

    load_plugin_textdomain( 'shoplite', false, $rel_path );

    // Backward compatibility
    load_plugin_textdomain( 'tienda-basica', false, $rel_path );
}
add_action( 'plugins_loaded', 'shoplite_load_textdomain', 1 );


/**
 * Language switcher shortcode.
 */
function shoplite_render_language_switcher() {
    global $shoplite_current_locale;

    $supported = shoplite_get_supported_locales();

    if ( empty( $shoplite_current_locale ) ) {
        $shoplite_current_locale = function_exists( 'determine_locale' )
            ? determine_locale()
            : get_locale();
    }

    ob_start();
    ?>
    <div class="shoplite-lang-switcher">
        <?php foreach ( $supported as $code => $label ) :
            $url   = add_query_arg( 'shoplite_lang', $code );
            $class = 'shoplite-lang-link';
            if ( $code === $shoplite_current_locale ) {
                $class .= ' shoplite-lang-current';
            }
            ?>
            <a class="<?php echo esc_attr( $class ); ?>" href="<?php echo esc_url( $url ); ?>">
                <?php echo esc_html( $label ); ?>
            </a>
        <?php endforeach; ?>
    </div>
    <?php

    return ob_get_clean();
}
add_shortcode( 'shoplite_lang_switcher', 'shoplite_render_language_switcher' );







// Bootstrap session for cart
add_action('init', function () {
  if (!session_id()) {
    @session_start([
      'cookie_httponly' => true,
      'cookie_secure'   => is_ssl(),
      'cookie_samesite' => 'Lax',
    ]);
  }
}, 1);

/* ============================================================
 * Options helpers
 * ============================================================ */

// Prevent duplicated renders per request
function shoplite_guard_once( $key ) {
  static $done = [];
  if ( isset($done[$key]) ) { return true; }
  $done[$key] = true;
  return false;
}

/* General */
function shoplite_get_currency() {
    return strtoupper( shoplite_get_option('shoplite_currency','EUR') );
}

function shoplite_get_success_url() {

    $val = trim( (string) shoplite_get_option('shoplite_success_url', '') );

    if ($val === '') {
        $val = trim( (string) get_option('tb_success_url', '') );
    }

    if ($val === '') { 
        return ''; 
    }

    if ( ! preg_match('~^https?://~i', $val) ) {
        $val = home_url('/' . ltrim($val,'/'));
    }

    $url = (strpos($val,'?') !== false) ? $val : trailingslashit($val);

    return esc_url_raw($url);
}

function shoplite_get_cancel_url() {

    $val = trim( (string) shoplite_get_option('shoplite_cancel_url', '') );

    if ($val === '') {
        $val = trim( (string) get_option('tb_cancel_url', '') );
    }

    if ($val === '') { 
        return ''; 
    }

    if ( ! preg_match('~^https?://~i', $val) ) {
        $val = home_url('/' . ltrim($val,'/'));
    }

    $url = (strpos($val,'?') !== false) ? $val : trailingslashit($val);

    return esc_url_raw($url);
}

/* ============================================================
 * Admin notice if checkout URLs are missing
 * ============================================================ */
add_action('admin_notices', function () {

    if ( ! current_user_can('manage_options') ) return;

    if ( ! function_exists('shoplite_get_success_url') || ! function_exists('shoplite_get_cancel_url') ) {
        return;
    }

    $success = shoplite_get_success_url();
    $cancel  = shoplite_get_cancel_url();

    if ($success === '' || $cancel === '') {
        echo '<div class="notice notice-warning"><p>';
        echo '⚠️ <strong>' . esc_html__( 'Shoplite', 'shoplite' ) . ':</strong> ';
        echo esc_html__( 'The success URL and/or cancel URL are not configured.', 'shoplite' ) . '<br>';
        echo esc_html__( 'Stripe Checkout cannot start until both URLs are set.', 'shoplite' ) . '<br>';
        echo esc_html__( 'Go to Shoplite → Payments and complete both fields.', 'shoplite' );
        echo '</p></div>';
    }
});

function shoplite_paypal_show_card()   { return (bool) shoplite_get_option('shoplite_paypal_show_card','1'); }
function shoplite_show_sandbox_badge() { return (bool) shoplite_get_option('shoplite_show_sandbox_badge','1'); }

/* Stripe */
function shoplite_get_modo_stripe() { return shoplite_get_option('stripe_modo', 'test') === 'live' ? 'live' : 'test'; }
function shoplite_stripe_pubkey()   { return shoplite_get_modo_stripe()==='live' ? shoplite_get_option('stripe_live_public','') : shoplite_get_option('stripe_test_public',''); }
function shoplite_stripe_seckey()   { return shoplite_get_modo_stripe()==='live' ? shoplite_get_option('stripe_live_secret','') : shoplite_get_option('stripe_test_secret',''); }

/* PayPal */
function shoplite_get_modo_paypal() { return shoplite_get_option('paypal_modo', 'sandbox') === 'live' ? 'live' : 'sandbox'; }
function shoplite_paypal_client_id(){ return shoplite_get_modo_paypal()==='live' ? shoplite_get_option('paypal_live_client','') : shoplite_get_option('paypal_sandbox_client',''); }
function shoplite_paypal_secret()   { return shoplite_get_modo_paypal()==='live' ? shoplite_get_option('paypal_live_secret','') : shoplite_get_option('paypal_sandbox_secret',''); }
function shoplite_paypal_api_base() { return shoplite_get_modo_paypal()==='live' ? 'https://api-m.paypal.com' : 'https://api-m.sandbox.paypal.com'; }

/* Stripe helpers */
function shoplite_stripe_amount_multiplier($currency){
    $zero = array('BIF','CLP','DJF','GNF','JPY','KMF','KRW','MGA','PYG','RWF','UGX','VND','VUV','XAF','XOF','XPF');
    return in_array(strtoupper($currency), $zero, true) ? 1 : 100;
}

/* PayPal helpers */
function shoplite_paypal_is_zero_decimal($currency){
    $zero = array('JPY','KRW','VND','CLP');
    return in_array(strtoupper($currency), $zero, true);
}



/* ============================================================
 * Demo catalog
 * ============================================================ */
if (!function_exists('shoplite_catalogo')) {
  function shoplite_catalogo() {
      return array(
          1 => array('nombre'=>'Taza mágica','precio'=>9.99),
          2 => array('nombre'=>'Camiseta divertida','precio'=>14.99),
          3 => array('nombre'=>'Bolsa de tela','precio'=>7.99),
      );
  }
}

/* ============================================================
 * CPT: Shoplite Products (public, with /producto/ slug)
 * ============================================================ */
add_action('init', 'shoplite_register_tb_producto_cpt');
function shoplite_register_tb_producto_cpt() {
  $labels = [
    'name'               => __('Shoplite - Products','shoplite'),
    'singular_name'      => __('Product','shoplite'),
    'add_new'            => __('Add New','shoplite'),
    'add_new_item'       => __('Add New Product','shoplite'),
    'edit_item'          => __('Edit Product','shoplite'),
    'new_item'           => __('New Product','shoplite'),
    'view_item'          => __('View Product','shoplite'),
    'search_items'       => __('Search Products','shoplite'),
    'not_found'          => __('No products found','shoplite'),
    'menu_name'          => __('Shoplite - Products','shoplite'),
  ];

  register_post_type('shoplite_producto', [
    'labels'             => $labels,
    'public'             => true,              // 🔹 public URLs
    'publicly_queryable' => true,              // 🔹 visible on frontend
    'show_ui'            => true,
    'show_in_rest'       => true,              // block editor + REST API
    'show_in_menu'       => defined('TS_MENU_SLUG') ? TS_MENU_SLUG : true,
    'exclude_from_search'=> false,             // 🔹 searchable (SEO)
    'has_archive'        => false,             // no /producto/ archive (for now)
    'rewrite'            => ['slug' => 'producto', 'with_front' => false],
    'hierarchical'       => false,
    'supports'           => ['title','editor','thumbnail','excerpt'], // 🔹 description + image
    'capability_type'    => 'post',
    'map_meta_cap'       => true,
    'query_var'          => true,
  ]);
}

add_action('init', function () {
  register_taxonomy('shoplite_categoria', 'shoplite_producto', [
    'labels' => [
      'name'          => __('Categories','shoplite'),
      'singular_name' => __('Category','shoplite'),
      'menu_name'     => __('Categories','shoplite'),
    ],
    'hierarchical'      => true,
    'show_ui'           => true,
    'show_admin_column' => true,
    'show_in_rest'      => true,
    'public'            => true,
    'rewrite'           => ['slug' => 'categoria-producto', 'with_front' => false],
  ]);
});


// —— Robust registration of AJAX endpoints (front, nologin and admin-ajax)
if ( ! function_exists( 'shoplite_register_ajax_endpoints' ) ) {
  function shoplite_register_ajax_endpoints() {

    add_action( 'wp_ajax_shoplite_add_to_cart',        'shoplite_ajax_add_to_cart' );
    add_action( 'wp_ajax_nopriv_shoplite_add_to_cart', 'shoplite_ajax_add_to_cart' );

    add_action( 'wp_ajax_shoplite_cart_action',        'shoplite_ajax_cart_action' );
    add_action( 'wp_ajax_nopriv_shoplite_cart_action', 'shoplite_ajax_cart_action' );

    // NEW: save customer email for abandoned carts + prefill checkout
    add_action( 'wp_ajax_shoplite_save_customer_email',        'shoplite_ajax_save_customer_email' );
    add_action( 'wp_ajax_nopriv_shoplite_save_customer_email', 'shoplite_ajax_save_customer_email' );
  }
}
add_action( 'init', 'shoplite_register_ajax_endpoints', 0 );



/**
 * AJAX: store customer email in session (for abandoned carts & checkout prefill).
 */
function shoplite_set_customer_email() {
  if ( function_exists('shoplite_session_start') ) {
    shoplite_session_start();
  } elseif ( ! session_id() ) {
    @session_start();
  }

  $email = isset($_POST['email']) ? sanitize_email( wp_unslash($_POST['email']) ) : '';
  if ( $email && is_email($email) ) {
    $_SESSION['shoplite_customer_email'] = $email;
    wp_send_json_success([ 'saved' => true ]);
  }

  wp_send_json_error([ 'saved' => false ], 400);
}



// Quick log to confirm whether our shortcodes are registered
add_action('init', function () {
  error_log('[Shoplite] sc shoplite_catalogo: ' . (shortcode_exists('shoplite_catalogo') ? 'OK' : 'NO'));
  error_log('[Shoplite] sc tsl_catalog: '       . (shortcode_exists('tsl_catalog') ? 'OK' : 'NO'));
  error_log('[Shoplite] sc tienda_catalogo: '   . (shortcode_exists('tienda_catalogo') ? 'OK' : 'NO'));
  error_log('[Shoplite] sc shoplite_search: '   . (shortcode_exists('shoplite_search') ? 'OK' : 'NO'));
});



// === Plan C: force output with our template before the theme touches anything
if ( ! function_exists('shoplite_render_single_direct') ) {
  function shoplite_render_single_direct() {
    if ( ! is_singular('shoplite_producto') ) return;
    if ( is_admin() || is_feed() || is_embed() || is_preview() ) return;

    $tpl = plugin_dir_path(__FILE__) . 'templates/single-shoplite_producto.php';
    if ( file_exists($tpl) ) {
      if ( function_exists('error_log') ) error_log('[shoplite] template_redirect -> include plugin single & exit');
      status_header(200);
      nocache_headers();
      include $tpl;
      exit; // <- key to prevent the theme from loading anything else
    }
  }
  // Very early priority to run before everything else
  add_action('template_redirect', 'shoplite_render_single_direct', 0);
}


// === Force Shoplite single template on frontend ===
function shoplite_plugin_single_template_path(){
  return plugin_dir_path(__FILE__) . 'templates/single-shoplite_producto.php';
}

function shoplite_force_single_template($single){
  if (is_singular('shoplite_producto')) {
    $tpl = shoplite_plugin_single_template_path();
    if (file_exists($tpl)) {
      if (function_exists('error_log')) error_log('[shoplite] single_template -> plugin');
      return $tpl;
    }
  }
  return $single;
}
add_filter('single_template', 'shoplite_force_single_template', 20);







// Do not hook filters here:
// add_filter('template_include', 'shoplite_template_include', 50);
// add_filter('template_include', function($template){ return $template; }, 99);


// Flush permalinks on plugin activate/deactivate
register_activation_hook(__FILE__, function () {

  // migrate legacy options (if exists)
  if ( function_exists('shoplite_migrate_options') ) {
    shoplite_migrate_options();
  }

  // Product CPT
  shoplite_register_tb_producto_cpt();

  // Digital delivery CPT + rewrite
  if ( function_exists('sl_register_cpt_delivery') ) {
    sl_register_cpt_delivery();
  }
  if ( function_exists('sl_delivery_add_rewrite') ) {
    sl_delivery_add_rewrite();
  }

  flush_rewrite_rules();
});

register_deactivation_hook(__FILE__, function () {
  flush_rewrite_rules();
});


  
/* … your register_post_type (tb_register_tb_producto_cpt) goes here … */



/* … shoplite_sync_carrito_login() continues here … */  

/* ============================================================
 * Metabox: Product data (price, SKU, physical, shipping)
 * ============================================================ */
add_action('add_meta_boxes', 'shoplite_register_tb_producto_metabox');
function shoplite_register_tb_producto_metabox() {
  add_meta_box(
    'shoplite_producto_meta',
    __('Product data', 'shoplite'),
    'shoplite_producto_meta_cb', // ← keep your callback as-is
    'shoplite_producto',
    'normal',
    'high'
  );
}


function shoplite_producto_meta_cb($post) {
  wp_nonce_field('shoplite_producto_meta_nonce', 'tb_producto_meta_nonce');

  $price         = get_post_meta($post->ID, '_tb_price', true);
  $is_physical   = get_post_meta($post->ID, '_tb_is_physical', true);
  $shipping_cost = get_post_meta($post->ID, '_tb_shipping_cost', true);
  $sku           = get_post_meta($post->ID, '_tb_sku', true);

  // NEW: digital download fields
  $download_attachment_id = (int) get_post_meta($post->ID, '_tb_download_attachment_id', true);
  $download_expiry_days   = (int) get_post_meta($post->ID, '_tb_download_expiry_days', true);

  $download_label = '';
  if ($download_attachment_id > 0) {
    $att_url = wp_get_attachment_url($download_attachment_id);
    $download_label = $att_url ? basename($att_url) : ('Attachment ID: ' . $download_attachment_id);
  }

  $currency = strtoupper(shoplite_get_currency() ?? 'EUR');
  ?>
  <style>
    .tb-grid { display:grid; grid-template-columns: 1fr 1fr; gap:16px; }
    .tb-field label { display:block; font-weight:600; margin-bottom:6px; }
    .tb-inline { display:flex; align-items:center; gap:8px; margin-top:6px; }
    .tb-help { font-size:12px; opacity:.8; }
    .tb-actions { display:flex; align-items:center; gap:10px; flex-wrap:wrap; }
    .tb-muted { opacity:.85; }
  </style>

  <div class="tb-grid">
    <div class="tb-field">
      <label><?php _e('Price', 'shoplite'); ?> (<?php echo esc_html($currency); ?>)</label>
      <input type="number" step="0.01" min="0" name="tb_price" value="<?php echo esc_attr($price !== '' ? $price : ''); ?>" style="width:100%;">
      <div class="tb-help"><?php _e('Example: 19.90', 'shoplite'); ?></div>
    </div>

    <div class="tb-field">
      <label><?php _e('SKU', 'shoplite'); ?></label>
      <input type="text" name="tb_sku" value="<?php echo esc_attr($sku !== '' ? $sku : ''); ?>" style="width:100%;">
      <div class="tb-help"><?php _e('Optional internal identifier', 'shoplite'); ?></div>
    </div>

    <div class="tb-field">
      <label><?php _e('Product type', 'shoplite'); ?></label>
      <div class="tb-inline">
        <label>
          <input type="checkbox" name="tb_is_physical" value="1" <?php checked($is_physical, '1'); ?>>
          <?php _e('Physical product (requires shipping)', 'shoplite'); ?>
        </label>
      </div>
      <div class="tb-help"><?php _e('Uncheck for digital products/services.', 'shoplite'); ?></div>
    </div>

    <div class="tb-field">
      <label><?php _e('Shipping cost', 'shoplite'); ?> (<?php echo esc_html($currency); ?>)</label>
      <input type="number" step="0.01" min="0" name="tb_shipping_cost" value="<?php echo esc_attr($shipping_cost !== '' ? $shipping_cost : ''); ?>" style="width:100%;">
      <div class="tb-help"><?php _e('Only applies if the product is physical.', 'shoplite'); ?></div>
    </div>

    <!-- NEW: Digital download file -->
    <div class="tb-field" style="grid-column: 1 / -1;">
      <label><?php _e('Digital download file', 'shoplite'); ?></label>

      <input
        type="hidden"
        name="tb_download_attachment_id"
        id="tb_download_attachment_id"
        value="<?php echo esc_attr($download_attachment_id ?: ''); ?>"
      >

      <div class="tb-actions">
        <button type="button" class="button" id="tb_pick_download_file">
          <?php _e('Select file', 'shoplite'); ?>
        </button>

        <button type="button" class="button" id="tb_clear_download_file">
          <?php _e('Clear', 'shoplite'); ?>
        </button>

        <span id="tb_download_file_label" class="tb-muted">
          <?php echo $download_label ? esc_html($download_label) : esc_html__('No file selected', 'shoplite'); ?>
        </span>
      </div>

      <div class="tb-help">
        <?php _e('Attach a PDF/ZIP/etc. for non-physical products. A secure token link will be sent after payment.', 'shoplite'); ?>
      </div>
    </div>

    <!-- NEW: Download expiry -->
    <div class="tb-field">
      <label><?php _e('Download expiry (days)', 'shoplite'); ?></label>
      <input
        type="number"
        min="0"
        step="1"
        name="tb_download_expiry_days"
        value="<?php echo esc_attr((string) max(0, $download_expiry_days)); ?>"
        style="width:100%;"
      >
      <div class="tb-help"><?php _e('0 = no expiry', 'shoplite'); ?></div>
    </div>

  </div>
  <?php
}


// 3) Save metadata safely
add_action('save_post_shoplite_producto', function ($post_id) {
  // Autosave / permissions
  if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
  if (!current_user_can('edit_post', $post_id)) return;

  // Metabox nonce (make sure you print it with:
  // wp_nonce_field('shoplite_producto_meta_nonce', 'tb_producto_meta_nonce'); )
  $nonce = $_POST['tb_producto_meta_nonce'] ?? '';
  if (!wp_verify_nonce($nonce, 'shoplite_producto_meta_nonce')) return;

  // Numeric helper: comma -> dot
  $to_float = function ($v) {
    if ($v === null || $v === '') return null;
    $v = wp_unslash($v);
    $v = str_replace(',', '.', $v);
    return is_numeric($v) ? (float)$v : null;
  };

  // Price
  $price = $to_float($_POST['tb_price'] ?? '');
  if ($price === null) {
    delete_post_meta($post_id, '_tb_price');
  } else {
    update_post_meta($post_id, '_tb_price', $price);
  }

  // SKU (text)
  $sku = isset($_POST['tb_sku']) ? sanitize_text_field(wp_unslash($_POST['tb_sku'])) : '';
  if ($sku === '') {
    delete_post_meta($post_id, '_tb_sku');
  } else {
    update_post_meta($post_id, '_tb_sku', $sku);
  }

  // Physical?
  $is_physical = !empty($_POST['tb_is_physical']) ? '1' : '0';
  update_post_meta($post_id, '_tb_is_physical', $is_physical);

  // Shipping
  $shipping = $to_float($_POST['tb_shipping_cost'] ?? '');
  if ($shipping === null) {
    delete_post_meta($post_id, '_tb_shipping_cost');
  } else {
    update_post_meta($post_id, '_tb_shipping_cost', $shipping);
  }

  /**
   * NEW: Digital download attachment ID (int)
   * Field name: tb_download_attachment_id
   * Meta key: _tb_download_attachment_id
   */
  $att_id = isset($_POST['tb_download_attachment_id'])
    ? (int) wp_unslash($_POST['tb_download_attachment_id'])
    : 0;

  if ($att_id > 0) {
    // Basic validation: ensure it's an attachment
    if (get_post_type($att_id) === 'attachment') {
      update_post_meta($post_id, '_tb_download_attachment_id', $att_id);
    } else {
      // If invalid, remove
      delete_post_meta($post_id, '_tb_download_attachment_id');
    }
  } else {
    delete_post_meta($post_id, '_tb_download_attachment_id');
  }

  /**
   * NEW: Download expiry days (int, 0 allowed)
   * Field name: tb_download_expiry_days
   * Meta key: _tb_download_expiry_days
   */
  $expiry_days = isset($_POST['tb_download_expiry_days'])
    ? (int) wp_unslash($_POST['tb_download_expiry_days'])
    : 0;

  if ($expiry_days < 0) $expiry_days = 0;
  update_post_meta($post_id, '_tb_download_expiry_days', $expiry_days);
});




add_action('admin_enqueue_scripts', function () {
  $screen = function_exists('get_current_screen') ? get_current_screen() : null;
  if (!$screen || $screen->post_type !== 'shoplite_producto') return;

  wp_enqueue_media();

  wp_add_inline_script('jquery', "
    jQuery(function($){
      var frame;

      $(document).on('click', '#tb_pick_download_file', function(e){
        e.preventDefault();

        if (frame) { frame.open(); return; }

        frame = wp.media({
          title: 'Select file',
          button: { text: 'Use this file' },
          multiple: false
        });

        frame.on('select', function(){
          var att = frame.state().get('selection').first().toJSON();
          $('#tb_download_attachment_id').val(att.id);
          $('#tb_download_file_label').text(att.filename || att.url || ('ID: ' + att.id));
        });

        frame.open();
      });

      $(document).on('click', '#tb_clear_download_file', function(e){
        e.preventDefault();
        $('#tb_download_attachment_id').val('');
        $('#tb_download_file_label').text('No file selected');
      });
    });
  ");
});







// CPT products (fallback to demo if there are none yet)
function shoplite_catalogo_real() {
  $out = [];
  $q = new WP_Query([
    'post_type'      => 'shoplite_producto',
    'post_status'    => 'publish',
    'posts_per_page' => -1,
    'orderby'        => ['menu_order' => 'ASC', 'title' => 'ASC'],
  ]);

  if ($q->have_posts()) {
    foreach ($q->posts as $p) {
      $pid    = (int) $p->ID;
      $price  = (float) get_post_meta($pid, '_tb_price', true);
      if ($price <= 0) continue; // hide items with no price

      $desc = get_the_excerpt($pid);
      if ($desc === '') {
        $desc = wp_trim_words( wp_strip_all_tags( get_post_field('post_content', $pid) ), 24 );
      }

      $out['tbp_'.$pid] = [
        'id'      => $pid,
        'nombre'  => get_the_title($pid),
        'precio'  => $price,
        'sku'     => get_post_meta($pid,'_tb_sku',true),
        'fisico'  => get_post_meta($pid,'_tb_is_physical',true) === '1',
        'envio'   => (float) get_post_meta($pid,'_tb_shipping_cost',true),
        'img'     => get_the_post_thumbnail_url($pid, 'medium'),
        'url'     => get_permalink($pid),
        'desc'    => wp_strip_all_tags($desc),
      ];
    }
    wp_reset_postdata();
  }

  // Fallback to demo if there are no real products
  if (empty($out) && function_exists('shoplite_catalogo')) {
    foreach (shoplite_catalogo() as $id => $it) {
      $out[(string)$id] = $it;
    }
  }
  return $out;
}



/* ============================================================
 * User-synced cart
 * ============================================================ */
function shoplite_sync_carrito_login() {
    if (!isset($_SESSION['carrito'])) $_SESSION['carrito'] = array();
    if (is_user_logged_in()) {
        $uid = get_current_user_id();
        $db = get_user_meta($uid, '_tienda_basica_carrito', true);
        if (is_array($db) && empty($_SESSION['carrito'])) {
            $_SESSION['carrito'] = $db;
        }
        add_action('shutdown', function() use ($uid){
            if (!empty($_SESSION['carrito'])) update_user_meta($uid, '_tienda_basica_carrito', $_SESSION['carrito']);
            else delete_user_meta($uid, '_tienda_basica_carrito');
        });
    }
}
add_action('init', 'shoplite_sync_carrito_login', 5);



// Encola CSS/JS frontend y pasa opciones al JS (solo donde hace falta)
add_action('wp_enqueue_scripts', function () {
  if (is_admin()) return;

  // Cargar si: hay shortcodes relevantes o es single de shoplite_producto
  $needs = false;
  if (is_singular()) {
    global $post;
    if ($post) {
      $c = $post->post_content ?? '';
      $needs =
           has_shortcode($c, 'shoplite_cart')
        || has_shortcode($c, 'tienda_carrito')
        || has_shortcode($c, 'shoplite_catalog')
        || has_shortcode($c, 'tienda_catalogo');
    }
  }
  if (!$needs && !is_singular('shoplite_producto')) return;

  // --- CSS ---
  wp_enqueue_style(
    'estilos-tienda',
    plugin_dir_url(__FILE__) . 'estilos-tienda.css',
    [],
    file_exists(plugin_dir_path(__FILE__) . 'estilos-tienda.css')
      ? filemtime(plugin_dir_path(__FILE__) . 'estilos-tienda.css')
      : null
  );

  // --- Stripe.js (solo si hay clave pública) ---
  $stripe_pub = function_exists('shoplite_stripe_pubkey') ? shoplite_stripe_pubkey() : '';
  if (!empty($stripe_pub)) {
    wp_enqueue_script('stripe_js', 'https://js.stripe.com/v3/', [], null, true);
  }

  // --- PayPal SDK (solo si hay Client ID) ---
  $deps = [];
  $paypal_client = function_exists('shoplite_paypal_client_id') ? shoplite_paypal_client_id() : '';
  if (!empty($paypal_client) && !wp_script_is('paypal-sdk', 'enqueued')) {
    $currency   = function_exists('shoplite_get_currency') ? shoplite_get_currency() : 'EUR';
    $paypal_url = 'https://www.paypal.com/sdk/js'
      . '?client-id=' . rawurlencode($paypal_client)
      . '&currency=' . rawurlencode($currency ?: 'EUR')
      . '&intent=capture';
    wp_enqueue_script('paypal-sdk', $paypal_url, [], null, true);
    $deps[] = 'paypal-sdk';
  }

  // --- JS principal (MISMO handle para enqueue y localize) ---
  $js_path = plugin_dir_path(__FILE__) . 'tienda-basica.js';
  $js_url  = plugins_url('tienda-basica.js', __FILE__);
  $ver     = file_exists($js_path) ? filemtime($js_path) : time();

  wp_enqueue_script(
    'tienda-basica',
    $js_url,
    $deps,
    $ver,
    true
  );

  // --- Localize tbAjax (URL absoluta para evitar 404 y con nonces) ---
  $success = trim((string) get_option('tb_success_url', '')) ?: home_url('/');
  $cancel  = trim((string) get_option('tb_cancel_url',  '')) ?: home_url('/');

  wp_localize_script('tienda-basica', 'tbAjax', [
    // URLs
    'ajaxUrl'     => admin_url('admin-ajax.php'), // absoluta
    'ajax_url'    => admin_url('admin-ajax.php'),
    'successUrl'  => $success,
    'cancelUrl'   => $cancel,

    // Aliases legacy
    'success_url' => $success,
    'cancel_url'  => $cancel,

    // Nonces
    'cartNonce'   => wp_create_nonce( defined('SHOPLITE_NONCE_CART_ACTION') ? SHOPLITE_NONCE_CART_ACTION : 'shoplite_cart_action' ),
    'stripeNonce' => wp_create_nonce( defined('SHOPLITE_NONCE_STRIPE') ? SHOPLITE_NONCE_STRIPE : 'shoplite_stripe_nonce' ),
    'paypalNonce' => wp_create_nonce( defined('SHOPLITE_NONCE_PAYPAL') ? SHOPLITE_NONCE_PAYPAL : 'shoplite_paypal_nonce' ),
    'nonce_pp'    => wp_create_nonce( defined('SHOPLITE_NONCE_PAYPAL') ? SHOPLITE_NONCE_PAYPAL : 'shoplite_paypal_nonce' ),

    // Claves/ajustes
    'stripe_pub'       => $stripe_pub,
    'stripePub'        => $stripe_pub, // alias camelCase
    'paypal_client'    => $paypal_client,
    'paypal_show_card' => function_exists('shoplite_paypal_show_card') && shoplite_paypal_show_card() ? '1' : '0',
    'currency'         => function_exists('shoplite_get_currency') ? shoplite_get_currency() : 'EUR',

    // Cupón actual desde la sesión (si existe)
    'coupon_code'      => isset($_SESSION['shoplite_coupon_code']) ? $_SESSION['shoplite_coupon_code'] : '',
  ]);

  // Aliases globales legacy (opcional)
  wp_add_inline_script('tienda-basica', 'try{
    window.ajaxUrl     = (tbAjax && (tbAjax.ajaxUrl    || tbAjax.ajax_url))    || window.ajaxUrl;
    window.paypalNonce = (tbAjax && tbAjax.paypalNonce)                         || window.paypalNonce;
    window.successUrl  = (tbAjax && (tbAjax.successUrl || tbAjax.success_url)) || window.successUrl;
    window.cancelUrl   = (tbAjax && (tbAjax.cancelUrl  || tbAjax.cancel_url))  || window.cancelUrl;
  }catch(e){}', 'after');

  if (function_exists('error_log')) {
    error_log('[shoplite] enqueue+localize tienda-basica ver=' . $ver);
  }
}, 20);






// Enqueue and localize tienda-basica.js (cache-busting + correct order)
add_action('wp_enqueue_scripts', function () {

  // Keys: read from functions/options here so you don't depend on external variables
  if (function_exists('shoplite_stripe_pubkey')) {
    $stripe_pub = (string) shoplite_stripe_pubkey();
  } else {
    $stripe_pub = (string) get_option('tb_stripe_pub', '');
  }

  if (function_exists('shoplite_paypal_client_id')) {
    $paypal_client = (string) shoplite_paypal_client_id();
  } else {
    $paypal_client = (string) get_option('tb_paypal_client_id', '');
  }

  // Return URLs (adjust to your functions if you have them)
  $success = trim((string) get_option('tb_success_url', ''));
  $cancel  = trim((string) get_option('tb_cancel_url',  ''));
  if ($success === '') { $success = home_url('/index.php/pago-exitoso/'); }
  if ($cancel  === '') { $cancel  = home_url('/index.php/carrito-de-compra/'); }

  // Cache-busting via filemtime
  $rel_js  = 'tienda-basica.js';
  $js_path = plugin_dir_path(__FILE__) . $rel_js;
  $js_url  = plugins_url($rel_js, __FILE__);
  $ver     = file_exists($js_path) ? filemtime($js_path) : time();

  // Enqueue before localize
  wp_register_script('tienda-basica', $js_url, [], $ver, true);
  wp_enqueue_script('tienda-basica');

  // Localize
  wp_localize_script('tienda-basica', 'tbAjax', [
    // Nonces
    'stripeNonce' => wp_create_nonce( SHOPLITE_NONCE_STRIPE ),
    'paypalNonce' => wp_create_nonce( SHOPLITE_NONCE_PAYPAL ),
    'nonce_pp'    => wp_create_nonce( SHOPLITE_NONCE_PAYPAL ),
    'cartNonce'   => wp_create_nonce( SHOPLITE_NONCE_CART_ACTION ),

    // URLs
    'ajaxUrl'     => admin_url('admin-ajax.php'),
    'successUrl'  => $success,
    'cancelUrl'   => $cancel,

    // Keys / settings
    'stripe_pub'       => $stripe_pub,
    'stripePub'        => $stripe_pub,   // camelCase alias
    'paypal_client'    => $paypal_client,
    'paypal_show_card' => (function_exists('shoplite_paypal_show_card') && shoplite_paypal_show_card()) ? '1' : '0',
    'currency'         => (function_exists('shoplite_get_currency') ? shoplite_get_currency() : get_option('tb_currency','EUR')),

    // 🔥 NEW: current coupon from session
    'coupon_code' => isset($_SESSION['shoplite_coupon_code'])
                        ? $_SESSION['shoplite_coupon_code']
                        : '',

    // Legacy aliases
    'ajax_url'    => admin_url('admin-ajax.php'),
    'success_url' => $success,
    'cancel_url'  => $cancel,
  ]);


  // Legacy global aliases (optional)
  wp_add_inline_script('tienda-basica', 'try{
    window.ajaxUrl    = (tbAjax && (tbAjax.ajaxUrl    || tbAjax.ajax_url))    || window.ajaxUrl;
    window.paypalNonce= (tbAjax && tbAjax.paypalNonce)                          || window.paypalNonce;
    window.successUrl = (tbAjax && (tbAjax.successUrl || tbAjax.success_url)) || window.successUrl;
    window.cancelUrl  = (tbAjax && (tbAjax.cancelUrl  || tbAjax.cancel_url))  || window.cancelUrl;
  }catch(e){}', 'after');

}, 20);





// Disable caching when the page contains our shortcodes
add_action('template_redirect', function () {
  if (!is_singular()) return;
  global $post; if (!$post) return;

  $c = $post->post_content ?? '';
  if ( has_shortcode($c, 'tienda_carrito') || has_shortcode($c, 'tienda_catalogo') ) {
    if (!defined('DONOTCACHEPAGE'))   define('DONOTCACHEPAGE', true);
    if (!defined('DONOTCACHEOBJECT')) define('DONOTCACHEOBJECT', true);
    nocache_headers();
  }
}, 0);


// === Style and text options ===
if (!function_exists('shoplite_get_button_style')) {
  function shoplite_get_button_style() {
    // Prefer the new setting; use the old one only as fallback
    $style = get_option('shoplite_button_style', '');
    if (!$style) $style = get_option('estilos_boton', '');
    return $style ?: 'clasico';
  }
}


if (!function_exists('shoplite_get_labels')) {
  function shoplite_get_labels() {
    $defaults = array(
      'your_cart'   => __('Your cart','shoplite'),
      'add_to_cart' => __('Add to cart','shoplite'),
      'remove'      => __('Remove','shoplite'),
      'empty_cart'  => __('Empty cart','shoplite'),
      'update_qty'  => __('Update quantities','shoplite'),
      'subtotal'    => __('Subtotal','shoplite'),
      'shipping'    => __('Shipping','shoplite'),
      'total'       => __('Total','shoplite'),
      'pay_stripe'  => __('Pay with Stripe','shoplite'),
      'pay_bizum'   => __('Pay with Bizum','shoplite'),
      'no_products' => __('No products yet','shoplite'),
      'cart_empty'  => __('Your cart is empty.','shoplite'),
    );
    $saved = get_option('shoplite_labels', array());
    if (!is_array($saved)) $saved = array();
    return array_merge($defaults, array_map('wp_kses_post', $saved));
    // ⬇️ Overrides from “Text & style”
    //    $opt_empty  = get_option('ts_btn_text_empty',  '');
    //    $opt_update = get_option('ts_btn_text_update','');
    //    if ($opt_empty  !== '') $labels['empty_cart'] = $opt_empty;
    //    if ($opt_update !== '') $labels['update_qty'] = $opt_update;
    //    return $labels;
  }
}  




function shoplite_render_settings_page() {
  if (!current_user_can('manage_options')) return;

  // Save
  if ( isset($_POST['tb_save_settings']) && check_admin_referer('shoplite_save_settings') ) {
    // 1) Button style
    $style = isset($_POST['tb_button_style']) ? sanitize_text_field($_POST['tb_button_style']) : 'clasico';
    update_option('shoplite_button_style', $style);
    // compat: also keep it in the old option in case legacy code reads it
    update_option('estilos_boton', $style);

    // 2) Text labels
    $fields = array(
      'your_cart','add_to_cart','remove','empty_cart','update_qty',
      'subtotal','shipping','total','pay_stripe','pay_bizum',
      'no_products','cart_empty'
    );
    $labels = array();
    foreach ($fields as $k) {
      $labels[$k] = isset($_POST['tb_labels'][$k]) ? sanitize_text_field($_POST['tb_labels'][$k]) : '';
    }
    update_option('shoplite_labels', $labels);

    // Redirect after saving (prevents resubmit and shows real DB state)
    $dest = add_query_arg('shoplite_saved', 1, menu_page_url('tb-ajustes', false));
    //wp_safe_redirect($dest);
    //exit;
  }

  // Post-save notice (after redirect)
  if ( isset($_GET['tb_saved']) ) {
    echo '<div class="updated"><p>'.esc_html__('Settings saved.','shoplite').'</p></div>';
  }


  $style  = shoplite_get_button_style();
  $labels = shoplite_get_labels();
  ?>
  <div class="wrap">
    <h1><?php esc_html_e('Basic Shop - Text & style','shoplite'); ?></h1>

    <form method="post">
      <?php wp_nonce_field('shoplite_save_settings'); ?>

      <h2><?php esc_html_e('Button style','shoplite'); ?></h2>
      <select name="tb_button_style">
        <?php foreach (array('clasico','moderno','minimalista','elegante','retro','oscuro','neon') as $opt): ?>
          <option value="<?php echo esc_attr($opt); ?>" <?php selected($style, $opt); ?>>
            <?php echo esc_html(ucfirst($opt)); ?>
          </option>
        <?php endforeach; ?>
      </select>

      <h2 style="margin-top:20px;"><?php esc_html_e('Frontend text labels','shoplite'); ?></h2>
      <table class="form-table">
        <tbody>
        <?php
          $rows = array(
            'your_cart'   => __('Title “Your cart”','shoplite'),
            'add_to_cart' => __('Button “Add to cart”','shoplite'),
            'remove'      => __('Button “Remove”','shoplite'),
            'empty_cart'  => __('Button “Empty cart”','shoplite'),
            'update_qty'  => __('Button “Update quantities”','shoplite'),
            'subtotal'    => __('Label “Subtotal”','shoplite'),
            'shipping'    => __('Label “Shipping”','shoplite'),
            'total'       => __('Label “Total”','shoplite'),
            'pay_stripe'  => __('Button “Pay with Stripe”','shoplite'),
            'pay_bizum'   => __('Button “Pay with Bizum”','shoplite'),
            'no_products' => __('Message “No products” (catalog)','shoplite'),
            'cart_empty'  => __('Message “Cart empty”','shoplite'),
          );
          foreach ($rows as $key => $label) : ?>
            <tr>
              <th scope="row"><label for="tb_<?php echo esc_attr($key); ?>"><?php echo esc_html($label); ?></label></th>
              <td><input type="text" id="tb_<?php echo esc_attr($key); ?>"
                         name="tb_labels[<?php echo esc_attr($key); ?>]"
                         value="<?php echo esc_attr($labels[$key]); ?>" class="regular-text"></td>
            </tr>
        <?php endforeach; ?>
        </tbody>
      </table>

      <p><button type="submit" name="tb_save_settings" class="button button-primary"><?php esc_html_e('Save settings','shoplite'); ?></button></p>
    </form>
  </div>
  <?php
}


/* ============================================================
 * Shortcode: catalog with WP_Query + pagination (?slp=)
 * ============================================================ */
function shoplite_shortcode_catalogo( $atts = [], $content = '' ){
  if ( function_exists('shoplite_guard_once') && shoplite_guard_once(__FUNCTION__) ) return '';

  // Optional shortcode attributes
  $atts = shortcode_atts([
    'categoria'   => '',                                   // comma-separated slugs (optional)
    'por_pagina'  => get_option('shoplite_per_page', 12),  // products per page
  ], $atts, 'shoplite_catalogo');

  $estilo = shoplite_get_button_style();
  $L      = function_exists('shoplite_get_labels') ? shoplite_get_labels() : [];

  // Current page URL (to return after adding to cart)
  $ts_menu_slug_id  = get_queried_object_id();
  $ts_menu_slug_url = $ts_menu_slug_id ? get_permalink($ts_menu_slug_id)
                       : home_url( add_query_arg([], $_SERVER['REQUEST_URI'] ?? '/') );

  // Pagination (use ?slp=N to avoid breaking other loops)
  $per_page = max(1, (int)$atts['por_pagina']);
  $ts_menu_slug_page    = isset($_GET['slp']) ? max(1, (int)$_GET['slp'])
             : max(1, (int)get_query_var('paged'), (int)get_query_var('page'));

  // Products query
  $args = [
    'post_type'           => 'shoplite_producto',
    'post_status'         => 'publish',
    'ignore_sticky_posts' => true,
    'posts_per_page'      => $per_page,
    'paged'               => $ts_menu_slug_page,
    'no_found_rows'       => false, // required for max_num_pages
    'orderby'             => ['menu_order' => 'ASC', 'date' => 'DESC'],
  ];

  // Filter by category (if categoria="slug1,slug2")
  if (!empty($atts['categoria'])) {
    $slugs = array_map('sanitize_title', array_map('trim', explode(',', $atts['categoria'])));
    $args['tax_query'] = [[
      'taxonomy' => 'shoplite_categoria',
      'field'    => 'slug',
      'terms'    => $slugs,
    ]];
  }




 // === SEARCH (title/content only) ===
$search = isset($_GET['q']) ? sanitize_text_field( wp_unslash($_GET['q']) ) : '';
if ($search === '' && isset($_GET['s'])) {
  $search = sanitize_text_field( wp_unslash($_GET['s']) );
}

if ($search !== '') {
  // enable native WP search in CPT 'shoplite_producto'
  $args['s'] = $search;

  // if orderby is an array, set a compatible one for 's'
  // (relevance only works when there's 's')
  $args['orderby'] = 'relevance date';
}

// (optional) make sure other plugins/theme don't modify the query
$args['suppress_filters'] = true;

// (optional) log to inspect the effective query
// error_log('[Shoplite] OLD query args: ' . json_encode($args));




global $wpdb;

/**
 * =========================================================
 * SHOPLITE – Search by TITLE ONLY (robust, filter-proof)
 * =========================================================
 */

$search = isset($search) && is_string($search) ? trim($search) : '';

if ( $search !== '' ) {

    // --- 1) Buscar IDs SOLO por título (robusto ante HTML, categorías, filtros)
    $ids = $wpdb->get_col(
        $wpdb->prepare(
            "SELECT ID
             FROM {$wpdb->posts}
             WHERE post_type = %s
               AND post_status = 'publish'
               AND INSTR(post_title, %s) > 0
             ORDER BY post_date DESC
             LIMIT 500",
            'shoplite_producto',
            $search
        )
    );

    

    // --- 2) Forzar WP_Query a esos IDs
    $args['post__in'] = ! empty($ids) ? array_map('intval', $ids) : array(0);

    // --- 3) Desactivar COMPLETAMENTE la búsqueda nativa de WP
    unset($args['s']);
    unset($args['search_columns']);

    // --- 4) Ordenar respetando el orden de IDs encontrados
    $args['orderby'] = 'post__in';
    $args['order']   = 'ASC';

    // --- 5) Evitar interferencias externas
    $args['suppress_filters'] = true;

    // --- 6) Evitar páginas vacías durante búsqueda
    $args['paged'] = 1;
}

global $wpdb;

// === DEBUG: estado real en BD (productos y categorías) ===
if ( defined('WP_DEBUG') && WP_DEBUG ) {

    $total = (int) $wpdb->get_var(
        "SELECT COUNT(1)
         FROM {$wpdb->posts}
         WHERE post_type = 'shoplite_producto'
           AND post_status = 'publish'"
    );

    $with_cat = (int) $wpdb->get_var(
        "SELECT COUNT(DISTINCT p.ID)
         FROM {$wpdb->posts} p
         INNER JOIN {$wpdb->term_relationships} tr ON tr.object_id = p.ID
         INNER JOIN {$wpdb->term_taxonomy} tt ON tt.term_taxonomy_id = tr.term_taxonomy_id
         WHERE p.post_type = 'shoplite_producto'
           AND p.post_status = 'publish'
           AND tt.taxonomy = 'shoplite_categoria'"
    );

    $without_cat = $total - $with_cat;

    // muestra 10 títulos que SÍ tienen categoría (para comprobar qué hay realmente en post_title)
    $samples = $wpdb->get_results(
        "SELECT p.ID, p.post_title
         FROM {$wpdb->posts} p
         INNER JOIN {$wpdb->term_relationships} tr ON tr.object_id = p.ID
         INNER JOIN {$wpdb->term_taxonomy} tt ON tt.term_taxonomy_id = tr.term_taxonomy_id
         WHERE p.post_type = 'shoplite_producto'
           AND p.post_status = 'publish'
           AND tt.taxonomy = 'shoplite_categoria'
         ORDER BY p.ID DESC
         LIMIT 10"
    );

    $sample_titles = [];
    foreach ( (array) $samples as $row ) {
        $sample_titles[] = [ (int)$row->ID, (string)$row->post_title ];
    }
  
}




// =========================================================
// Ejecutar la query FINAL
// =========================================================
$q = new WP_Query($args);






  ob_start(); ?>
  <div class="tienda-catalogo">
    <?php if ($q->have_posts()) : ?>
      <div class="tb-grid">
        <?php while ($q->have_posts()) : $q->the_post();
          $pid     = get_the_ID();
          $nombre  = get_the_title();
          $precio  = (float) get_post_meta($pid, '_tb_price', true);
          if ($precio <= 0) { continue; }
          $img     = get_the_post_thumbnail_url($pid, 'medium');
          $url     = get_permalink($pid);
          $add_url = shoplite_add_to_cart_url($pid, $ts_menu_slug_url); // your existing helper
        ?>
          <div class="tb-card">
            <?php if ($img): ?>
              <a class="tb-thumb" href="<?php echo esc_url($url); ?>">
                <img src="<?php echo esc_url($img); ?>" alt="<?php echo esc_attr($nombre); ?>">
              </a>
            <?php endif; ?>

            <h4 class="tb-title">
              <a href="<?php echo esc_url($url); ?>"><?php echo esc_html($nombre); ?></a>
            </h4>

            <div class="tb-price">€<?php echo esc_html(number_format($precio, 2)); ?></div>

            <div class="tb-actions">
              <a href="<?php echo esc_url($add_url); ?>"
                 <?php echo function_exists('ts_get_btn_style_attr') ? ts_get_btn_style_attr() : ''; ?>
                 class="boton-tienda <?php echo esc_attr($estilo); ?> tb-add">
                <?php echo esc_html( function_exists('ts_get_btn_text') ? ts_get_btn_text('cart') : get_option('ts_btn_text_cart', __('Add to cart','shoplite')) ); ?>
              </a>

              <a class="tb-link" href="<?php echo esc_url($url); ?>">
                <?php echo esc_html__('View details','shoplite'); ?>
              </a>
            </div>
          </div>
        <?php endwhile; ?>
      </div>

      <?php
        // Pager (?slp=N)
        if ($q->max_num_pages > 1) {
          $base = add_query_arg('slp', '%#%', $ts_menu_slug_url);
          echo '<nav class="tb-pagination">';
          echo paginate_links([
            'base'      => esc_url($base),
            'format'    => '',
            'current'   => $ts_menu_slug_page,
            'total'     => (int) $q->max_num_pages,
            'prev_text' => '« '. __('Previous','shoplite'),
            'next_text' => __('Next','shoplite') .' »',
            'type'      => 'list',
          ]);
          echo '</nav>';
        }
        wp_reset_postdata();
      ?>

    <?php else: ?>
      <p><?php echo esc_html($L['no_products'] ?? __('No products found.','shoplite')); ?></p>
    <?php endif; ?>
  </div>
  <?php
  return ob_get_clean();
}





/* ============================================================
 * Shortcode: cart (reviewed and robust with session)
 * ============================================================ */
function shoplite_shortcode_carrito(){
  if ( function_exists('shoplite_guard_once') && shoplite_guard_once(__FUNCTION__) ) return '';

if ( function_exists('shoplite_session_start') ) {
  shoplite_session_start();
} else {
  if (!session_id()) @session_start();
}

  $legacy  = isset($_SESSION['carrito'])       ? (array) $_SESSION['carrito']       : [];
  $new     = isset($_SESSION['shoplite_cart']) ? (array) $_SESSION['shoplite_cart'] : [];
  $carrito = !empty($new) ? $new : $legacy;

  // Sync both keys so the rest of the plugin sees the same data
  $_SESSION['shoplite_cart'] = $carrito;
  $_SESSION['carrito']       = $carrito;

  // Debug log (optional)
  if (function_exists('error_log')) {
    error_log('[shoplite] CART render: items=' . count($carrito));
  }

  $estilo = function_exists('shoplite_get_button_style') ? shoplite_get_button_style() : '';
  $L      = function_exists('shoplite_get_labels') ? shoplite_get_labels() : [];

  ob_start(); ?>
  <div class="tienda-carrito">
    <h3><?php echo esc_html($L['your_cart'] ?? __('Your cart','shoplite')); ?></h3>

    <?php if (!empty($carrito)) : ?>
      <form method="post" class="carrito-form" action="">
        <?php wp_nonce_field( SHOPLITE_NONCE_CART_ACTION ); ?>
        <input type="hidden" name="nonce" value="<?php echo esc_attr( wp_create_nonce( SHOPLITE_NONCE_CART_ACTION ) ); ?>">
        <div class="grid-carrito">
          <?php
            $subtotal    = 0.0;
            $envio_total = 0.0;
            $touched     = false; // persist auto-heal only once at the end

            foreach ($carrito as $lineKey => $item) {
              $prodId = isset($item['id']) ? (int)$item['id']
                       : (preg_match('/^tbp_(\d+)/', (string)$lineKey, $m) ? (int)$m[1] : 0);

              // Fill fisico/envio if missing (compat with old items)
              if (!isset($item['fisico'])) {
                $item['fisico'] = ($prodId && get_post_meta($prodId, '_tb_is_physical', true) === '1') ? 1 : 0;
                $item['envio']  = $prodId ? (float) get_post_meta($prodId, '_tb_shipping_cost', true) : 0.0;
                $carrito[$lineKey] = $item;
                $touched = true;
              }

              $precio = (float)($item['precio'] ?? $item['price'] ?? 0);
              $cant   = max(1, (int)($item['cantidad'] ?? $item['qty'] ?? 1));

              $sub         = $precio * $cant;
              $subtotal   += $sub;
              $envio_unit  = !empty($item['fisico']) ? (float)($item['envio'] ?? 0) : 0.0;
              $envio_total += $envio_unit * $cant;

              $nombre = $item['nombre'] ?? ($prodId ? ('Product '.$prodId) : __('Product','shoplite'));
          ?>
            <div class="card-carrito" data-id="<?php echo esc_attr($lineKey); ?>">
              <div class="carrito-row">
                <div class="carrito-info">
                  <div class="carrito-nombre"><?php echo esc_html($nombre); ?></div>
                  <div class="carrito-precio-unit">€<?php echo esc_html(number_format($precio, 2)); ?> / <?php echo esc_html__('unit','shoplite'); ?></div>
                </div>
                <div class="carrito-cantidad">
                  <label class="sr-only" for="qty-<?php echo esc_attr($lineKey); ?>">
                    <?php echo esc_html__('Quantity','shoplite'); ?>
                  </label>
                  <input id="qty-<?php echo esc_attr($lineKey); ?>" type="number"
                         name="qty[<?php echo esc_attr($lineKey); ?>]"
                         value="<?php echo esc_attr($cant); ?>" min="1" step="1">
                </div>
              </div>

              <div class="carrito-row">
                <div class="carrito-subtotal"><strong>€<?php echo esc_html(number_format($sub, 2)); ?></strong></div>
                <div class="carrito-actions">
                  <button type="submit" name="remove" value="<?php echo esc_attr($lineKey); ?>"
                          <?php echo function_exists('ts_get_btn_style_attr') ? ts_get_btn_style_attr() : ''; ?>
                          class="boton-tienda <?php echo esc_attr($estilo); ?> tb-remove btn-borrar">
                    <?php echo esc_html(get_option('ts_btn_text_remove', __('Remove','shoplite'))); ?>
                  </button>
                </div>
              </div>
            </div>
          <?php } // foreach ?>

          <?php
            // If we touched items (auto-heal), persist NOW only once
            if ($touched) {
              $_SESSION['shoplite_cart'] = $carrito;
              $_SESSION['carrito']       = $carrito;
            }
          ?>
        </div>

        <?php
          // Base totals (before coupons)
          $totals = [
            'subtotal' => (float) $subtotal,
            'shipping' => (float) $envio_total,
            'discount' => 0.0,
            'total'    => (float) $subtotal + (float) $envio_total,
          ];

          // Allow Shoplite Pro (or others) to modify totals (apply coupons, etc.)
          if ( function_exists('apply_filters') ) {
            $totals = apply_filters('shoplite_cart_totals', $totals, $carrito);
          }
        ?>

        <div class="carrito-total">
          <div class="row">
            <span class="label"><?php echo esc_html($L['subtotal'] ?? __('Subtotal','shoplite')); ?>:</span>
            <span class="value">
              €<?php echo esc_html( number_format($totals['subtotal'], 2) ); ?>
            </span>
          </div>

          <?php if ( ! empty($totals['shipping']) ) : ?>
            <div class="row">
              <span class="label"><?php echo esc_html($L['shipping'] ?? __('Shipping','shoplite')); ?>:</span>
              <span class="value">
                €<?php echo esc_html( number_format($totals['shipping'], 2) ); ?>
              </span>
            </div>
          <?php endif; ?>

          <?php if ( ! empty($totals['discount']) ) : ?>
            <div class="row">
              <span class="label"><?php echo esc_html($L['discount'] ?? __('Discount','shoplite')); ?>:</span>
              <span class="value">
                -€<?php echo esc_html( number_format($totals['discount'], 2) ); ?>
              </span>
            </div>
          <?php endif; ?>

          <div class="row total">
            <span class="label"><?php echo esc_html($L['total'] ?? __('Total','shoplite')); ?>:</span>
            <span class="value">
              <strong>€<?php echo esc_html( number_format($totals['total'], 2) ); ?></strong>
            </span>
          </div>
        </div>


        
        <?php
          // Session for current coupon + detailed coupon info (Pro)
          if ( function_exists( 'shoplite_maybe_session_start' ) ) {
              shoplite_maybe_session_start();
          } else {
              if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
                  @session_start();
              }
          }

          $coupon_current = isset($_SESSION['shoplite_coupon_code'])
            ? $_SESSION['shoplite_coupon_code']
            : '';

          $coupon_info = isset($_SESSION['shoplite_coupon_applied'])
            ? $_SESSION['shoplite_coupon_applied']
            : null;

          if ( $coupon_info && ! empty($coupon_info['code']) && ! empty($totals['discount']) ) {
              $code_label = strtoupper( (string) $coupon_info['code'] );
              $names      = isset($coupon_info['products']) && is_array($coupon_info['products'])
                              ? array_filter(array_map('trim', $coupon_info['products']))
                              : [];
              $is_global  = ! empty($coupon_info['global']);
              ?>
              <div class="carrito-cupon-info"
                   style="margin:8px 0 12px 0; padding:8px 10px; background:#f5f7fb; border-left:3px solid #2271b1; font-size:0.9rem;">
                <?php if ( $is_global || empty($names) ) : ?>
                  <?php
                    printf(
                      /* translators: %s is the coupon code */
                      esc_html__('Coupon %s applied to the cart total.', 'shoplite'),
                      esc_html($code_label)
                    );
                  ?>
                <?php else : ?>
                  <?php
                    $list = implode(', ', $names);
                    printf(
                      /* translators: 1: coupon code, 2: product list */
                      esc_html__('Coupon %1$s applied to: %2$s', 'shoplite'),
                      esc_html($code_label),
                      esc_html($list)
                    );
                  ?>
                <?php endif; ?>
              </div>
              <?php
          }
        ?>

        <div class="carrito-cupon" style="margin-top:12px;">
          <p><strong><?php echo esc_html( $L['coupon_label'] ?? __('Do you have a coupon?','shoplite') ); ?></strong></p>
          <p style="display:flex;gap:6px;flex-wrap:wrap;margin-top:4px;">
            <input type="text"
                   name="shoplite_coupon_code"
                   value="<?php echo esc_attr($coupon_current); ?>"
                   placeholder="<?php echo esc_attr( $L['coupon_placeholder'] ?? __('Enter your coupon code','shoplite') ); ?>">
            <button type="submit"
                    name="shoplite_apply_coupon"
                    value="1"
                    <?php echo function_exists('ts_get_btn_style_attr') ? ts_get_btn_style_attr() : ''; ?>
                    class="boton-tienda <?php echo esc_attr($estilo); ?>">
              <?php echo esc_html( $L['coupon_apply'] ?? __('Apply coupon','shoplite') ); ?>
            </button>
          </p>
        </div>


                <p class="carrito-actions-footer">

          <button type="submit" name="update" value="1"
                  <?php echo function_exists('ts_get_btn_style_attr') ? ts_get_btn_style_attr() : ''; ?>
                  class="boton-tienda <?php echo esc_attr($estilo); ?> tb-update">
            <?php echo esc_html(get_option('ts_btn_text_update', __('Update quantities','shoplite'))); ?>
          </button>

          <button type="submit" name="empty" value="1"
                  <?php echo function_exists('ts_get_btn_style_attr') ? ts_get_btn_style_attr() : ''; ?>
                  class="boton-tienda <?php echo esc_attr($estilo); ?> tb-empty">
            <?php echo esc_html(get_option('ts_btn_text_empty', __('Empty cart','shoplite'))); ?>
          </button>
        </p>

        <?php
          // Prefill email from session (used for checkout + abandoned carts)
          if ( function_exists( 'shoplite_maybe_session_start' ) ) {
            shoplite_maybe_session_start();
          } else {
            if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
              @session_start();
            }
          }
          $prefill_email = isset($_SESSION['shoplite_customer_email'])
            ? sanitize_email((string) $_SESSION['shoplite_customer_email'])
            : '';
        ?>

        <div class="shoplite-checkout-customer" style="margin: 14px 0;">
          <label for="shoplite-email" style="display:block; margin-bottom:6px;">
            <?php echo esc_html__( 'Email', 'shoplite' ); ?>
          </label>

          <input
            id="shoplite-email"
            type="email"
            name="shoplite_email"
            placeholder="you@example.com"
            autocomplete="email"
            required
            style="width:100%; max-width:520px; padding:10px;"
            value="<?php echo esc_attr( $prefill_email ); ?>"
          />

          <p style="margin:6px 0 0; opacity:.8; font-size:13px;">
            <?php echo esc_html__( 'We’ll use this email to send your receipt and (if needed) a cart recovery reminder.', 'shoplite' ); ?>
          </p>
        </div>

        <!-- Payment buttons (Stripe + Bizum + PayPal) -->
        <div class="em-pay-buttons" style="margin-top:14px;">
          <?php if ( function_exists('shoplite_stripe_pubkey') && shoplite_stripe_pubkey() ) : ?>
            <div class="pay-slot">
              <button
                type="button"
                id="tb-pagar-stripe"
                class="boton-tienda <?php echo esc_attr($estilo); ?> stripe-btn"
                data-nonce="<?php echo esc_attr( wp_create_nonce( SHOPLITE_NONCE_STRIPE ) ); ?>">
                <?php echo esc_html($L['pay_stripe'] ?? __('Pay with card','shoplite')); ?>
              </button>
            </div>



    <?php
      $shoplite_disable_bizum = ( defined('SHOPLITE_DISABLE_BIZUM') && SHOPLITE_DISABLE_BIZUM );
      $shoplite_is_eur        = ( strtolower( shoplite_get_currency() ?? 'eur' ) === 'eur' );
    ?>

    <?php if ( $shoplite_is_eur && ! $shoplite_disable_bizum ) : ?>
      <div class="pay-slot">
        <button
          type="button"
          id="tb-pagar-bizum"
          class="boton-tienda <?php echo esc_attr($estilo); ?> stripe-btn alt"
          data-nonce="<?php echo esc_attr( wp_create_nonce( SHOPLITE_NONCE_STRIPE ) ); ?>">
          <?php echo esc_html($L['pay_bizum'] ?? __('Pay with Bizum','shoplite')); ?>
        </button>
      </div>
    <?php endif; ?>

  <?php endif; ?>

          <?php if ( function_exists('shoplite_paypal_client_id') && shoplite_paypal_client_id() ) : ?>
            <div class="pay-slot">
              <div id="tb-paypal-container"></div>
            </div>
          <?php endif; ?>
        </div>

        <?php if ( function_exists('shoplite_show_sandbox_badge') && (shoplite_get_modo_stripe()!=='live' || shoplite_get_modo_paypal()!=='live') ) : ?>
          <div style="margin-top:8px;font-size:.85rem;opacity:.7;">
            <?php if (shoplite_get_modo_stripe()!=='live') echo esc_html__('Stripe: SANDBOX', 'shoplite'); ?>
            <?php if (shoplite_get_modo_paypal()!=='live') echo esc_html__('PayPal: SANDBOX', 'shoplite'); ?>
          </div>
        <?php endif; ?>

      </form>
 <?php else : ?>
  <p><?php echo esc_html($L['cart_empty'] ?? __('Your cart is empty.','shoplite')); ?></p>
<?php endif; ?>
</div>

<?php /* ← Exit PHP to output raw HTML */ ?>
<div id="tb-payments" class="tb-payments" style="margin-top:16px">
  <div id="paypal-buttons"></div>
</div>
<?php /* ← Back to PHP */ ?>

<?php
  return ob_get_clean();
}



// ============================================================
// Shortcode: successful payment page (Stripe)
// ============================================================
if ( ! function_exists( 'shoplite_shortcode_pago_exitoso' ) ) {

    function shoplite_shortcode_pago_exitoso() {

        // Avoid double execution if called multiple times
        if ( function_exists( 'shoplite_guard_once' ) && shoplite_guard_once( __FUNCTION__ ) ) {
            return '';
        }

        // By default we show a generic message
        $mensaje = '<p>' . esc_html__( 'Thank you for your purchase. Your payment was processed successfully.', 'shoplite' ) . '</p>';

        // Check that we are coming back from Stripe
        $payment    = isset( $_GET['payment'] ) ? sanitize_text_field( wp_unslash( $_GET['payment'] ) ) : '';
        $session_id = isset( $_GET['session_id'] ) ? sanitize_text_field( wp_unslash( $_GET['session_id'] ) ) : '';

        if ( $payment !== 'stripe' || $session_id === '' ) {
            // Not a Stripe return → just show the generic message
            return $mensaje;
        }

        // Stripe secret key
        if ( ! function_exists( 'shoplite_stripe_seckey' ) ) {
            return $mensaje;
        }

        $secret = shoplite_stripe_seckey();
        if ( empty( $secret ) ) {
            return $mensaje;
        }

        // 1) Prevent creating the same order twice: does it already exist?
        $existing = get_posts( [
            'post_type'      => 'shoplite_pedido',
            'posts_per_page' => 1,
            'fields'         => 'ids',
            'meta_query'     => [
                [
                    'key'   => '_shoplite_stripe_session_id',
                    'value' => $session_id,
                ],
            ],
        ] );

        if ( ! empty( $existing ) ) {
            // Order already registered, just empty the cart and return the message
            if ( function_exists( 'shoplite_empty_cart_sessions' ) ) {
                shoplite_empty_cart_sessions();
            }
            return $mensaje;
        }


      
        // 2) Fetch the session from Stripe
        $resp = wp_remote_get(
            'https://api.stripe.com/v1/checkout/sessions/' . rawurlencode( $session_id ),
            [
                'headers' => [
                    'Authorization' => 'Bearer ' . $secret,
                ],
                'timeout' => 20,
            ]
        );

        if ( is_wp_error( $resp ) ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( '[shoplite][STRIPE-SUCCESS] ' . $resp->get_error_message() );
            }
            return $mensaje;
        }

        $code = wp_remote_retrieve_response_code( $resp );
        $body = wp_remote_retrieve_body( $resp );
        $data = json_decode( $body, true );

        if ( $code < 200 || $code >= 300 || ! is_array( $data ) ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( '[shoplite][STRIPE-SUCCESS] HTTP ' . $code . ' BODY: ' . $body );
            }
            return $mensaje;
        }

        // 3) Extract relevant data from Stripe
        $amount_total  = isset( $data['amount_total'] ) ? (int) $data['amount_total'] : 0;
        $currency      = isset( $data['currency'] ) ? strtoupper( $data['currency'] ) : shoplite_get_currency();
        $status_stripe = isset( $data['payment_status'] ) ? $data['payment_status'] : ( $data['status'] ?? '' );

        $mode_currency = strtolower( $currency );
        $mult          = function_exists( 'shoplite_stripe_amount_multiplier' )
            ? shoplite_stripe_amount_multiplier( $mode_currency )
            : 100;

        if ( $mult <= 0 ) {
            $mult = 100;
        }

        $total = $amount_total > 0 ? ( $amount_total / $mult ) : 0.0;

        $customer_email = $data['customer_details']['email'] ?? '';
        $customer_name  = $data['customer_details']['name']  ?? '';

        // Internal order status
        $order_status = 'pending';
        $normalized   = strtolower( $status_stripe );
        if ( in_array( $normalized, [ 'paid', 'complete', 'completed' ], true ) ) {
            $order_status = 'completed';
        }

        // 4) Create internal order using Shoplite_Orders + session (cart) data
        $order_id = 0;

        if ( class_exists( 'Shoplite_Orders' ) && function_exists( 'shoplite_build_order_data_from_session' ) ) {

            // Base data built from session (cart)
            $order_data = shoplite_build_order_data_from_session( 'stripe' );

            if ( $order_data ) {
                // Override with Stripe "official" values where it makes sense
                $order_data['total']          = (float) $total;
                $order_data['currency']       = $currency ?: ( $order_data['currency'] ?? 'EUR' );
                $order_data['email']          = $customer_email ?: ( $order_data['email'] ?? '' );
                $order_data['name']           = $customer_name ?: ( $order_data['name'] ?? '' );
                $order_data['status']         = $order_status;
                $order_data['payment_method'] = 'stripe';

                // This will create CPT shoplite_pedido and save _shoplite_order_items
                $order_id = Shoplite_Orders::create_from_checkout( $order_data );
            }
        }

        // If it couldn't be created for any reason, we don't attempt to "guess" anything:
        if ( is_wp_error( $order_id ) || ! $order_id ) {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( '[shoplite][STRIPE-SUCCESS] Could not create the internal order for session ' . $session_id );
            }
            // Still empty the cart to avoid leftovers
            if ( function_exists( 'shoplite_empty_cart_sessions' ) ) {
                shoplite_empty_cart_sessions();
            }
            return $mensaje;
        }

                // Save Stripe-specific metadata on the newly created order
        update_post_meta( $order_id, '_shoplite_order_gateway',        'stripe' );
        update_post_meta( $order_id, '_shoplite_stripe_session_id',    $session_id );

        if ( $customer_email ) {
            update_post_meta( $order_id, '_shoplite_customer_email', sanitize_email( $customer_email ) );
        }
        if ( $customer_name ) {
            update_post_meta( $order_id, '_shoplite_customer_name', sanitize_text_field( $customer_name ) );
        }

        // ✅ NEW: fire "order paid" hook (only for completed payments)
        if ( $order_status === 'completed' ) {
            // Guard against double firing (page refreshes, etc.)
            if ( ! get_post_meta( $order_id, '_shoplite_delivery_tokens_created', true ) ) {
                do_action( 'shoplite_order_paid', $order_id );
            }
        }

        // 5) Empty cart / session after registering the order
        if ( function_exists( 'shoplite_empty_cart_sessions' ) ) {
            shoplite_empty_cart_sessions();
        } else {
            // Fallback, just in case
            if ( function_exists( 'shoplite_maybe_session_start' ) ) {
                shoplite_maybe_session_start();
            } elseif ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
                @session_start();
            }
            unset(
                $_SESSION['shoplite_cart'],
                $_SESSION['carrito'],
                $_SESSION['shoplite_coupon_code']
            );
        }


        // Return the success message
        return $mensaje;
    }
}


/**
 * Clears the cart and coupon session variables.
 */
if ( ! function_exists( 'shoplite_empty_cart_sessions' ) ) {
    function shoplite_empty_cart_sessions() {
        if ( function_exists( 'shoplite_maybe_session_start' ) ) {
            shoplite_maybe_session_start();
        } elseif ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
            @session_start();
        }

        unset(
            $_SESSION['shoplite_cart'],
            $_SESSION['carrito'],
            $_SESSION['shoplite_coupon_code']
        );
    }
}

// Register shortcode
add_shortcode( 'shoplite_pago_exitoso', 'shoplite_shortcode_pago_exitoso' );




// Disable cache on pages with our shortcodes and when there are cart actions
add_action('template_redirect', function () {
  if (is_admin()) return;

  $sensitiva = false;

  // 1) Pages using our shortcodes
  if (function_exists('is_singular') && is_singular()) {
    global $post; $c = $post->post_content ?? '';
    if ($c && ( has_shortcode($c, 'tienda_carrito')
             || has_shortcode($c, 'tienda_catalogo')
             || has_shortcode($c, 'tienda_basica') )) {
      $sensitiva = true;
    }
  }

  // 2) Cart actions (GET/POST)
  if ( 
    isset($_GET['add_to_cart']) 
    || isset($_GET['eliminar_item']) 
    || isset($_GET['vaciar_carrito'])
    || isset($_POST['tienda_basica_update_qty'])
    || isset($_POST['shoplite_coupon_code']) 
  ) {
    $sensitiva = true;
  }

// Process coupon code submitted from checkout
if ( isset($_POST['shoplite_coupon_code']) ) {
    $code = sanitize_text_field( wp_unslash($_POST['shoplite_coupon_code']) );

    // Session is already started in init, so normally session_start() isn't needed.
    // If you want to be extra safe:
    if ( function_exists( 'shoplite_maybe_session_start' ) ) {
        shoplite_maybe_session_start();
    } else {
        if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
            @session_start();
        }
    }

    $_SESSION['shoplite_coupon_code'] = $code;
}


  if (!$sensitiva) return;

  // Standard anti-cache signals (LiteSpeed, WP Super Cache, Varnish/CDN…)
  if (!defined('DONOTCACHEPAGE'))   define('DONOTCACHEPAGE',   true);
  if (!defined('DONOTCACHEOBJECT')) define('DONOTCACHEOBJECT', true);
  if (!defined('DONOTCACHEDB'))     define('DONOTCACHEDB',     true);

  if (function_exists('nocache_headers')) nocache_headers();
  @header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
  @header('Pragma: no-cache');
}, 0);

add_shortcode('tienda_catalogo','shoplite_shortcode_catalogo');
add_shortcode('tienda_carrito','shoplite_shortcode_carrito');
/* Combined shortcode */
add_shortcode('tienda_basica', function(){ return shoplite_shortcode_catalogo() . shoplite_shortcode_carrito(); });

// === Shortcode aliases for "Talariasoft Shoplite" ===
add_shortcode('talariasoft_shop_lite', function () {
  return shoplite_shortcode_catalogo() . shoplite_shortcode_carrito();
});
add_shortcode('tsl_catalog', 'shoplite_shortcode_catalogo');
add_shortcode('tsl_cart',    'shoplite_shortcode_carrito');





/* ============================================================
 * AJAX: Stripe Checkout via REST
 * ============================================================ */
add_action('wp_ajax_tb_stripe_create_checkout','shoplite_stripe_create_checkout');
add_action('wp_ajax_nopriv_tb_stripe_create_checkout','shoplite_stripe_create_checkout');


/* ============================================================
 * AJAX: Stripe Checkout via REST (unified)
 * ============================================================ */
function shoplite_stripe_create_checkout() {

    // 1) Receive nonce (accept multiple names for compatibility)
    $nonce = null;

    if ( isset($_POST['nonce']) ) {
        $nonce = sanitize_text_field( wp_unslash($_POST['nonce']) );
    } elseif ( isset($_POST['stripeNonce']) ) {
        // legacy / other JS
        $nonce = sanitize_text_field( wp_unslash($_POST['stripeNonce']) );
    } elseif ( isset($_POST['tb_stripe_nonce']) ) {
        // additional compatibility
        $nonce = sanitize_text_field( wp_unslash($_POST['tb_stripe_nonce']) );
    }

    // Diagnostic log (avoid dumping sensitive fields in production)
    if ( defined('WP_DEBUG') && WP_DEBUG && function_exists('error_log') ) {
        $safe_post = $_POST;
        if ( isset($safe_post['nonce']) )             $safe_post['nonce'] = '***';
        if ( isset($safe_post['stripeNonce']) )       $safe_post['stripeNonce'] = '***';
        if ( isset($safe_post['tb_stripe_nonce']) )   $safe_post['tb_stripe_nonce'] = '***';
        error_log('[shoplite][STRIPE-AJAX] POST=' . print_r($safe_post, true));
        error_log('[shoplite][STRIPE-AJAX] nonce_received=' . ($nonce ? 'yes' : 'no'));
    }

    // 2) Validate nonce
    if ( empty($nonce) || ! wp_verify_nonce( $nonce, SHOPLITE_NONCE_STRIPE ) ) {
        wp_send_json_error([
            'message' => esc_html__( 'Invalid nonce (Stripe).', 'shoplite' ),
        ], 403);
    }

    // 3) Session + cart (canonical + compatibility)
    if ( function_exists( 'shoplite_maybe_session_start' ) ) {
        shoplite_maybe_session_start();
    } else {
        if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
            @session_start();
        }
    }

    $cart = $_SESSION['shoplite_cart'] ?? ($_SESSION['carrito'] ?? []);
    if ( empty($cart) || ! is_array($cart) ) {
        wp_send_json_error([
            'message' => esc_html__( 'Your cart is empty.', 'shoplite' )
        ], 400);
    }

    /**
     * =========================================================
     * ✅ NEW: Inventory validation (server-side, pre-checkout)
     * =========================================================
     *
     * - Extract product_id from:
     *   - $item['product_id'] OR
     *   - $lineKey like "tbp_123"
     * - Extract qty from:
     *   - $item['cantidad'] OR $item['qty']
     * - Only validate products with _sl_manage_stock = '1'
     * - Stock is taken from _sl_stock_qty (int)
     */
    try {
        $lines = [];

        foreach ( $cart as $lineKey => $item ) {
            if ( ! is_array($item) ) continue;

            $qty = isset($item['cantidad'])
                ? (int) $item['cantidad']
                : ( isset($item['qty']) ? (int) $item['qty'] : 1 );

            $qty = max(1, $qty);

            $pid = isset($item['product_id']) ? (int) $item['product_id'] : 0;

            if ( $pid <= 0 && is_string($lineKey) && preg_match('/^tbp_(\d+)$/', (string)$lineKey, $m) ) {
                $pid = (int) $m[1];
            }

            // If we can't resolve PID, we can't stock-check that line (ignore gracefully)
            if ( $pid <= 0 ) continue;

            // Aggregate by product_id (same product across lines)
            if ( ! isset($lines[$pid]) ) $lines[$pid] = 0;
            $lines[$pid] += $qty;
        }

        foreach ( $lines as $pid => $qty ) {

            $manage = get_post_meta( (int)$pid, '_sl_manage_stock', true );
            if ( $manage !== '1' ) continue;

            $stock = (int) get_post_meta( (int)$pid, '_sl_stock_qty', true );

            if ( $stock < (int)$qty ) {
                // Optional: show a nicer message with product title
                $title = get_the_title( (int)$pid );
                if ( ! $title ) $title = sprintf('#%d', (int)$pid);

                $msg = sprintf(
                    /* translators: 1: product title, 2: requested qty, 3: available qty */
                    __( 'Not enough stock for "%1$s". Requested: %2$d. Available: %3$d.', 'shoplite' ),
                    $title,
                    (int)$qty,
                    max(0, (int)$stock)
                );

                throw new Exception( $msg );
            }
        }

    } catch ( Exception $e ) {
        if ( defined('WP_DEBUG') && WP_DEBUG && function_exists('error_log') ) {
            error_log('[shoplite][INVENTORY] stock validation failed: ' . $e->getMessage());
        }

        wp_send_json_error([
            'message' => $e->getMessage(),
        ], 409);
    }

    // 4) Secret key
    if ( ! function_exists('shoplite_stripe_seckey') ) {
        wp_send_json_error(['message' => esc_html__( 'Stripe is not available (missing key helper).', 'shoplite' )], 500);
    }
    $secret = (string) shoplite_stripe_seckey();
    if ( $secret === '' ) {
        wp_send_json_error(['message' => esc_html__( 'Please configure your Stripe secret key in the admin.', 'shoplite' )], 500);
    }

    // 5) URLs configured in the backend (required)
    $success_base = function_exists('shoplite_get_success_url') ? (string) shoplite_get_success_url() : '';
    $cancel_base  = function_exists('shoplite_get_cancel_url')  ? (string) shoplite_get_cancel_url()  : '';

    if ( $success_base === '' || $cancel_base === '' ) {
        wp_send_json_error([
            'message' => esc_html__( 'Please configure the Success and Cancel URLs in the backend (Shoplite → Payments).', 'shoplite' )
        ], 500);
    }

    // 6) Currency and Stripe multiplier
    $currency = function_exists('shoplite_get_currency')
        ? strtolower( (string) shoplite_get_currency() )
        : 'eur';

    $mult = function_exists('shoplite_stripe_amount_multiplier')
        ? (int) shoplite_stripe_amount_multiplier( $currency )
        : 100;

    if ( $mult < 1 ) $mult = 100;

    // Build URLs (respecting existing query strings)
    $success = add_query_arg(
        ['payment' => 'stripe', 'session_id' => '{CHECKOUT_SESSION_ID}'],
        $success_base
    );
    $cancel  = add_query_arg(
        ['payment' => 'stripe-cancel'],
        $cancel_base
    );

    // Base session payload (form-encoded)
    $body = [
        'mode'        => 'payment',
        'success_url' => $success,
        'cancel_url'  => $cancel,
        // payment methods decided by Stripe dashboard / account settings
    ];

    /**
     * =========================================================
     * ✅ Optional (NEW): attach a reservation token
     * =========================================================
     * This doesn't enforce stock by itself, but it lets you implement
     * true reservation/anti-oversell later with minimal changes.
     */
    $reservation_token = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : (string) time() . '-' . wp_rand(1000, 9999);
    $body["metadata[sl_reservation]"] = $reservation_token;
    $body["payment_intent_data[metadata][sl_reservation]"] = $reservation_token;

    // =========================
    // Calculate SUBTOTAL + SHIPPING
    // =========================
    $subtotal    = 0.0;
    $envio_total = 0.0;

    foreach ( $cart as $lineKey => $item ) {

        if ( ! is_array($item) ) continue;

        $precio = isset($item['precio'])
            ? (float) $item['precio']
            : ( isset($item['price']) ? (float) $item['price'] : 0.0 );

        $cant = isset($item['cantidad'])
            ? max(1, (int) $item['cantidad'])
            : ( isset($item['qty']) ? max(1, (int) $item['qty']) : 1 );

        $subtotal += $precio * $cant;

        $es_fisico  = ! empty($item['fisico']);
        $envio_unit = 0.0;

        if ( $es_fisico ) {
            $envio_unit = (float) ( $item['envio'] ?? 0 );
        } elseif ( is_string($lineKey) && preg_match('/^tbp_(\d+)$/', (string) $lineKey, $m) ) {
            $pid       = (int) $m[1];
            $es_fisico = ( get_post_meta($pid, '_tb_is_physical', true) === '1' );
            if ( $es_fisico ) {
                $envio_unit = (float) get_post_meta($pid, '_tb_shipping_cost', true);

                // Optional: update session for future calls
                if ( isset($_SESSION['shoplite_cart'][$lineKey]) && is_array($_SESSION['shoplite_cart'][$lineKey]) ) {
                    $_SESSION['shoplite_cart'][$lineKey]['fisico'] = 1;
                    $_SESSION['shoplite_cart'][$lineKey]['envio']  = $envio_unit;
                }
                if ( isset($_SESSION['carrito'][$lineKey]) && is_array($_SESSION['carrito'][$lineKey]) ) {
                    $_SESSION['carrito'][$lineKey]['fisico'] = 1;
                    $_SESSION['carrito'][$lineKey]['envio']  = $envio_unit;
                }
            }
        }

        if ( $es_fisico && $envio_unit > 0 ) {
            $envio_total += $envio_unit * $cant;
        }
    }

    // =========================
    // Apply COUPON (Pro) to subtotal only
    // =========================
    $subtotal_original = $subtotal;
    $discount          = 0.0;

    $coupon_code = isset($_SESSION['shoplite_coupon_code'])
        ? trim( (string) $_SESSION['shoplite_coupon_code'] )
        : '';

    if (
        defined('SHOPLITE_PRO') &&
        class_exists('Shoplite_Pro_Coupons') &&
        method_exists('Shoplite_Pro_Coupons', 'apply_coupon_to_total')
    ) {
        $coupon_result = Shoplite_Pro_Coupons::apply_coupon_to_total(
            (float) $subtotal,
            $coupon_code !== '' ? $coupon_code : null
        );

        if ( is_array($coupon_result) && isset($coupon_result['total']) ) {
            $subtotal = (float) $coupon_result['total'];
            $discount = isset($coupon_result['discount']) ? (float) $coupon_result['discount'] : 0.0;
        }
    }

    // FINAL TOTAL for Stripe = subtotal (with coupon) + shipping
    $grand_total = $subtotal + $envio_total;

    if ( $grand_total <= 0 ) {
        wp_send_json_error(['message' => esc_html__( 'Order total is zero or negative.', 'shoplite' )], 400);
    }

    // =========================
    // SINGLE line item with the total already discounted
    // =========================
    $body["line_items[0][price_data][currency]"]            = $currency;
    $body["line_items[0][price_data][product_data][name]"]  = __( 'Shoplite order', 'shoplite' );
    $body["line_items[0][price_data][unit_amount]"]         = (int) round( $grand_total * $mult );
    $body["line_items[0][quantity]"]                        = 1;

    // =========================================================
    // (Optional legacy defaults) - keep if you want, but bridge can override
    // =========================================================
    $domain = parse_url( home_url('/'), PHP_URL_HOST ) ?: '';
    $sl_product = 'shoplite_pro';
    $sl_plan    = 'pro';

    $body["metadata[sl_product]"] = $sl_product;
    $body["metadata[sl_plan]"]    = $sl_plan;
    $body["metadata[sl_source]"]  = "shoplite_plugin";
    $body["metadata[sl_domain]"]  = $domain;

    $body["payment_intent_data[metadata][sl_product]"] = $sl_product;
    $body["payment_intent_data[metadata][sl_plan]"]    = $sl_plan;
    $body["payment_intent_data[metadata][sl_source]"]  = "shoplite_plugin";
    $body["payment_intent_data[metadata][sl_domain]"]  = $domain;

    // =========================================================
    // IMPORTANT: bridge hook point (MU-plugin injects sl_product/sl_plan/sl_qty...)
    // =========================================================
    $body = apply_filters('shoplite_stripe_checkout_body', $body, $cart, [
        'currency'     => $currency,
        'grand_total'  => $grand_total,
        'subtotal'     => $subtotal,
        'shipping'     => $envio_total,
        'discount'     => $discount,
        'success_url'  => $success,
        'cancel_url'   => $cancel,
    ]);

    // =========================
    // Stripe request
    // =========================
    $res = wp_remote_post('https://api.stripe.com/v1/checkout/sessions', [
        'headers' => [
            'Authorization' => 'Bearer ' . $secret,
            'Content-Type'  => 'application/x-www-form-urlencoded',
        ],
        'body'    => $body,
        'timeout' => 25,
    ]);

    if ( is_wp_error($res) ) {
        wp_send_json_error(['message' => $res->get_error_message()], 502);
    }

    $code = wp_remote_retrieve_response_code($res);
    $data = json_decode(wp_remote_retrieve_body($res), true);

    if ( $code >= 200 && $code < 300 && ! empty($data['id']) ) {
        wp_send_json_success([
            'id' => $data['id'],
            // helpful to debug and later implement real reservations
            'reservation' => $reservation_token,
        ]);
    }

    $msg = ! empty($data['error']['message'])
        ? (string) $data['error']['message']
        : esc_html__( 'Failed to create the Stripe session.', 'shoplite' );

    wp_send_json_error(['message' => $msg], 502);
}








/* ============================================================
 * PayPal: OAuth2 token
 * ============================================================ */
function shoplite_paypal_get_token(){
    $client = shoplite_paypal_client_id();
    $secret = shoplite_paypal_secret();
    if (empty($client) || empty($secret)) {
        return new WP_Error(
            'paypal_creds',
            esc_html__( 'Please configure PayPal Client ID and Secret.', 'shoplite' )
        );
    }

    $cache_key = 'tb_paypal_token_'.shoplite_get_modo_paypal();
    $cached = get_transient($cache_key);
    if ($cached) return $cached;

    $url = shoplite_paypal_api_base().'/v1/oauth2/token';
    $args = array(
        'headers' => array(
            'Authorization' => 'Basic '.base64_encode($client.':'.$secret),
            'Accept'        => 'application/json',
            'Accept-Language'=> 'en_US',
        ),
        'body'    => array('grant_type' => 'client_credentials'),
        'timeout' => 20,
    );
    $res = wp_remote_post($url, $args);
    if (is_wp_error($res)) return $res;

    $code = wp_remote_retrieve_response_code($res);
    $body = json_decode(wp_remote_retrieve_body($res), true);
    if ($code >= 200 && $code < 300 && !empty($body['access_token'])){
        set_transient($cache_key, $body['access_token'], min(60*55, (int)($body['expires_in'] ?? 3600)));
        return $body['access_token'];
    }

    // Debug logs (keep as-is; these are for developers)
    error_log("PAYPAL_CAPTURE_FAIL -------------------");
    error_log("ORDER ID: " . ($_POST['orderID'] ?? '??'));
    error_log("BODY: " . print_r($resp, true));
    error_log("CART TOTAL (server): " . $total_calculado);

    return new WP_Error(
        'paypal_token',
        esc_html__( 'Could not obtain a PayPal token.', 'shoplite' )
    );
}


function shoplite_paypal_create_order() {

    // Tu JS envía body.append("nonce", paypalNonce);
    check_ajax_referer( SHOPLITE_NONCE_PAYPAL, 'nonce' );

    // 1) Construir datos del pedido desde sesión (server-side)
    if ( ! function_exists( 'shoplite_build_order_data_from_session' ) ) {
        wp_send_json_error([ 'message' => 'Shoplite order helper not available.' ], 500);
    }

    // Si quieres, aquí puedes leer el cupón que llega por POST,
    // pero lo normal es que ya esté aplicado en sesión por tu lógica.
    // $coupon_code = isset($_POST['coupon_code']) ? sanitize_text_field(wp_unslash($_POST['coupon_code'])) : '';

    $order_data = shoplite_build_order_data_from_session( 'paypal' );

    if ( empty( $order_data ) || ! is_array( $order_data ) ) {
        wp_send_json_error([ 'message' => 'Could not build order data from session.' ], 400);
    }

    // 2) Sacar currency y total
    // Ajusta estas claves si en tu order_data usan otro nombre.
    $currency = strtoupper( (string) ( $order_data['currency'] ?? $order_data['currency_code'] ?? 'EUR' ) );

    // Intenta varias claves típicas para el total
    $total_raw = null;
    foreach ( [ 'total', 'total_amount', 'amount', 'grand_total', 'order_total' ] as $k ) {
        if ( isset( $order_data[ $k ] ) && $order_data[ $k ] !== '' && $order_data[ $k ] !== null ) {
            $total_raw = $order_data[ $k ];
            break;
        }
    }

    if ( $total_raw === null ) {
        // Log para que veas qué estructura trae realmente
        error_log('PAYPAL_CREATE_FAIL: order_data keys = ' . implode(',', array_keys($order_data)));
        wp_send_json_error([ 'message' => 'Order total not found in session order data.' ], 400);
    }

    // Normaliza a float (por si viene como string)
    $total_float = (float) $total_raw;

    if ( $total_float <= 0 ) {
        wp_send_json_error([ 'message' => 'Order total is zero or negative.' ], 400);
    }

    // 3) Formatear amount.value correctamente para PayPal
    // PayPal exige string; 2 decimales normalmente; 0 decimales para monedas "zero-decimal".
    $is_zero_decimal = function_exists('shoplite_paypal_is_zero_decimal')
        ? (bool) shoplite_paypal_is_zero_decimal( $currency )
        : false;

    if ( $is_zero_decimal ) {
        $value = (string) (int) round( $total_float );
    } else {
        $value = number_format( $total_float, 2, '.', '' );
    }

    // 4) Token PayPal
    $token = shoplite_paypal_get_token();
    if ( ! $token ) {
        wp_send_json_error([ 'message' => 'No se pudo obtener token de PayPal.' ], 500);
    }

    // 5) Crear order en PayPal
    $url = shoplite_paypal_api_base() . '/v2/checkout/orders';

    $payload = [
        'intent' => 'CAPTURE',
        'purchase_units' => [[
            'amount' => [
                'currency_code' => $currency,
                'value' => $value,
            ],
        ]],
    ];

    $args = [
        'timeout' => 30,
        'headers' => [
            'Authorization' => 'Bearer ' . $token,
            'Content-Type'  => 'application/json',
            'Prefer'        => 'return=representation',
        ],
        'body' => wp_json_encode( $payload ),
    ];

    $resp = wp_remote_post( $url, $args );

    if ( is_wp_error( $resp ) ) {
        wp_send_json_error([ 'message' => $resp->get_error_message() ], 500);
    }

    $code = wp_remote_retrieve_response_code( $resp );
    $body = wp_remote_retrieve_body( $resp );
    $json = json_decode( $body, true );

    if ( $code < 200 || $code >= 300 ) {
        error_log("PAYPAL_CREATE_FAIL HTTP $code");
        error_log("PAYPAL_CREATE_FAIL REQ: " . wp_json_encode($payload));
        error_log("PAYPAL_CREATE_FAIL RES: $body");
        wp_send_json_error([
            'message' => 'PayPal create order failed',
            'http'    => $code,
            'pp'      => $json ?: $body,
        ], 500);
    }

    $id = $json['id'] ?? '';
    if ( ! $id ) {
        wp_send_json_error([ 'message' => 'PayPal no devolvió id de orden.', 'pp' => $json ], 500);
    }

    // Tu JS espera json.data.id
    wp_send_json_success([ 'id' => $id ]);
}



/**
 * AJAX: capture a PayPal order (after onApprove in JS).
 */
function shoplite_paypal_capture_order() {

    // 1) Check nonce (JS envía "nonce")
    $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
    if ( ! $nonce || ! wp_verify_nonce( $nonce, SHOPLITE_NONCE_PAYPAL ) ) {
        wp_send_json_error([ 'message' => esc_html__( 'Invalid PayPal nonce.', 'shoplite' ) ], 400);
    }

    // 2) Read order_id sent from JS (PayPal order id)
    $paypal_order_id = isset($_POST['order_id']) ? sanitize_text_field(wp_unslash($_POST['order_id'])) : '';
    if ( empty( $paypal_order_id ) ) {
        wp_send_json_error([ 'message' => esc_html__( 'Missing PayPal order_id.', 'shoplite' ) ], 400);
    }

    // 3) Get PayPal token
    $token = shoplite_paypal_get_token();
    if ( ! $token || is_wp_error( $token ) ) {
        wp_send_json_error([ 'message' => esc_html__( 'Could not obtain a PayPal token to capture the payment.', 'shoplite' ) ], 500);
    }

    $api_base = shoplite_paypal_api_base();
    $url      = $api_base . '/v2/checkout/orders/' . rawurlencode( $paypal_order_id ) . '/capture';

    // 4) Call PayPal API to capture
    $args = [
        'method'  => 'POST',
        'headers' => [
            'Authorization' => 'Bearer ' . $token,
            'Content-Type'  => 'application/json',
            'Prefer'        => 'return=representation',
        ],
        'body'    => '{}',
        'timeout' => 30,
    ];

    $resp = wp_remote_post( $url, $args );

    if ( is_wp_error( $resp ) ) {
        error_log('PAYPAL_CAPTURE_FAIL (HTTP) -------------------');
        error_log('ORDER ID: ' . $paypal_order_id);
        error_log('ERROR: ' . $resp->get_error_message());

        wp_send_json_error([ 'message' => esc_html__( 'Could not capture the PayPal payment (connection error).', 'shoplite' ) ], 502);
    }

    $code = wp_remote_retrieve_response_code( $resp );
    $body = wp_remote_retrieve_body( $resp );
    $data = json_decode( $body, true );

    if ( $code < 200 || $code >= 300 ) {
        error_log('PAYPAL_CAPTURE_FAIL (API) -------------------');
        error_log('ORDER ID: ' . $paypal_order_id);
        error_log('HTTP CODE: ' . $code);
        error_log('BODY: ' . $body);

        $msg = esc_html__( 'Could not capture the PayPal payment (invalid response).', 'shoplite' );
        if ( is_array($data) && ! empty( $data['message'] ) ) {
            $msg .= ' ' . $data['message'];
        }

        wp_send_json_error([ 'message' => $msg, 'pp' => $data ?: $body ], 500);
    }

    // 5) Verify status (normally COMPLETED)
    $status = isset( $data['status'] ) ? (string) $data['status'] : '';
    if ( $status !== 'COMPLETED' ) {
        error_log('PAYPAL_CAPTURE_FAIL (STATUS) -------------------');
        error_log('ORDER ID: ' . $paypal_order_id);
        error_log('STATUS: ' . $status);
        error_log('BODY: ' . $body);

        wp_send_json_error([
            'message' => sprintf(
                esc_html__( 'The PayPal payment is not COMPLETED (status: %s).', 'shoplite' ),
                $status
            ),
            'pp' => $data,
        ], 409);
    }

    // 6) Create internal Shoplite order
    $created_order_id = 0;

    if ( class_exists( 'Shoplite_Orders' ) && function_exists('shoplite_build_order_data_from_session') ) {
        $order_data = shoplite_build_order_data_from_session( 'paypal' );

        if ( is_array( $order_data ) ) {
            // Inyecta el PayPal order id para rastreo interno
            $order_data['paypal_order_id'] = $paypal_order_id;

            // Opcional: intenta incluir info del payer si PayPal la devuelve
            if ( ! empty( $data['payer']['email_address'] ) ) {
                $order_data['paypal_payer_email'] = sanitize_email( $data['payer']['email_address'] );
            }

            $maybe_id = Shoplite_Orders::create_from_checkout( $order_data );
            if ( is_numeric( $maybe_id ) ) {
                $created_order_id = (int) $maybe_id;
            }
        }
    }

    // Fallback: encontrar el pedido más reciente por email + método paypal
    if ( $created_order_id <= 0 ) {

        // Email: primero desde sesión, si no, desde PayPal payer (sandbox: suele ser @example.com)
        $email = '';

        if ( function_exists('shoplite_maybe_session_start') ) {
            shoplite_maybe_session_start();
        } else {
            if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
                @session_start();
            }
        }

        if ( ! empty( $_SESSION['shoplite_customer_email'] ) ) {
            $email = sanitize_email( $_SESSION['shoplite_customer_email'] );
        } elseif ( ! empty( $data['payer']['email_address'] ) ) {
            $email = sanitize_email( $data['payer']['email_address'] );
        }

        if ( $email ) {
            $q = new WP_Query([
                'post_type'      => 'shoplite_pedido',
                'post_status'    => 'publish',
                'posts_per_page' => 1,
                'orderby'        => 'date',
                'order'          => 'DESC',
                'meta_query'     => [
                    [
                        'key'   => '_shoplite_customer_email',
                        'value' => $email,
                    ],
                    [
                        'key'   => '_shoplite_payment_method',
                        'value' => 'paypal',
                    ],
                ],
                'fields' => 'ids',
            ]);
            if ( ! empty( $q->posts[0] ) ) {
                $created_order_id = (int) $q->posts[0];
            }
        }
    }

    // 7) Guardar trazabilidad + disparar hook de "paid"
    if ( $created_order_id > 0 ) {

        update_post_meta( $created_order_id, '_shoplite_paypal_order_id', $paypal_order_id );

        // (Opcional) guardar datos útiles para debug / backoffice
        if ( ! empty( $data['payer']['email_address'] ) ) {
            update_post_meta( $created_order_id, '_shoplite_paypal_payer_email', sanitize_email( $data['payer']['email_address'] ) );
        }
        if ( ! empty( $data['purchase_units'][0]['payments']['captures'][0]['id'] ) ) {
            update_post_meta( $created_order_id, '_shoplite_paypal_capture_id', sanitize_text_field( $data['purchase_units'][0]['payments']['captures'][0]['id'] ) );
        }

        /**
         * IMPORTANTE para Sandbox:
         * Si tu "digital-delivery.php" envía el email al email del cliente guardado en el pedido,
         * en sandbox ese email suele ser @example.com y NO recibirá correos reales.
         *
         * Para poder probar emails en Sandbox sin tocar tu digital-delivery.php, guardamos
         * un meta extra con el email real de pruebas (admin). Así tu delivery puede usarlo
         * si lo deseas (fallback).
         */
        if ( function_exists('shoplite_get_modo_paypal') && shoplite_get_modo_paypal() === 'sandbox' ) {
            update_post_meta( $created_order_id, '_shoplite_test_email', sanitize_email( get_option('admin_email') ) );
        }

        do_action( 'shoplite_order_paid', $created_order_id );

    } else {
        error_log('PAYPAL_CAPTURE_OK_BUT_NO_INTERNAL_ORDER_ID -------------------');
        error_log('PAYPAL ORDER ID: ' . $paypal_order_id);
        error_log('BODY: ' . $body);
    }

    // 8) Clear cart/coupon after successful payment
    if ( function_exists( 'shoplite_maybe_session_start' ) ) {
        shoplite_maybe_session_start();
    } else {
        if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
            @session_start();
        }
    }

    unset(
        $_SESSION['shoplite_cart'],
        $_SESSION['carrito'],
        $_SESSION['shoplite_coupon_code']
    );

    wp_send_json_success([
        'ok'              => true,
        'status'          => $status,
        'paypal_order_id' => $paypal_order_id,
        'id'              => isset($data['id']) ? $data['id'] : '',
        'internal_order'  => $created_order_id,
    ]);
}




// Procesa "Actualizar cantidades" (POST) con nonce y redirección limpia
add_action('init', function () {
  if (empty($_POST['tienda_basica_update_qty'])) return;

  // ✅ Verificación del nonce del formulario (ya lo imprimes con wp_nonce_field('tienda_basica_qty'))
  if (
    empty($_POST['_wpnonce']) ||
    ! wp_verify_nonce($_POST['_wpnonce'], SHOPLITE_NONCE_CART_ACTION)
  ) {
    return; // o wp_die('Nonce inválido');
  }

  if (!session_id()) { @session_start(); }

  $qtys = isset($_POST['qty']) ? (array) $_POST['qty'] : [];
  foreach ($qtys as $id => $qty) {
    $id = sanitize_text_field($id);
    $q  = max(1, (int) $qty);
    if (isset($_SESSION['carrito'][$id])) {
      $_SESSION['carrito'][$id]['cantidad'] = $q;
    }
  }

  // 🔁 Redirige para evitar reenvío del formulario y limpia la URL
  $ref  = wp_get_referer();
  $dest = $ref ? $ref : home_url('/');
  wp_safe_redirect( remove_query_arg(['add_to_cart','eliminar_item','vaciar_carrito','tienda_basica_update_qty','_wpnonce'], $dest) );
  exit;
});



// === Página "Asistente" mínima (fallback, solo si no existe el real) ===

// 1) Fallback render (ok)
if ( ! function_exists('shoplite_render_setup_page') ) {
    function shoplite_render_setup_page() {
        echo '<div class="wrap"><h1>' . esc_html__( 'Shoplite Setup Assistant', 'shoplite' ) . '</h1>';
        echo '<p>' . esc_html__( 'Welcome. A step-by-step setup wizard will go here.', 'shoplite' ) . '</p></div>';
    }
}

// 2) Menú del asistente (SOLO si el setup wizard real NO está disponible)
add_action('admin_menu', function () {

    // Si el asistente real ya existe, no añadimos el fallback (evita duplicados)
    if ( class_exists('Shoplite_Setup_Wizard') || class_exists('Shoplite_SetupWizard') ) {
        return;
    }

    if ( ! defined('TS_MENU_SLUG') ) {
        return;
    }

    add_submenu_page(
        TS_MENU_SLUG,
        __('Shoplite Setup Assistant', 'shoplite'),
        __('Setup Assistant', 'shoplite'),
        'manage_options',
        'shoplite-setup',
        'shoplite_render_setup_page'
    );
}, 99);





/* Panel principal (estilos de botones) */
function shoplite_admin_panel(){ ?>
    <div class="wrap">
        <h1><?php echo esc_html__('Basic Shop', 'shoplite'); ?></h1>
        <p>
          <?php echo esc_html__('Shortcodes:', 'shoplite'); ?>
          <code>[tienda_catalogo]</code>, <code>[tienda_carrito]</code> <?php echo esc_html__('or', 'tienda-basica'); ?> <code>[tienda_basica]</code>.
        </p>
        <p>
          <?php echo esc_html__('Buttons:', 'shoplite'); ?>
          <?php echo esc_html__('class', 'shoplite'); ?> <code>.boton-tienda</code> + <?php echo esc_html__('saved style (default', 'tienda-basica'); ?> <code>clasico</code>).
        </p>

        <form method="post" action="">
            <?php
            if (isset($_POST['estilo']) && isset($_POST['tienda_basica_nonce']) && wp_verify_nonce($_POST['tienda_basica_nonce'],'guardar_estilo')) {
                update_option('estilos_boton', sanitize_text_field($_POST['estilo']));
                echo "<div class='updated'><p>✅ " . esc_html__('Style saved.', 'shoplite') . "</p></div>";
            }
            $estilo_guardado = get_option('estilos_boton','clasico');
            $estilos = array('clasico','moderno','minimalista','elegante','retro','oscuro','neon');
            ?>
            <?php wp_nonce_field('guardar_estilo','tienda_basica_nonce'); ?>
            <label for="estilo"><?php echo esc_html__('Button style:', 'shoplite'); ?></label>
            <select name="estilo" id="estilo">
                <?php foreach ($estilos as $e): $sel = selected($estilo_guardado,$e,false); echo "<option value='".esc_attr($e)."' $sel>".esc_html($e)."</option>"; endforeach; ?>
            </select>
            <input type="submit" class="button button-primary" value="<?php echo esc_attr__('Save style','shoplite'); ?>">
            <hr>
            <h2><?php echo esc_html__('Preview','tienda-basica'); ?></h2>
            <?php foreach ($estilos as $e) echo "<div style='margin:6px 0;'><button class='boton-tienda ".esc_attr($e)."'>".esc_html__('Button','shoplite')." ".esc_html($e)."</button> <span style='opacity:.7'>(.".esc_html($e).")</span></div>"; ?>
        </form>
    </div>
<?php }

/* Subpágina: Pagos */
function shoplite_admin_pagos(){ ?>
    <div class="wrap">
        <h1><?php echo esc_html__('Payments (Stripe + PayPal)','shoplite'); ?></h1>
        <form method="post" action="options.php">
            <?php settings_fields('shoplite_pagos_group'); do_settings_sections('shoplite-pagos'); submit_button(); ?>
        </form>
    </div>
<?php }

add_action('admin_init', function () {

  $group = 'shoplite_pagos_group';
  $ts_menu_slug = 'shoplite-pagos';

  /* ---------- REGISTRO DE OPCIONES (GENERAL) ---------- */
  register_setting($group, 'tb_currency', [
    'type' => 'string', 'sanitize_callback' => 'sanitize_text_field', 'default' => 'EUR',
  ]);
  register_setting($group, 'tb_success_url', [
    'type' => 'string', 'sanitize_callback' => 'esc_url_raw', 'default' => '',
  ]);
  register_setting($group, 'tb_cancel_url', [
    'type' => 'string', 'sanitize_callback' => 'esc_url_raw', 'default' => '',
  ]);
  register_setting($group, 'tb_paypal_show_card', [
    'type' => 'boolean', 'sanitize_callback' => 'rest_sanitize_boolean', 'default' => false,
  ]);
  register_setting($group, 'tb_show_sandbox_badge', [
    'type' => 'boolean', 'sanitize_callback' => 'rest_sanitize_boolean', 'default' => false,
  ]);

  add_settings_section(
    'shoplite_sec_general', esc_html__('General', 'shoplite'),
    function () { echo '<p>'.esc_html__('Currency, URLs and interface preferences.','shoplite').'</p>'; },
    $ts_menu_slug
  );

  add_settings_field('tb_currency', esc_html__('Currency', 'shoplite'), function () {
    $cur = get_option('tb_currency', 'EUR');
    $ops = ['EUR'=>'EUR','USD'=>'USD','MXN'=>'MXN','ARS'=>'ARS','COP'=>'COP','CLP'=>'CLP','PEN'=>'PEN','BRL'=>'BRL'];
    echo '<select name="tb_currency">';
    foreach ($ops as $k => $v) {
      printf('<option value="%s"%s>%s</option>', esc_attr($k), selected($cur, $k, false), esc_html($v));
    }
    echo '</select>';
  }, $ts_menu_slug, 'shoplite_sec_general');

  add_settings_field('tb_success_url', esc_html__('Success URL', 'shoplite'), function () {
    printf("<input type='url' name='tb_success_url' value='%s' size='60' placeholder='%s'>",
      esc_attr(get_option('tb_success_url','')),
      esc_attr(home_url('/finalized'))
    );
  }, $ts_menu_slug, 'shoplite_sec_general');

  add_settings_field('tb_cancel_url', esc_html__('Cancel URL', 'shoplite'), function () {
    printf("<input type='url' name='tb_cancel_url' value='%s' size='60' placeholder='%s'>",
      esc_attr(get_option('tb_cancel_url','')),
      esc_attr(home_url('/cart'))
    );
  }, $ts_menu_slug, 'shoplite_sec_general');

  add_settings_field('tb_paypal_show_card', esc_html__('Show Card button (PayPal)', 'shoplite'), function () {
    $v = (bool) get_option('tb_paypal_show_card', false);
    printf(
      "<label><input type='checkbox' name='tb_paypal_show_card' value='1' %s> %s</label>",
      checked($v, true, false),
      esc_html__('Also show the PayPal “Debit or Credit Card” button','shoplite')
    );
  }, $ts_menu_slug, 'shoplite_sec_general');

  add_settings_field('tb_show_sandbox_badge', esc_html__('Show SANDBOX badge', 'shoplite'), function () {
    $v = (bool) get_option('tb_show_sandbox_badge', false);
    printf(
      "<label><input type='checkbox' name='tb_show_sandbox_badge' value='1' %s> %s</label>",
      checked($v, true, false),
      esc_html__('Show a SANDBOX notice on the frontend when the mode is not Live','shoplite')
    );
  }, $ts_menu_slug, 'shoplite_sec_general');

  /* ---------- STRIPE ---------- */
  register_setting($group,'stripe_modo',         ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'test']);
  register_setting($group,'stripe_test_public',  ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);
  register_setting($group,'stripe_test_secret',  ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);
  register_setting($group,'stripe_live_public',  ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);
  register_setting($group,'stripe_live_secret',  ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);

  add_settings_section(
    'shoplite_sec_stripe', esc_html__('Stripe','shoplite'),
    function () { echo '<p>'.esc_html__('Configure your keys and mode. Stripe.js loads only when a public key is set.','shoplite').'</p>'; },
    $ts_menu_slug
  );

  add_settings_field('stripe_modo', esc_html__('Mode','shoplite'), function () {
    $m = get_option('stripe_modo','test'); ?>
    <select name="stripe_modo">
      <option value="test" <?php selected($m,'test');?>><?php echo esc_html__('Test','shoplite'); ?></option>
      <option value="live" <?php selected($m,'live');?>><?php echo esc_html__('Live','shoplite'); ?></option>
    </select>
  <?php }, $ts_menu_slug, 'shoplite_sec_stripe');

  add_settings_field('stripe_test_public', esc_html__('Public key (Test)','shoplite'), function () {
    printf("<input type='text' name='stripe_test_public' value='%s' size='50'>", esc_attr(get_option('stripe_test_public','')));
  }, $ts_menu_slug, 'shoplite_sec_stripe');

  add_settings_field('stripe_test_secret', esc_html__('Secret key (Test)','shoplite'), function () {
    printf("<input type='text' name='stripe_test_secret' value='%s' size='50'>", esc_attr(get_option('stripe_test_secret','')));
  }, $ts_menu_slug, 'shoplite_sec_stripe');

  add_settings_field('stripe_live_public', esc_html__('Public key (Live)','shoplite'), function () {
    printf("<input type='text' name='stripe_live_public' value='%s' size='50'>", esc_attr(get_option('stripe_live_public','')));
  }, $ts_menu_slug, 'shoplite_sec_stripe');

  add_settings_field('stripe_live_secret', esc_html__('Secret key (Live)','shoplite'), function () {
    printf("<input type='text' name='stripe_live_secret' value='%s' size='50'>", esc_attr(get_option('stripe_live_secret','')));
  }, $ts_menu_slug, 'shoplite_sec_stripe');

  /* ---------- PAYPAL ---------- */
  register_setting($group,'paypal_modo',           ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'sandbox']);
  register_setting($group,'paypal_sandbox_client', ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);
  register_setting($group,'paypal_sandbox_secret', ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);
  register_setting($group,'paypal_live_client',    ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);
  register_setting($group,'paypal_live_secret',    ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);

  add_settings_section(
    'shoplite_sec_paypal', esc_html__('PayPal','shoplite'),
    function () { echo '<p>'.esc_html__('Configure PayPal. The SDK loads only when a Client ID is set.','shoplite').'</p>'; },
    $ts_menu_slug
  );

  add_settings_field('paypal_modo', esc_html__('Mode','shoplite'), function () use ($ts_menu_slug) {
    $m = get_option('paypal_modo','sandbox'); ?>
    <select name="paypal_modo">
      <option value="sandbox" <?php selected($m,'sandbox');?>><?php echo esc_html__('Sandbox','shoplite'); ?></option>
      <option value="live"    <?php selected($m,'live');?>><?php echo esc_html__('Live','shoplite'); ?></option>
    </select>
  <?php }, $ts_menu_slug, 'shoplite_sec_paypal');

  add_settings_field('paypal_sandbox_client', esc_html__('Client ID (Sandbox)','shoplite'), function () {
    printf("<input type='text' name='paypal_sandbox_client' value='%s' size='60'>", esc_attr(get_option('paypal_sandbox_client','')));
  }, $ts_menu_slug, 'shoplite_sec_paypal');

  add_settings_field('paypal_sandbox_secret', esc_html__('Secret (Sandbox)','shoplite'), function () {
    printf("<input type='text' name='paypal_sandbox_secret' value='%s' size='60'>", esc_attr(get_option('paypal_sandbox_secret','')));
  }, $ts_menu_slug, 'shoplite_sec_paypal');

  add_settings_field('paypal_live_client', esc_html__('Client ID (Live)','shoplite'), function () {
    printf("<input type='text' name='paypal_live_client' value='%s' size='60'>", esc_attr(get_option('paypal_live_client','')));
  }, $ts_menu_slug, 'shoplite_sec_paypal');

  add_settings_field('paypal_live_secret', esc_html__('Secret (Live)','shoplite'), function () {
    printf("<input type='text' name='paypal_live_secret' value='%s' size='60'>", esc_attr(get_option('paypal_live_secret','')));
  }, $ts_menu_slug, 'shoplite_sec_paypal');

});


/* ===== Text & Style: options ===== */

/* 1) Register options and fields (Text & Style) */
add_action('admin_init', 'ts_register_appearance_settings');
function ts_register_appearance_settings() {

  /* Options */
  register_setting('shoplite_appearance_group','ts_btn_text_buy',      ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>__('Buy','shoplite')]);
  register_setting('shoplite_appearance_group','ts_btn_text_cart',     ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>__('Add to cart','shoplite')]);
  register_setting('shoplite_appearance_group','ts_btn_text_checkout', ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>__('Checkout','shoplite')]);
  register_setting('shoplite_appearance_group','ts_btn_text_empty',    ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>__('Empty cart','shoplite')]);
  register_setting('shoplite_appearance_group','ts_btn_text_update',   ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>__('Update quantities','shoplite')]);
  register_setting('shoplite_appearance_group','ts_btn_text_remove',   ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>__('Remove','shoplite')]);
  register_setting('shoplite_appearance_group','ts_btn_text_small',    ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);
  register_setting('shoplite_appearance_group','ts_btn_color',         ['type'=>'string','sanitize_callback'=>'sanitize_hex_color','default'=>'#2ea44f']);
  register_setting('shoplite_appearance_group','ts_btn_radius',        ['type'=>'integer','sanitize_callback'=>'absint','default'=>6]);
  register_setting('shoplite_appearance_group','ts_btn_font_size',     ['type'=>'integer','sanitize_callback'=>'absint','default'=>16]);

  // Compat with old option (no field shown, just keeps value if it existed)
  register_setting('shoplite_appearance_group','ts_btn_text',          ['type'=>'string','sanitize_callback'=>'sanitize_text_field','default'=>'']);

  /* Section */
  add_settings_section(
    'shoplite_appearance_section',
    __('Purchase buttons','shoplite'),
    '__return_false',
    'shoplite-appearance' // submenu slug: Text & Style
  );

  /* Fields */
  add_settings_field('ts_btn_text_buy', __('“Buy” button text','shoplite'), function () {
    $legacy = get_option('ts_btn_text','');
    $v = get_option('ts_btn_text_buy', $legacy !== '' ? $legacy : __('Buy','shoplite'));
    echo '<input type="text" name="ts_btn_text_buy" value="'.esc_attr($v).'" class="regular-text" />';
  }, 'shoplite-appearance','shoplite_appearance_section');

  add_settings_field('ts_btn_text_cart', __('“Add to cart” text','shoplite'), function () {
    $v = get_option('ts_btn_text_cart', __('Add to cart','shoplite'));
    echo '<input type="text" name="ts_btn_text_cart" value="'.esc_attr($v).'" class="regular-text" />';
  }, 'shoplite-appearance','shoplite_appearance_section');

  add_settings_field('ts_btn_text_remove', __('“Remove” text','shoplite'), function () {
    $v = get_option('ts_btn_text_remove', __('Remove','shoplite'));
    echo '<input type="text" name="ts_btn_text_remove" value="'.esc_attr($v).'" class="regular-text" />';
  }, 'shoplite-appearance','shoplite_appearance_section');

  add_settings_field('ts_btn_text_checkout', __('“Checkout” button text','shoplite'), function () {
    $v = get_option('ts_btn_text_checkout', __('Checkout','shoplite'));
    echo '<input type="text" name="ts_btn_text_checkout" value="'.esc_attr($v).'" class="regular-text" />';
  }, 'shoplite-appearance','shoplite_appearance_section');

  add_settings_field('ts_btn_text_empty', __('“Empty cart” text','shoplite'), function () {
    $v = get_option('ts_btn_text_empty', __('Empty cart','shoplite'));
    echo '<input type="text" name="ts_btn_text_empty" value="'.esc_attr($v).'" class="regular-text" />';
  }, 'shoplite-appearance','shoplite_appearance_section');

  add_settings_field('ts_btn_text_update', __('“Update quantities” text','shoplite'), function () {
    $v = get_option('ts_btn_text_update', __('Update quantities','shoplite'));
    echo '<input type="text" name="ts_btn_text_update" value="'.esc_attr($v).'" class="regular-text" />';
  }, 'shoplite-appearance','shoplite_appearance_section');

  add_settings_field('ts_btn_text_small', __('Secondary text','shoplite'), function () {
    $v = get_option('ts_btn_text_small','');
    echo '<input type="text" name="ts_btn_text_small" value="'.esc_attr($v).'" class="regular-text" />';
    echo '<p class="description">'.esc_html__('Example: “Secure payment with Stripe”.','shoplite').'</p>';
  }, 'shoplite-appearance','shoplite_appearance_section');

  add_settings_field('ts_btn_color', __('Button color','shoplite'), function () {
    $v = get_option('ts_btn_color','#2ea44f');
    echo '<input type="color" name="ts_btn_color" value="'.esc_attr($v).'" /> <code>'.esc_html($v).'</code>';
  }, 'shoplite-appearance','shoplite_appearance_section');

  add_settings_field('ts_btn_radius', __('Border radius (px)','shoplite'), function () {
    $v = (int) get_option('ts_btn_radius',6);
    echo '<input type="number" min="0" max="40" step="1" name="ts_btn_radius" value="'.esc_attr($v).'" />';
  }, 'shoplite-appearance','shoplite_appearance_section');

  add_settings_field('ts_btn_font_size', __('Font size (px)','shoplite'), function () {
    $v = (int) get_option('ts_btn_font_size',16);
    echo '<input type="number" min="10" max="32" step="1" name="ts_btn_font_size" value="'.esc_attr($v).'" />';
  }, 'shoplite-appearance','shoplite_appearance_section');
}


/* 2) Page render (submenu callback) */
function shoplite_admin_appearance() {
  if (!current_user_can('manage_options')) return;

  echo '<div class="wrap"><h1>' . esc_html__('Text & Style', 'shoplite') . '</h1>';
  echo '<form method="post" action="options.php">';
  settings_fields('shoplite_appearance_group');   // group defined above
  do_settings_sections('shoplite-appearance');    // submenu slug
  submit_button( __('Save changes', 'shoplite') );
  echo '</form></div>';
}






  // Extract product ID: accepts "123", "tbp_123" or "tbp_tbp_123"
if ( ! isset($_REQUEST['add_to_cart']) ) {
  if (defined('WP_DEBUG') && WP_DEBUG && function_exists('error_log')) {
    error_log('[shoplite] ADD missing add_to_cart; abort');
  }
  return;
}

$raw = wp_unslash( $_REQUEST['add_to_cart'] );
if (is_array($raw)) { $raw = reset($raw); }            // in case it arrives as an array
$raw = sanitize_text_field( $raw );

if (function_exists('error_log')) error_log('[shoplite] ADD raw='.var_export($raw, true));

if (is_string($raw) && preg_match('/^(?:tbp_)+(\d+)/', $raw, $m)) {
  $pid = (int) $m[1];  // grab the number even if there are multiple "tbp_"
} else {
  $pid = (int) $raw;
}

if (function_exists('error_log')) error_log('[shoplite] ADD pid='.$pid);
if ($pid <= 0) return;


    // Validate product
    $pt = get_post_type($pid);
    $st = get_post_status($pid);
    if (function_exists('error_log')) error_log('[shoplite] ADD post_type='.$pt.' status='.$st);
    if ($pt !== 'shoplite_producto' || $st !== 'publish') return;

    // Product data
    $price  = (float) get_post_meta($pid, '_tb_price', true);
    $fisico = (get_post_meta($pid, '_tb_is_physical', true) === '1') ? 1 : 0;
    $envio  = (float) get_post_meta($pid, '_tb_shipping_cost', true);
    $nombre = get_the_title($pid);
    $qty    = max(1, (int)($_REQUEST['qty'] ?? 1));

    if (!session_id()) session_start();
    $cart = $_SESSION['shoplite_cart'] ?? ($_SESSION['carrito'] ?? []);

    // Stable line key
    $lineKey = 'tbp_'.$pid;

    if (isset($cart[$lineKey])) {
      $cart[$lineKey]['cantidad'] = max(1, (int)($cart[$lineKey]['cantidad'] ?? 1)) + $qty;
      $cart[$lineKey]['qty']      = $cart[$lineKey]['cantidad'];
    } else {
      $cart[$lineKey] = [
        'id'       => $pid,
        'nombre'   => $nombre ?: ('Product '.$pid),
        'precio'   => $price,
        'cantidad' => $qty,
        'qty'      => $qty,
        'fisico'   => $fisico,
        'envio'    => $envio,
      ];
    }

    // Persist (canonical + legacy mirror)
    $_SESSION['shoplite_cart'] = $cart;
    $_SESSION['carrito']       = $cart;

    // Redirect back to the SAME clean URL (no parameters)
    $back = wp_get_referer();
    if (!$back && !empty($_SERVER['HTTP_HOST']) && !empty($_SERVER['REQUEST_URI'])) {
      $scheme = is_ssl() ? 'https://' : 'http://';
      $back   = $scheme.$_SERVER['HTTP_HOST'].$_SERVER['REQUEST_URI'];
    }
    $back = remove_query_arg(['add_to_cart','qty','_wpnonce','_wp_http_referer'], $back ?: home_url('/'));
    if (function_exists('error_log')) error_log('[shoplite] ADD redirect => '.$back);

    // Redirect back (WP-safe when available; fallback to header)
$redirect = $back ?: home_url('/');

// Ensure no cart params remain
$redirect = remove_query_arg(['add_to_cart','qty','_wpnonce','_wp_http_referer'], $redirect);

if ( function_exists('wp_safe_redirect') ) {
  wp_safe_redirect( $redirect );
  exit;
}

// Fallback: sanitize and redirect
$safe_redirect = preg_replace('|[^a-z0-9-~+_.?#=!&;,/:%@$\|*\'()\\x80-\\xff]|i', '', $redirect);
header("Location: " . $safe_redirect);
exit;



/* ============================================================
 * Cart POST handler (remove / update / empty)
 * ============================================================ */

if ( ! function_exists('shoplite_cart_process_post_once') ) {
  function shoplite_cart_process_post_once() {
    static $done = false;
    if ($done) return; // prevent double execution
    $done = true;

    // Only process POST requests
    if ( strtoupper($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST' ) return;

    // LOG: did the POST reach here?
    if ( function_exists('error_log') ) {
      error_log('[shoplite] CART POST hit? method='.(isset($_SERVER['REQUEST_METHOD'])?$_SERVER['REQUEST_METHOD']:'(none)').' keys='.implode(',', array_keys($_POST ?? [])));
    }

    // ✅ Unified nonce verification for the cart form (accepts _wpnonce and nonce)
    $nonce = '';
    if ( isset($_POST['_wpnonce']) ) {
      $nonce = sanitize_text_field( wp_unslash($_POST['_wpnonce']) );
    } elseif ( isset($_POST['nonce']) ) {
      $nonce = sanitize_text_field( wp_unslash($_POST['nonce']) );
    }

    if ( ! $nonce || ! wp_verify_nonce( $nonce, SHOPLITE_NONCE_CART_ACTION ) ) {
      if ( defined('WP_DEBUG') && WP_DEBUG && function_exists('error_log') ) {
        error_log('[shoplite] CART POST nonce FAIL value=' . substr((string)$nonce, 0, 10));
      }
      return;
    }

    // Asegura sesión y estructura
    // Sesión (controlada por helper unificado)
    if ( function_exists( 'shoplite_maybe_session_start' ) ) {
        shoplite_maybe_session_start();
    } else {
        if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
            @session_start();
        }
    }

    if ( empty($_SESSION['shoplite_cart']) || !is_array($_SESSION['shoplite_cart']) ) {
      $_SESSION['shoplite_cart'] = [];
    }
    $cart =& $_SESSION['shoplite_cart']; // reference

    // 1) Empty cart
    if ( isset($_POST['empty']) ) {
      if (function_exists('error_log')) error_log('[shoplite] CART action: EMPTY');
      $cart = [];
      $_SESSION['carrito'] = []; // legacy mirror
    }

    // 2) Remove an item by real key
    if ( isset($_POST['remove']) ) {
      $rem = sanitize_key( wp_unslash($_POST['remove']) );
      if (function_exists('error_log')) error_log('[shoplite] CART action: REMOVE '.$rem);
      if ( isset($cart[$rem]) ) {
        unset($cart[$rem]);
      }
      $_SESSION['carrito'] = $cart;
    }

    // 3) Update quantities
    if ( isset($_POST['update']) && isset($_POST['qty']) ) {
      // uses your global sanitization helper if available
      if ( function_exists('shoplite_clean_qty_array') ) {
        $qtys = shoplite_clean_qty_array( $_POST['qty'] );
      } else {
        // fallback sanitize
        $qtys = [];
        foreach ( (array) $_POST['qty'] as $k => $q ) {
          $k = sanitize_text_field( wp_unslash($k) );
          $qtys[$k] = max(1, absint($q));
        }
      }

      foreach ($qtys as $k => $q) {
        if ( isset($cart[$k]) ) {
          $cart[$k]['cantidad'] = $q;
        }
      }
      $_SESSION['carrito'] = $cart;
      if (function_exists('error_log')) error_log('[shoplite] CART action: UPDATE');
    }

    // PRG redirect: back to same page and clean params
    $redirect = wp_get_referer();
    if ( ! $redirect && !empty($_SERVER['HTTP_HOST']) && !empty($_SERVER['REQUEST_URI']) ) {
      $scheme   = is_ssl() ? 'https://' : 'http://';
      $redirect = $scheme . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
    }

    $redirect = remove_query_arg(
      ['add_to_cart','eliminar_item','vaciar_carrito','tienda_basica_update_qty','_wpnonce','_wp_http_referer'],
      $redirect
    );

    if ( function_exists('wp_safe_redirect') ) {
      wp_safe_redirect( $redirect ?: home_url('/') );
    } else {
      header('Location: '. ( $redirect ?: '/' ), true, 302);
    }
    exit;
  }
}


// Hook: ensure it exists only once and early enough
add_action('template_redirect', 'shoplite_cart_process_post_once', 1);


// 1) Use plugin template for the CPT (let WP load header/footer/scripts normally)
add_filter('single_template', function ($template) {
  if (is_singular('shoplite_producto')) {
    $t = plugin_dir_path(__FILE__) . 'templates/single-shoplite_producto.php';
    if (file_exists($t)) return $t;  // WP continues normal flow (wp_head/wp_footer)
  }
  return $template;
});




// ... TODO YOUR CURRENT SHOPLITE.PHP CODE ...

/* ============================================================
 * SHORTCODES & MAIN FEATURES
 * ============================================================ */

// ... your existing shortcodes like shoplite_shortcode_carrito() ...

/* ============================================================
 * AJAX Handlers for the cart - NEW (ADD AT THE END)
 * ============================================================ */

// Add to cart from single product page
add_action('wp_ajax_shoplite_add_to_cart',        'shoplite_ajax_add_to_cart');
add_action('wp_ajax_nopriv_shoplite_add_to_cart', 'shoplite_ajax_add_to_cart');

function shoplite_ajax_add_to_cart() {

  // Nonce verification (field 'nonce' sent by JS)
  if ( ! check_ajax_referer( SHOPLITE_NONCE_CART_ACTION, 'nonce', false ) ) {
    wp_send_json_error( ['message' => esc_html__( 'Invalid nonce.', 'shoplite' )], 400 );
  }

  // Inputs
  $product_id = isset($_POST['product_id']) ? absint($_POST['product_id']) : 0;
  $quantity   = isset($_POST['quantity'])   ? max(1, absint($_POST['quantity'])) : 1;

  if ( ! $product_id ) {
    wp_send_json_error( ['message' => esc_html__( 'Invalid product.', 'shoplite' )], 400 );
  }

  // Session (unified, no warnings)
  if ( function_exists( 'shoplite_maybe_session_start' ) ) {
    shoplite_maybe_session_start();
  } else {
    if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
      @session_start();
    }
  }

  // Cart logic (stable key: tbp_ID)
  $cart = isset($_SESSION['shoplite_cart']) && is_array($_SESSION['shoplite_cart'])
    ? $_SESSION['shoplite_cart']
    : [];

  $key = 'tbp_' . $product_id;

  if ( ! isset($cart[$key]) || ! is_array($cart[$key]) ) {
    $cart[$key] = [
      'id'       => $product_id,
      'nombre'   => get_the_title($product_id),
      'precio'   => (float) get_post_meta($product_id, '_tb_price', true),
      'cantidad' => 0,
      'fisico'   => get_post_meta($product_id, '_tb_is_physical', true) === '1' ? 1 : 0,
      'envio'    => (float) get_post_meta($product_id, '_tb_shipping_cost', true),
    ];
  }

  $cart[$key]['cantidad'] = max(0, (int)($cart[$key]['cantidad'] ?? 0)) + $quantity;
  $cart[$key]['qty']      = $cart[$key]['cantidad']; // legacy compat if your JS reads qty

  $_SESSION['shoplite_cart'] = $cart;
  $_SESSION['carrito']       = $cart;

  wp_send_json_success([
    'message'    => esc_html__( 'Product added to cart.', 'shoplite' ),
    'cart_count' => count($cart),
    // 'redirect_url' => get_permalink( get_option('shoplite_cart_page') ), // optional
  ]);
}

// Cart actions (update/remove/empty)
add_action('wp_ajax_shoplite_cart_action',        'shoplite_ajax_cart_action');
add_action('wp_ajax_nopriv_shoplite_cart_action', 'shoplite_ajax_cart_action');

function shoplite_ajax_cart_action() {

  // Nonce verification (only once)
  if ( ! check_ajax_referer( SHOPLITE_NONCE_CART_ACTION, 'nonce', false ) ) {
    wp_send_json_error( ['message' => esc_html__( 'Invalid nonce.', 'shoplite' )], 400 );
  }

  // Session (unified, no warnings)
  if ( function_exists( 'shoplite_maybe_session_start' ) ) {
    shoplite_maybe_session_start();
  } else {
    if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
      @session_start();
    }
  }

  $cart = isset($_SESSION['shoplite_cart']) && is_array($_SESSION['shoplite_cart'])
    ? $_SESSION['shoplite_cart']
    : [];

  $what = isset($_POST['cart_action']) ? sanitize_key( wp_unslash($_POST['cart_action']) ) : '';

  switch ($what) {
    case 'update':
      $raw = isset($_POST['quantities']) ? wp_unslash($_POST['quantities']) : '';
      $quantities = json_decode($raw, true);
      if ( ! is_array($quantities) ) {
        wp_send_json_error(['message' => esc_html__( 'quantities must be valid JSON.', 'shoplite' )], 400);
      }
      foreach ($quantities as $line_key => $q) {
        $lk = sanitize_text_field($line_key);
        $qv = max(1, absint($q));
        if ( isset($cart[$lk]) ) {
          $cart[$lk]['cantidad'] = $qv;
          $cart[$lk]['qty']      = $qv; // legacy compat
        }
      }
      break;

    case 'remove':
      $item_key = isset($_POST['item_key']) ? sanitize_text_field( wp_unslash($_POST['item_key']) ) : '';
      if ( ! $item_key || ! isset($cart[$item_key]) ) {
        wp_send_json_error(['message' => esc_html__( 'Invalid item_key.', 'shoplite' )], 400);
      }
      unset($cart[$item_key]);
      break;

    case 'empty':
      $cart = [];
      break;

    default:
      wp_send_json_error(['message' => esc_html__( 'Invalid cart_action.', 'shoplite' )], 400);
  }

  $_SESSION['shoplite_cart'] = $cart;
  $_SESSION['carrito']       = $cart;

  ob_start();
  echo do_shortcode('[tienda_carrito]'); // uses your real shortcode
  $cart_html = ob_get_clean();

  wp_send_json_success([
    'html'       => $cart_html,
    'cart_count' => count($cart),
  ]);
}



// Save customer email (used for abandoned carts + prefill checkout)
add_action( 'wp_ajax_shoplite_save_customer_email',        'shoplite_ajax_save_customer_email' );
add_action( 'wp_ajax_nopriv_shoplite_save_customer_email', 'shoplite_ajax_save_customer_email' );

function shoplite_ajax_save_customer_email() {

  // Session (unified, no warnings)
  if ( function_exists( 'shoplite_maybe_session_start' ) ) {
    shoplite_maybe_session_start();
  } else {
    if ( session_status() === PHP_SESSION_NONE && ! headers_sent() ) {
      @session_start();
    }
  }

  $email = isset($_POST['email']) ? sanitize_email( wp_unslash($_POST['email']) ) : '';
  $name  = isset($_POST['name'])  ? sanitize_text_field( wp_unslash($_POST['name']) ) : '';

  if ( $email && is_email( $email ) ) {
    $_SESSION['shoplite_customer_email'] = $email;
    if ( $name !== '' ) {
      $_SESSION['shoplite_customer_name'] = $name;
    }
    wp_send_json_success([ 'saved' => true ]);
  }

  unset($_SESSION['shoplite_customer_email']);
  wp_send_json_success([ 'saved' => false ]);
}





// Ensure shortcodes inside content are processed
add_filter('the_content', 'do_shortcode', 11);

// Payment confirmation reading session_id from URL (Stripe Checkout)
// Shortcode: [shoplite_pago_exitoso] – REST version
add_action('init', function () {

  add_shortcode('shoplite_pago_exitoso', function () {
    $sid = isset($_GET['session_id']) ? sanitize_text_field($_GET['session_id']) : '';
    if ($sid === '') {
      return '<p>' . esc_html__( 'Thank you for your purchase.', 'shoplite' ) . '</p>';
    }

    if (function_exists('shoplite_guard_once') && shoplite_guard_once('shortcode_pago_exitoso')) {
      return '';
    }

    $secret = function_exists('shoplite_stripe_seckey') ? shoplite_stripe_seckey() : '';
    if (empty($secret)) {
      return '<p>' . esc_html__( 'Payment received. We are verifying the information…', 'shoplite' ) . '</p>';
    }

    $res = wp_remote_get("https://api.stripe.com/v1/checkout/sessions/" . rawurlencode($sid), [
      'headers' => [ 'Authorization' => 'Bearer ' . $secret ],
      'timeout' => 15,
    ]);

    if (is_wp_error($res)) {
      return '<p>' . esc_html__( 'Thank you for your purchase. We are verifying your payment…', 'shoplite' ) . '</p>';
    }

    $code = wp_remote_retrieve_response_code($res);
    $data = json_decode(wp_remote_retrieve_body($res), true);

    if ($code === 200 && is_array($data) && !empty($data['payment_status']) && $data['payment_status'] === 'paid') {
      $btn = '<p><a class="button" href="'.esc_url(home_url('/')).'">' . esc_html__( 'Back to the shop', 'shoplite' ) . '</a></p>';
      return '<p><strong>' . esc_html__( 'Payment confirmed ✅', 'shoplite' ) . '</strong> ' . esc_html__( 'Thank you for your purchase!', 'shoplite' ) . '</p>' . $btn;
    }

    return '<p>' . esc_html__( 'Thank you for your purchase. We are verifying your payment…', 'shoplite' ) . '</p>';
  });

});


// =============================
// Paginated catalog shortcode
// Usage: [shoplite_catalogo per_page="16" cat="ofertas,novedades" orderby="date" order="DESC"]
// Alias: [tsl_catalog] (same behavior)
// =============================
// =============================
// Catalog with search + pagination (robust)
// Usage:
//   [shoplite_catalogo per_page=5]
//   [tsl_catalog por_pagina=5]
//   [tienda_catalogo per_page=12 cat="ofertas,novedades"]
// Adjust $cpt_slug and $taxo_slug if your real slugs are different.
// =============================
// =============================
// Catalog with search + pagination (robust, without <form> that some themes may block)
// =============================
add_action('init', function () {

  // Re-enforce posts_per_page if another hook overrides it (max priority)
  add_action('pre_get_posts', function($q){
    if ($q->get('shoplite_catalogo_flag')) {
      $ppp = (int) $q->get('shoplite_catalogo_ppp');
      if ($ppp > 0) $q->set('posts_per_page', $ppp);
    }
  }, PHP_INT_MAX);

  $callback = function($atts = []) {
    $atts = shortcode_atts([
      'per_page'    => '',
      'por_pagina'  => '',
      'cat'         => '',
      'categoria'   => '',
      'orderby'     => 'date',
      'ordenar_por' => '',
      'order'       => 'DESC',
      'orden'       => '',
    ], $atts, 'shoplite_catalogo');

    $per_page = (int) ( $atts['per_page'] !== '' ? $atts['per_page'] : $atts['por_pagina'] );
    if ($per_page <= 0) $per_page = 16;

    $search = isset($_GET['q']) ? sanitize_text_field(wp_unslash($_GET['q'])) : '';
    if ($search === '' && isset($_GET['s'])) {
      $search = sanitize_text_field(wp_unslash($_GET['s']));
    }

    $orderby = sanitize_key( $atts['orderby'] !== '' ? $atts['orderby'] : $atts['ordenar_por'] );
    if (!$orderby) $orderby = 'date';
    $order = strtoupper($atts['order'] !== '' ? $atts['order'] : $atts['orden']) === 'ASC' ? 'ASC' : 'DESC';
    $cats  = trim( $atts['cat'] !== '' ? $atts['cat'] : $atts['categoria'] );

    $paged = 1;
    $qv = max((int)get_query_var('paged'), (int)get_query_var('page'));
    if ($qv > 1) $paged = $qv;
    if (isset($_GET['slp'])) { $p = (int) $_GET['slp']; if ($p > 0) $paged = $p; }

    // =============================
// Slugs reales (confirmados)
// =============================
$cpt_slug  = 'shoplite_producto';
$taxo_slug = 'shoplite_categoria';

// =============================
// Base query args
// =============================
$args = [
  'post_type'              => $cpt_slug,
  'post_status'            => 'publish',
  'posts_per_page'         => $per_page,
  'paged'                  => $paged,
  'orderby'                => $orderby,
  'order'                  => $order,

  'no_found_rows'          => false,
  'update_post_meta_cache' => true,
  'update_post_term_cache' => true,

  'suppress_filters'       => false,

  'shoplite_catalogo_flag' => true,
  'shoplite_catalogo_ppp'  => $per_page,
];

// ===============================
// Filtro por categorías (si aplica)
// ===============================
if ( $cats !== '' ) {
  $terms = array_filter(
    array_map('sanitize_title', array_map('trim', explode(',', $cats)))
  );

  if ( $terms ) {
    $args['tax_query'] = [[
      'taxonomy' => $taxo_slug,
      'field'    => 'slug',
      'terms'    => $terms,
    ]];
  }
}

// ===============================
// BÚSQUEDA SOLO POR TÍTULO (robusta)
// ===============================
$search = is_string($search) ? trim($search) : '';

if ( $search !== '' ) {

  global $wpdb;

  $ids = $wpdb->get_col(
    $wpdb->prepare(
      "SELECT ID
       FROM {$wpdb->posts}
       WHERE post_type = %s
         AND post_status = 'publish'
         AND INSTR(post_title, %s) > 0
       ORDER BY post_date DESC
       LIMIT 500",
      $cpt_slug,
      $search
    )
  );

  // Forzar WP_Query a estos IDs
  $args['post__in'] = ! empty($ids)
    ? array_map('intval', $ids)
    : [0];

  // Mantener orden por relevancia simple
  $args['orderby'] = 'post__in';
  $args['order']   = 'ASC';

  // Evitar páginas vacías durante búsqueda
  $args['paged'] = 1;

  // Ya no usamos búsqueda nativa
  unset($args['s']);
}

// ===============================
// Ejecutar la query
// ===============================
$q = new WP_Query( $args );

    

    // =====================
    // END OF CATALOG / SHORTCODES (FRONTEND)
    // =====================

    // === Robust base URL ===
    $page_id  = get_queried_object_id();
    $page_url = $page_id ? get_permalink($page_id) : (is_singular() ? get_permalink() : home_url(add_query_arg([])));

    // === Search UI without <form> (for themes that block it) + noscript fallback ===
    $uid = 'slq_' . wp_generate_password(6, false, false);
    ?>
    <div class="shoplite-search" style="margin:0 0 1rem 0;">
      <input
        id="<?php echo esc_attr($uid); ?>"
        type="search"
        value="<?php echo esc_attr($search); ?>"
        placeholder="<?php echo esc_attr__( 'Search products…', 'shoplite' ); ?>"
      />
      <button type="button" data-slq="<?php echo esc_attr($uid); ?>">
        <?php echo esc_html__( 'Search', 'shoplite' ); ?>
      </button>
    </div>
    <script>
    (function(){
      var btn = document.querySelector('button[data-slq="<?php echo esc_js($uid); ?>"]');
      if(!btn) return;
      btn.addEventListener('click', function(){
        var inp = document.getElementById('<?php echo esc_js($uid); ?>');
        var v = inp ? (inp.value || '') : '';
        try {
          var url = new URL('<?php echo esc_js($page_url); ?>', window.location.origin);
          if (v) url.searchParams.set('q', v); else url.searchParams.delete('q');
          url.searchParams.delete('slp'); // reset pagination when searching
          window.location.href = url.toString();
        } catch(e) {
          // Simple fallback
          var base = '<?php echo esc_js(add_query_arg([], $page_url)); ?>';
          window.location.href = base + (v ? ('?q=' + encodeURIComponent(v)) : '');
        }
      });
    })();
    </script>
    <noscript>
      <form method="get" action="<?php echo esc_url($page_url); ?>" class="shoplite-search">
        <input
          type="search"
          name="q"
          value="<?php echo esc_attr($search); ?>"
          placeholder="<?php echo esc_attr__( 'Search products…', 'shoplite' ); ?>"
        />
        <button type="submit"><?php echo esc_html__( 'Search', 'shoplite' ); ?></button>
      </form>
    </noscript>
    <?php
    

    // === Grid ===
    if ($q->have_posts()) {
      echo '<div class="shoplite-grid">';
      while ($q->have_posts()) { $q->the_post();
        echo '<article class="shoplite-item">';
          echo '<a href="'.esc_url(get_permalink()).'">';
            if (has_post_thumbnail()) the_post_thumbnail('medium');
            echo '<h3>'.esc_html(get_the_title()).'</h3>';
          echo '</a>';
        echo '</article>';
      }
      echo '</div>';

      // Pagination preserving search
      $base_url = add_query_arg(array_filter(['q' => ($search !== '' ? $search : null)]), $page_url);
      $links = paginate_links([
        'total'     => max(1, (int)$q->max_num_pages),
        'current'   => max(1, $paged),
        'type'      => 'list',
        'base'      => esc_url( add_query_arg('slp', '%#%', $base_url) ),
        'format'    => '',
        'prev_text' => '« ' . esc_html__( 'Previous', 'shoplite' ),
        'next_text' => esc_html__( 'Next', 'shoplite' ) . ' »',
      ]);
      if ($q->max_num_pages > 1 && $links) {
        echo '<nav class="shoplite-pagination">'.$links.'</nav>';
      }

      wp_reset_postdata();
    } else {
      echo '<p>' . esc_html__( 'No products found.', 'shoplite' ) . '</p>';
    }

    // Minimal CSS
    ?>
    <style>
      .shoplite-grid { display:grid; grid-template-columns:repeat(auto-fill,minmax(220px,1fr)); gap:16px; }
      .shoplite-item img { width:100%; height:auto; display:block; }
      .shoplite-pagination ul { list-style:none; padding:0; display:flex; gap:.5rem; flex-wrap:wrap; }
      .shoplite-pagination a, .shoplite-pagination span { padding:.35rem .6rem; border:1px solid #ddd; border-radius:6px; text-decoration:none; }
      .shoplite-pagination .current { font-weight:600; }
      .shoplite-search input[type="search"] { padding:.4rem .6rem; }
      .shoplite-search button { padding:.4rem .6rem; margin-left:.25rem; cursor:pointer; }
    </style>
    <?php

    echo "\n<!-- shoplite per_page={$per_page} paged={$paged} found={$q->found_posts} pages={$q->max_num_pages} -->\n";
    return ob_get_clean();
  };

  add_shortcode('shoplite_catalogo', $callback);
  add_shortcode('tsl_catalog',       $callback);
  add_shortcode('tienda_catalogo',   $callback);
}, 20);


//Debug i18n
function shoplite_debug_i18n_notice() {
    if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $user_id = get_current_user_id();
    $meta    = $user_id ? get_user_meta( $user_id, 'shoplite_language', true ) : '(no-user)';

    $loaded  = is_textdomain_loaded( 'shoplite' ) ? 'yes' : 'no';

    $t = get_translations_for_domain( 'shoplite' );
    $lang_header = '';
    if ( $t && is_object( $t ) ) {
        $lang_header = $t->get_header( 'Language' );
    }

    echo '<div class="notice notice-info"><p>'
        . 'Shoplite debug — user_meta shoplite_language: <strong>' . esc_html( (string) $meta ) . '</strong>'
        . ' | determine_locale(): <strong>' . esc_html( determine_locale() ) . '</strong>'
        . ' | textdomain loaded: <strong>' . esc_html( $loaded ) . '</strong>'
        . ' | MO header Language: <strong>' . esc_html( (string) $lang_header ) . '</strong>'
        . '</p><p>'
        . 'Test string: <strong>' . esc_html__( 'Text & Style', 'shoplite' ) . '</strong>'
        . '</p></div>';
}
add_action( 'admin_notices', 'shoplite_debug_i18n_notice' );


/**
 * Shoplite: limit product search to titles (avoid matches in HTML like "<span>").
 */
add_filter('posts_search', function ($search_sql, $wp_query) {
    global $wpdb;

    if ( is_admin() ) return $search_sql;

    // Only for Shoplite products
    $post_type = $wp_query->get('post_type');
    if ( $post_type !== 'shoplite_producto' ) return $search_sql;

    $s = $wp_query->get('s');
    if ( ! is_string($s) || trim($s) === '' ) return $search_sql;

    $terms = preg_split('/\s+/', trim($s));
    $terms = array_values(array_filter($terms, fn($t) => $t !== ''));

    if ( empty($terms) ) return $search_sql;

    $likes = [];
    foreach ($terms as $t) {
        $likes[] = $wpdb->prepare("{$wpdb->posts}.post_title LIKE %s", '%' . $wpdb->esc_like($t) . '%');
    }

    return ' AND (' . implode(' AND ', $likes) . ') ';
}, 20, 2);
