<?php
// Seeder seguro para Shoplite – APAGADO por defecto
// - No se carga salvo que definas SHOPLITE_ENABLE_SEEDER=true
// - Genera productos de prueba y los marca con meta _shoplite_seed=1 para borrarlos fácil
// - Ofrece WP-CLI y una pantalla de admin (solo Admin)

/** ===== Helpers internos ===== */

if (!function_exists('shoplite_seed_make_term')) {
  function shoplite_seed_make_term($taxonomy, $name) {
    $term = term_exists($name, $taxonomy);
    if (!$term) {
      $term = wp_insert_term($name, $taxonomy);
    }
    return is_wp_error($term) ? 0 : (int)($term['term_id'] ?? $term['term_id'] ?? 0);
  }
}

if (!function_exists('shoplite_seed_make_categories')) {
  function shoplite_seed_make_categories() {
    // Cambia 'shoplite_cat' si usas otra taxonomía para productos
    $tax = 'shoplite_cat';
    if (!taxonomy_exists($tax)) return [];

    $cats = ['General', 'Ofertas', 'Novedades', 'Top Ventas'];
    $ids  = [];
    foreach ($cats as $c) { $id = shoplite_seed_make_term($tax, $c); if ($id) $ids[] = $id; }
    return $ids;
  }
}

if (!function_exists('shoplite_seed_rand_price')) {
  function shoplite_seed_rand_price() {
    // 5.00 – 99.00
    return round(mt_rand(500, 9900) / 100, 2);
  }
}

if (!function_exists('shoplite_seed_product_args')) {
  function shoplite_seed_product_args($i) {
    return [
      'post_title'   => "Producto demo #$i",
      'post_type'    => 'shoplite_producto',
      'post_status'  => 'publish',
      'post_name'    => 'producto-demo-' . $i,
      'post_content' => 'Demo product generated automatically for performance tests.',
    ];
  }
}

if (!function_exists('shoplite_seed_create_products')) {
  function shoplite_seed_create_products($count = 150, $physical_ratio = 0.5) {
    if (!post_type_exists('shoplite_producto')) {
      return new WP_Error('no_cpt', 'El CPT shoplite_producto no existe.');
    }

    $cat_ids = shoplite_seed_make_categories(); // puede ser []
    $created = 0;

    for ($i = 1; $i <= $count; $i++) {
      $post_id = wp_insert_post(shoplite_seed_product_args($i), true);
      if (is_wp_error($post_id) || !$post_id) continue;

      // Meta: precio + físico/envío
      $price   = shoplite_seed_rand_price();
      $is_phys = (mt_rand() / mt_getrandmax()) < $physical_ratio; // ~50% físicos
      $ship    = $is_phys ? round(mt_rand(200, 1500) / 100, 2) : 0.0;

      // Ajusta keys de meta a las que uses realmente en tu plugin
      update_post_meta($post_id, '_tb_price', $price);
      update_post_meta($post_id, '_tb_is_physical', $is_phys ? '1' : '0');
      update_post_meta($post_id, '_tb_shipping_cost', $ship);

      // Marca para poder borrar después
      update_post_meta($post_id, '_shoplite_seed', 1);

      // Categoría aleatoria si existe taxonomía
      if (!empty($cat_ids)) {
        $cat = $cat_ids[array_rand($cat_ids)];
        wp_set_post_terms($post_id, [$cat], 'shoplite_cat', true);
      }

      $created++;
    }

    return $created;
  }
}

if (!function_exists('shoplite_seed_delete_products')) {
  function shoplite_seed_delete_products() {
    $q = new WP_Query([
      'post_type'      => 'shoplite_producto',
      'posts_per_page' => -1,
      'post_status'    => 'any',
      'meta_key'       => '_shoplite_seed',
      'meta_value'     => 1,
      'fields'         => 'ids',
      'no_found_rows'  => true,
    ]);

    $deleted = 0;
    foreach ($q->posts as $pid) {
      wp_delete_post($pid, true);
      $deleted++;
    }
    return $deleted;
  }
}

/** ===== WP-CLI (opcional) ===== */
if (defined('WP_CLI') && WP_CLI) {
  WP_CLI::add_command('shoplite seed', function($args, $assoc_args) {
    $count = isset($assoc_args['count']) ? max(1, (int)$assoc_args['count']) : 150;
    $ratio = isset($assoc_args['physical']) ? max(0, min(1, (float)$assoc_args['physical'])) : 0.5;

    $res = shoplite_seed_create_products($count, $ratio);
    if (is_wp_error($res)) {
      WP_CLI::error($res->get_error_message());
    } else {
      WP_CLI::success("Creado(s) $res producto(s) de prueba.");
    }
  });
  WP_CLI::add_command('shoplite seed-clean', function() {
    $n = shoplite_seed_delete_products();
    WP_CLI::success("Eliminado(s) $n producto(s) de prueba.");
  });
}

/** ===== Pantalla de admin (Tools) ===== */
add_action('admin_menu', function () {
  if (!current_user_can('manage_options')) return;

  add_submenu_page(
    'options-general.php',
    'Shoplite – Herramientas',
    'Shoplite – Herramientas',
    'manage_options',
    'shoplite-seed-tools',
    'shoplite_seed_tools_render'
  );
});

if (!function_exists('shoplite_seed_tools_render')) {
  function shoplite_seed_tools_render() {
    if (!current_user_can('manage_options')) { wp_die('Permisos insuficientes'); }

    $action  = isset($_POST['shoplite_seed_action']) ? sanitize_text_field($_POST['shoplite_seed_action']) : '';
    $count   = isset($_POST['shoplite_seed_count']) ? max(1, (int)$_POST['shoplite_seed_count']) : 150;
    $ratio   = isset($_POST['shoplite_seed_ratio']) ? max(0, min(1, (float)$_POST['shoplite_seed_ratio'])) : 0.5;
    $message = '';

    if (!empty($action) && check_admin_referer('shoplite_seed_tools')) {
      if ($action === 'seed') {
        $res = shoplite_seed_create_products($count, $ratio);
        $message = is_wp_error($res) ? $res->get_error_message() : "Creado(s) $res producto(s) de prueba.";
      } elseif ($action === 'clean') {
        $n = shoplite_seed_delete_products();
        $message = "Eliminado(s) $n producto(s) de prueba.";
      }
    }

    ?>
    <div class="wrap">
      <h1>Shoplite – Herramientas (Seeder)</h1>
      <?php if ($message): ?>
        <div class="notice notice-success"><p><?php echo esc_html($message); ?></p></div>
      <?php endif; ?>

      <form method="post">
        <?php wp_nonce_field('shoplite_seed_tools'); ?>
        <table class="form-table" role="presentation">
          <tr>
            <th scope="row">Cantidad de productos</th>
            <td><input type="number" name="shoplite_seed_count" value="<?php echo esc_attr($count); ?>" min="1" max="1000" /></td>
          </tr>
          <tr>
            <th scope="row">Proporción físicos</th>
            <td>
              <input type="number" name="shoplite_seed_ratio" step="0.1" min="0" max="1" value="<?php echo esc_attr($ratio); ?>" />
              <p class="description">0 = todos digitales, 1 = todos físicos (por defecto 0.5)</p>
            </td>
          </tr>
        </table>
        <p>
          <button type="submit" name="shoplite_seed_action" value="seed" class="button button-primary">Generar productos de prueba</button>
          <button type="submit" name="shoplite_seed_action" value="clean" class="button">Borrar productos de prueba</button>
        </p>
      </form>

      <hr/>
      <p><strong>Nota:</strong> todo lo generado se marca con el meta <code>_shoplite_seed=1</code> para poder borrarlo sin afectar tus productos reales.</p>
    </div>
    <?php
  }
}
