<?php
if (!defined('ABSPATH')) exit;

/**
 * Inventory admin page (submenu) for Shoplite.
 *
 * Product CPT: shoplite_producto
 * Uses meta:
 *  - _sl_manage_stock ( '1' | '0' )
 *  - _sl_stock_qty    ( int )
 * Physical filter:
 *  - _tb_is_physical  ( '1' )
 */

add_action('admin_menu', function () {

    // Parent slug MUST match your existing Shoplite menu slug.
    // If this doesn't show, change 'talariasoft-shoplite' to your real parent slug.
    add_submenu_page(
        'talariasoft-shoplite',
        __('Inventario', 'shoplite'),
        __('Inventario', 'shoplite'),
        'manage_options',
        'shoplite-inventory',
        'shoplite_render_inventory_page'
    );

}, 30);

/**
 * Handle POST updates (bulk-style, but simple).
 */
function shoplite_inventory_handle_post() {
    if ( ! is_admin() ) return;
    if ( ! isset($_POST['shoplite_inventory_save']) ) return;

    if ( ! current_user_can('manage_options') ) {
        wp_die( esc_html__('You do not have permission to manage inventory.', 'shoplite') );
    }

    check_admin_referer('shoplite_inventory_save', 'shoplite_inventory_nonce');

    $rows = isset($_POST['inv']) && is_array($_POST['inv']) ? $_POST['inv'] : [];
    $updated = 0;

    foreach ($rows as $product_id => $row) {
        $pid = (int) $product_id;
        if ($pid <= 0) continue;
        if ( ! is_array($row) ) continue;

        // Only for our CPT
        if ( get_post_type($pid) !== 'shoplite_producto' ) continue;

        $manage = ! empty($row['manage']) ? '1' : '0';
        $qty    = isset($row['qty']) ? (int) $row['qty'] : 0;
        if ($qty < 0) $qty = 0;

        update_post_meta($pid, '_sl_manage_stock', $manage);
        update_post_meta($pid, '_sl_stock_qty', (string)$qty);

        $updated++;
    }

    // Redirect to avoid resubmission
    $url = add_query_arg([
        'page'    => 'shoplite-inventory',
        'updated' => $updated,
    ], admin_url('admin.php'));

    // Preserve filter/sort in redirect
    foreach (['view','orderby','order','s'] as $k) {
        if (isset($_GET[$k]) && $_GET[$k] !== '') {
            $url = add_query_arg($k, sanitize_text_field(wp_unslash($_GET[$k])), $url);
        }
    }

    wp_safe_redirect($url);
    exit;
}
add_action('admin_init', 'shoplite_inventory_handle_post');

/**
 * Render inventory page.
 */
function shoplite_render_inventory_page() {
    if ( ! current_user_can('manage_options') ) return;

    $view    = isset($_GET['view']) ? sanitize_text_field(wp_unslash($_GET['view'])) : 'all';
    $orderby = isset($_GET['orderby']) ? sanitize_text_field(wp_unslash($_GET['orderby'])) : 'title';
    $order   = isset($_GET['order']) ? strtoupper(sanitize_text_field(wp_unslash($_GET['order']))) : 'ASC';
    $search  = isset($_GET['s']) ? sanitize_text_field(wp_unslash($_GET['s'])) : '';

    if (!in_array($view, ['all','oos','low','unmanaged'], true)) $view = 'all';
    if (!in_array($orderby, ['title','stock'], true)) $orderby = 'title';
    if (!in_array($order, ['ASC','DESC'], true)) $order = 'ASC';

    $low_threshold = (int) apply_filters('shoplite_low_stock_threshold', 5);

    // Feedback message
    $updated = isset($_GET['updated']) ? (int) $_GET['updated'] : 0;

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__('Inventario', 'shoplite') . '</h1>';

    if ($updated > 0) {
        echo '<div class="notice notice-success is-dismissible"><p>' .
            sprintf(esc_html__('%d producto(s) actualizados.', 'shoplite'), $updated) .
            '</p></div>';
    }

    // View links
    $base = admin_url('admin.php?page=shoplite-inventory');
    $views = [
        'all'       => __('Todos', 'shoplite'),
        'oos'       => __('Sin stock', 'shoplite'),
        'low'       => __('Stock bajo', 'shoplite'),
        'unmanaged' => __('No gestionado', 'shoplite'),
    ];

    echo '<ul class="subsubsub">';
    $i = 0;
    foreach ($views as $k => $label) {
        $i++;
        $url = add_query_arg(['view' => $k], $base);
        if ($search !== '') $url = add_query_arg('s', $search, $url);
        $class = ($view === $k) ? ' class="current"' : '';
        echo '<li><a' . $class . ' href="' . esc_url($url) . '">' . esc_html($label) . '</a>' . ($i < count($views) ? ' | ' : '') . '</li>';
    }
    echo '</ul>';

    // Search box
    echo '<form method="get" style="margin-top:12px">';
    echo '<input type="hidden" name="page" value="shoplite-inventory" />';
    echo '<input type="hidden" name="view" value="' . esc_attr($view) . '" />';
    echo '<p class="search-box">';
    echo '<label class="screen-reader-text" for="shoplite-inv-search">' . esc_html__('Buscar productos', 'shoplite') . '</label>';
    echo '<input id="shoplite-inv-search" type="search" name="s" value="' . esc_attr($search) . '" />';
    echo '<input type="submit" class="button" value="' . esc_attr__('Buscar', 'shoplite') . '" />';
    echo '</p>';
    echo '</form>';

    // Query products (physical only)
    $args = [
        'post_type'      => 'shoplite_producto',
        'post_status'    => 'publish',
        'posts_per_page' => 200, // MVP; if you need pagination later, we switch to WP_List_Table
        's'              => $search,
        'orderby'        => ($orderby === 'title') ? 'title' : 'title', // we'll sort by stock in PHP
        'order'          => $order,
        'meta_query'     => [
            [
                'key'   => '_tb_is_physical',
                'value' => '1',
            ],
        ],
    ];

    $products = get_posts($args);

    // Build rows with stock info
    $rows = [];
    foreach ($products as $p) {
        $pid = (int) $p->ID;

        $manage = (get_post_meta($pid, '_sl_manage_stock', true) === '1');
        $stock  = (int) get_post_meta($pid, '_sl_stock_qty', true);

        // Apply view filters
        if ($view === 'unmanaged' && $manage) continue;
        if ($view === 'oos' && (! $manage || $stock > 0)) continue;
        if ($view === 'low' && (! $manage || $stock <= 0 || $stock > $low_threshold)) continue;

        $state = 'ok';
        if (!$manage) $state = 'unmanaged';
        elseif ($stock <= 0) $state = 'oos';
        elseif ($stock <= $low_threshold) $state = 'low';

        $rows[] = [
            'id'     => $pid,
            'title'  => (string) $p->post_title,
            'manage' => $manage,
            'stock'  => $stock,
            'state'  => $state,
        ];
    }

    // Sort by stock if requested
    if ($orderby === 'stock') {
        usort($rows, function($a, $b) use ($order) {
            if ($a['stock'] === $b['stock']) return 0;
            $cmp = ($a['stock'] < $b['stock']) ? -1 : 1;
            return ($order === 'ASC') ? $cmp : -$cmp;
        });
    }

    // Sort links
    $sort_url = function($col) use ($base, $view, $orderby, $order, $search) {
        $new_order = 'ASC';
        if ($orderby === $col && $order === 'ASC') $new_order = 'DESC';
        $u = add_query_arg([
            'view'    => $view,
            'orderby' => $col,
            'order'   => $new_order,
        ], $base);
        if ($search !== '') $u = add_query_arg('s', $search, $u);
        return $u;
    };

    $arrow = function($col) use ($orderby, $order) {
        if ($orderby !== $col) return '';
        return $order === 'ASC' ? ' ▲' : ' ▼';
    };

    echo '<hr class="wp-header-end" />';

    if (empty($rows)) {
        echo '<p>' . esc_html__('No hay productos que coincidan con los filtros.', 'shoplite') . '</p>';
        echo '</div>';
        return;
    }

    echo '<form method="post">';
    wp_nonce_field('shoplite_inventory_save', 'shoplite_inventory_nonce');
    echo '<input type="hidden" name="shoplite_inventory_save" value="1" />';

    echo '<table class="widefat striped" style="max-width:1100px">';
    echo '<thead><tr>';
    echo '<th><a href="' . esc_url($sort_url('title')) . '">' . esc_html__('Producto', 'shoplite') . $arrow('title') . '</a></th>';
    echo '<th style="width:120px"><a href="' . esc_url($sort_url('stock')) . '">' . esc_html__('Stock', 'shoplite') . $arrow('stock') . '</a></th>';
    echo '<th style="width:160px">' . esc_html__('Gestionar stock', 'shoplite') . '</th>';
    echo '<th style="width:160px">' . esc_html__('Estado', 'shoplite') . '</th>';
    echo '<th style="width:120px">' . esc_html__('Acción', 'shoplite') . '</th>';
    echo '</tr></thead><tbody>';

    foreach ($rows as $r) {
        $pid    = (int) $r['id'];
        $title  = $r['title'];
        $manage = (bool) $r['manage'];
        $stock  = (int) $r['stock'];

        $state_label = __('OK', 'shoplite');
        if ($r['state'] === 'unmanaged') $state_label = __('No gestionado', 'shoplite');
        elseif ($r['state'] === 'oos') $state_label = __('Sin stock', 'shoplite');
        elseif ($r['state'] === 'low') $state_label = __('Bajo', 'shoplite');

        echo '<tr>';

        echo '<td>';
        echo '<strong><a href="' . esc_url(get_edit_post_link($pid)) . '">' . esc_html($title) . '</a></strong>';
        echo '<div class="row-actions">';
        echo '<span class="edit"><a href="' . esc_url(get_edit_post_link($pid)) . '">' . esc_html__('Editar', 'shoplite') . '</a></span>';
        echo '</div>';
        echo '</td>';

        echo '<td>';
        echo '<input type="number" min="0" step="1" name="inv[' . esc_attr($pid) . '][qty]" value="' . esc_attr($stock) . '" style="width:100px" ' . ($manage ? '' : 'disabled') . ' />';
        echo '</td>';

        echo '<td>';
        $checked = $manage ? 'checked' : '';
        echo '<label><input type="checkbox" name="inv[' . esc_attr($pid) . '][manage]" value="1" ' . $checked . ' /> ' . esc_html__('Sí', 'shoplite') . '</label>';
        echo '</td>';

        echo '<td>' . esc_html($state_label) . '</td>';

        echo '<td>';
        echo '<a class="button button-small" href="' . esc_url(get_permalink($pid)) . '" target="_blank" rel="noopener">' . esc_html__('Ver', 'shoplite') . '</a>';
        echo '</td>';

        echo '</tr>';
    }

    echo '</tbody></table>';

    echo '<p style="margin-top:12px">';
    echo '<button type="submit" class="button button-primary">' . esc_html__('Guardar cambios', 'shoplite') . '</button>';
    echo '</p>';

    echo '</form>';
    echo '</div>';

    // Small inline JS: enable/disable qty when manage checkbox toggled
    echo '<script>
    (function(){
      document.addEventListener("change", function(e){
        var el = e.target;
        if (!el || el.type !== "checkbox") return;
        if (!el.name || el.name.indexOf("[manage]") === -1) return;

        var pid = el.name.match(/inv\\[(\\d+)\\]\\[manage\\]/);
        if (!pid || !pid[1]) return;

        var qty = document.querySelector("input[name=\'inv[" + pid[1] + "][qty]\']");
        if (!qty) return;

        qty.disabled = !el.checked;
      });
    })();
    </script>';
}
