<?php
/**
 * ShopLite - CSV Importer
 * CPT: tb_producto
 * Metas: _tb_price, _tb_sku, _tb_is_physical, _tb_shipping_cost
 */
if (!defined('ABSPATH')) exit;

/* ========================= SETTINGS ========================= */

// Real CPT
if (!defined('SHOPLITE_CPT'))           define('SHOPLITE_CPT', 'tb_producto');

// Real metas used by your editor
if (!defined('SHOPLITE_META_PRICE'))    define('SHOPLITE_META_PRICE', '_tb_price');
if (!defined('SHOPLITE_META_SKU'))      define('SHOPLITE_META_SKU',   '_tb_sku');
if (!defined('SHOPLITE_META_PHYSICAL')) define('SHOPLITE_META_PHYSICAL','_tb_is_physical');
if (!defined('SHOPLITE_META_SHIP'))     define('SHOPLITE_META_SHIP',   '_tb_shipping_cost');

// (Optional) taxonomy if you use one; otherwise ignored
if (!defined('SHOPLITE_TAX_CAT'))       define('SHOPLITE_TAX_CAT',     'shoplite_categoria');

// Nonce
if (!defined('SHOPLITE_NONCE_IMPORT_CSV')) define('SHOPLITE_NONCE_IMPORT_CSV', 'shoplite_import_csv');

/* ========================= MENU ========================= */

// Menu: attached where the CPT is (or under 'shoplite' if it exists)
add_action('admin_menu', function () {
    $parent = 'edit.php?post_type=' . SHOPLITE_CPT;

    // If the CPT declares a custom parent menu, use it
    $pto = get_post_type_object(SHOPLITE_CPT);
    if ($pto) {
        if (is_string($pto->show_in_menu) && !empty($pto->show_in_menu)) {
            $parent = $pto->show_in_menu; // e.g. 'shoplite' or 'tienda-basica'
        } elseif ($pto->show_in_menu === true) {
            $parent = 'edit.php?post_type=' . SHOPLITE_CPT;
        }
    }

    // Main submenu (where your CPT really hangs)
    add_submenu_page(
        $parent,
        __('Import products (CSV)', 'shoplite'),
        __('Import (CSV)', 'shoplite'),
        'manage_options',
        'shoplite_import',
        'shoplite_render_import_page'
    );

    // Fallback: also add under top-level 'shoplite' if it exists and is not the same
    if ($parent !== 'shoplite') {
        add_submenu_page(
            'shoplite',
            __('Import products (CSV)', 'shoplite'),
            __('Import (CSV)', 'shoplite'),
            'manage_options',
            'shoplite_import_alt',
            'shoplite_render_import_page'
        );
    }

}, 99); // high priority so that CPT is already registered


// Also optionally under Tools > Import
add_action('admin_init', function () {
    if (function_exists('register_importer') && current_user_can('manage_options')) {
        register_importer(
            'shoplite_csv',
            __('ShopLite products (CSV)', 'shoplite'),
            __('Import products (SKU, price, categories, images).', 'shoplite'),
            'shoplite_render_import_page'
        );
    }
});

/* ========================= VIEW ========================= */

function shoplite_render_import_page() {
    if (!current_user_can('manage_options')) return;

    $done = null; $msg = ''; $stats = ['created'=>0,'updated'=>0,'skipped'=>0,'errors'=>0,'report'=>''];

    if (!empty($_POST) && check_admin_referer(SHOPLITE_NONCE_IMPORT_CSV)) {
        $update_by_sku = !empty($_POST['update_by_sku']);
        $res = shoplite_handle_csv_import($update_by_sku);
        $done  = $res['ok'];
        $msg   = $res['msg'];
        $stats = array_merge($stats, array_intersect_key($res, $stats));
    }
    ?>
    <div class="wrap shoplite-pro-import">
    <?php
    if ( function_exists( 'shoplite_render_brand_header' ) ) {
        shoplite_render_brand_header();
    }
    ?>
    <h1><?php esc_html_e( 'Import products (CSV) - ShopLite', 'shoplite' ); ?></h1>

    <?php if ( $done !== null ): ?>
        <div class="notice notice-<?php echo $done ? 'success' : 'error'; ?>"><p>
            <?php echo esc_html( $msg ); ?>
            <?php if ( $done ): ?>
                <br>
                <?php printf(
                    'Created: %d - Updated: %d - Skipped: %d - Errors: %d',
                    (int) $stats['created'],
                    (int) $stats['updated'],
                    (int) $stats['skipped'],
                    (int) $stats['errors']
                ); ?>
                <?php if ( ! empty( $stats['report'] ) ): ?>
                    <br><a href="<?php echo esc_url( $stats['report'] ); ?>" target="_blank" rel="noopener">
                        Download report
                    </a>
                <?php endif; ?>
            <?php endif; ?>
        </p></div>
    <?php endif; ?>


        <p><strong>Accepted headers (ES/EN):</strong>
            title, (price|precio), (sku|SKU), description, image_url,
            (shipping_cost|coste_envio), (is_physical|fisico),
            category (optional), gallery (URLs separated with <code>|</code>).
            Separator: <code>;</code> or <code>,</code>. Encoding: UTF-8.</p>

        <form method="post" enctype="multipart/form-data">
            <?php wp_nonce_field(SHOPLITE_NONCE_IMPORT_CSV); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="csv">CSV file</label></th>
                    <td><input type="file" name="csv" id="csv" accept=".csv,text/csv,.txt" required></td>
                </tr>
                <tr>
                    <th scope="row">Options</th>
                    <td>
                        <label><input type="checkbox" name="update_by_sku" value="1"> Update if a product with the same SKU exists</label>
                    </td>
                </tr>
            </table>
            <p class="submit">
                <button type="submit" class="button button-primary">Import</button>
            </p>
        </form>

        <p><strong>Recommendation:</strong> make sure products include <em>Price</em> and <em>SKU</em>.
        Products with missing or zero price (<= 0) are skipped to avoid empty catalog pages.</p>
    </div>
    <?php
}

/* ========================= LOGIC ========================= */

function shoplite_handle_csv_import($update_by_sku = false) {
    $out = ['ok'=>false,'msg'=>'', 'created'=>0,'updated'=>0,'skipped'=>0,'errors'=>0,'report'=>''];

    if (empty($_FILES['csv']['tmp_name'])) {
        $out['msg'] = 'CSV file not received.'; return $out;
    }

    @set_time_limit(0);
    @ini_set('auto_detect_line_endings', true);
    @ini_set('memory_limit', '256M');

    $tmp = $_FILES['csv']['tmp_name'];
    $fh  = fopen($tmp, 'r');
    if (!$fh) { $out['msg'] = 'Could not open CSV file.'; return $out; }

    // Skip BOM if present
    $first = fread($fh, 3);
    if ($first !== "\xEF\xBB\xBF") fseek($fh, 0);

    // Detect separator
    $probe = fgets($fh);
    if ($probe === false) { fclose($fh); $out['msg']='Empty CSV file.'; return $out; }
    $delims = [',',';']; $sep = ','; $max=0;
    foreach ($delims as $d) {
        $c = str_getcsv($probe, $d);
        if (count($c)>$max){ $max=count($c); $sep=$d; }
    }
    fseek($fh, ($first === "\xEF\xBB\xBF") ? 3 : 0);

    // Headers + normalizer
    $headers = fgetcsv($fh, 0, $sep);
    if (!$headers) { fclose($fh); $out['msg']='CSV file without headers.'; return $out; }

    $norm = function($s){
        $s = trim(mb_strtolower((string)$s, 'UTF-8'));
        return strtr(
            $s,
            [
                "\xC3\xA1" => 'a', // a-acute
                "\xC3\xA9" => 'e', // e-acute
                "\xC3\xAD" => 'i', // i-acute
                "\xC3\xB3" => 'o', // o-acute
                "\xC3\xBA" => 'u', // u-acute
                "\xC3\xB1" => 'n', // n-tilde
            ]
        );
    };

    $aliases = [
        'title'         => ['title','titulo'],
        'sku'           => ['sku'],
        'price'         => ['price','precio'],
        'description'   => ['description','descripcion','content'],
        'image_url'     => ['image_url','imagen','imagen_url','url_imagen'],
        'shipping_cost' => ['shipping_cost','coste_envio','gastos_envio','envio','coste_de_envio'],
        'is_physical'   => ['is_physical','fisico','physical','producto_fisico'],
        'category'      => ['category','categoria'],
        'gallery'       => ['gallery','galeria'],
    ];

    $col = [];
    foreach ($headers as $i=>$h){
        $h = $norm($h);
        foreach ($aliases as $canon=>$alts){
            foreach ($alts as $a){
                if ($h === $norm($a)) { $col[$canon]=$i; break 2; }
            }
        }
    }

    // Minimum required headers
    foreach (['title','price','sku'] as $req){
        if (!isset($col[$req])) { fclose($fh); $out['msg']="Missing required header: $req"; return $out; }
    }

    // Helpers
    $to_float = function($v){
        $v = trim((string)$v);
        if ($v==='') return null;
        $v = str_replace(',', '.', $v);
        return is_numeric($v) ? (float)$v : null;
    };

    $errors = [];
    wp_defer_term_counting(true);
    wp_defer_comment_counting(true);

    while (($row = fgetcsv($fh, 0, $sep)) !== false) {
        if (count(array_filter($row, fn($x)=>trim((string)$x) !== '')) === 0) continue;

        $title = trim($row[$col['title']] ?? '');
        $sku   = sanitize_text_field($row[$col['sku']] ?? '');
        $price = $to_float($row[$col['price']] ?? null);

        if ($title===''){ $out['skipped']++; $errors[]=['fila'=>ftell($fh),'sku'=>$sku,'error'=>'Empty title']; continue; }
        if ($sku===''){   $out['skipped']++; $errors[]=['fila'=>ftell($fh),'sku'=>'','error'=>'Empty SKU']; continue; }
        if ($price===null || $price<=0){ $out['skipped']++; $errors[]=['fila'=>ftell($fh),'sku'=>$sku,'error'=>'Empty price or <= 0']; continue; }

        // Search by SKU if requested
        $post_id = 0;
        if ($update_by_sku) {
            $q = new WP_Query([
                'post_type'      => SHOPLITE_CPT,
                'posts_per_page' => 1,
                'fields'         => 'ids',
                'meta_query'     => [[
                    'key'   => SHOPLITE_META_SKU,
                    'value' => $sku,
                ]],
                'no_found_rows'  => true,
            ]);
            if ($q->have_posts()) $post_id = (int)$q->posts[0];
            wp_reset_postdata();
        }

        // Create/update
        $args = [
            'post_type'   => SHOPLITE_CPT,
            'post_title'  => $title,
            'post_status' => 'publish',
        ];
        if (isset($col['description'])) $args['post_content'] = wp_kses_post($row[$col['description']] ?? '');

        if ($post_id) {
            $args['ID'] = $post_id;
            $res = wp_update_post($args, true);
            if (is_wp_error($res)) { $out['errors']++; $errors[]=['fila'=>ftell($fh),'sku'=>$sku,'error'=>'wp_update_post: '.$res->get_error_message()]; continue; }
        } else {
            $res = wp_insert_post($args, true);
            if (is_wp_error($res)) { $out['errors']++; $errors[]=['fila'=>ftell($fh),'sku'=>$sku,'error'=>'wp_insert_post: '.$res->get_error_message()]; continue; }
            $post_id = (int)$res;
        }

        // Metas: exact keys used by your plugin
        update_post_meta($post_id, SHOPLITE_META_PRICE, $price);
        update_post_meta($post_id, SHOPLITE_META_SKU,   $sku);

        if (isset($col['is_physical'])) {
            $raw = $norm($row[$col['is_physical']] ?? '');
            $is_physical = in_array($raw, ['1','si','yes','true','y']) ? '1' : '0';
            update_post_meta($post_id, SHOPLITE_META_PHYSICAL, $is_physical);
        }
        if (isset($col['shipping_cost'])) {
            $ship = $to_float($row[$col['shipping_cost']] ?? null);
            if ($ship === null) delete_post_meta($post_id, SHOPLITE_META_SHIP);
            else update_post_meta($post_id, SHOPLITE_META_SHIP, $ship);
        }

        // Categories (if you use a taxonomy)
        if (taxonomy_exists(SHOPLITE_TAX_CAT) && isset($col['category'])) {
            $raw = (string)($row[$col['category']] ?? '');
            if ($raw !== '') {
                $parts = preg_split('/[;,]/', $raw);
                $terms = array_values(array_filter(array_map('trim', $parts), fn($v)=>$v!==''));
                if ($terms) {
                    foreach ($terms as $t) {
                        if (!term_exists($t, SHOPLITE_TAX_CAT)) wp_insert_term($t, SHOPLITE_TAX_CAT);
                    }
                    wp_set_object_terms($post_id, $terms, SHOPLITE_TAX_CAT, false);
                }
            }
        }

        // Featured image (simple download by URL)
        if (isset($col['image_url'])) {
            $img = trim((string)($row[$col['image_url']] ?? ''));
            if ($img !== '') {
                $att_id = shoplite_attach_image_from_url($img, $post_id, $title);
                if ($att_id) set_post_thumbnail($post_id, $att_id);
            }
        }

        // Gallery
        if (isset($col['gallery'])) {
            $raw = (string)($row[$col['gallery']] ?? '');
            if ($raw !== '') {
                $urls = array_values(array_filter(array_map('trim', explode('|', $raw)), fn($u)=>$u!==''));
                $ids  = [];
                foreach ($urls as $u) {
                    $aid = shoplite_attach_image_from_url($u, $post_id, $title);
                    if ($aid) $ids[] = $aid;
                }
                if ($ids) update_post_meta($post_id, '_shoplite_gallery_ids', $ids);
            }
        }

        $update_by_sku && $q->have_posts() ? $out['updated']++ : $out['created']++;
    }

    fclose($fh);
    wp_defer_term_counting(false);
    wp_defer_comment_counting(false);

    if (!empty($errors)) $out['report'] = shoplite_write_import_report($errors);

    $out['ok']  = true;
    $out['msg'] = 'Import finished.';
    return $out;
}

/* ========================= UTILITIES ========================= */

// Download image by URL -> attachment_id (0 if it fails)
function shoplite_attach_image_from_url($url, $post_id = 0, $desc = '') {
    if (!function_exists('media_sideload_image')) {
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';
    }
    $tmp = download_url($url);
    if (is_wp_error($tmp)) return 0;

    $name = basename(parse_url($url, PHP_URL_PATH));
    $file_array = ['name' => $name ?: wp_unique_filename(sys_get_temp_dir(), 'image'), 'tmp_name' => $tmp];

    $results = wp_handle_sideload($file_array, ['test_form'=>false]);
    if (!empty($results['error'])) { @unlink($tmp); return 0; }

    $attachment = [
        'post_mime_type' => $results['type'],
        'post_title'     => sanitize_file_name($file_array['name']),
        'post_content'   => '',
        'post_status'    => 'inherit',
        'post_parent'    => $post_id,
    ];
    $attach_id = wp_insert_attachment($attachment, $results['file'], $post_id);
    if (is_wp_error($attach_id)) return 0;

    $attach_data = wp_generate_attachment_metadata($attach_id, $results['file']);
    wp_update_attachment_metadata($attach_id, $attach_data);

    return (int)$attach_id;
}

// Write CSV error report (row, sku, error) -> URL
function shoplite_write_import_report(array $rows) {
    $dir = WP_CONTENT_DIR . '/uploads/shoplite-imports/';
    if (!wp_mkdir_p($dir)) return '';
    $file = $dir . 'import_report_' . date('Ymd_His') . '.csv';

    $h = fopen($file, 'w'); if (!$h) return '';
    fputcsv($h, ['row','sku','error']);
    foreach ($rows as $r) fputcsv($h, [$r['fila'] ?? '', $r['sku'] ?? '', $r['error'] ?? '']);
    fclose($h);

    return content_url('uploads/shoplite-imports/' . basename($file));
}
