<?php
if (!defined('ABSPATH')) exit;

/**
 * Register internal CPT for digital deliveries.
 */
function sl_register_cpt_delivery() {
  $args = array(
    'labels' => array(
      'name' => 'Shoplite Deliveries',
      'singular_name' => 'Shoplite Delivery',
    ),
    'public' => false,
    'show_ui' => false,
    'show_in_menu' => false,
    'exclude_from_search' => true,
    'publicly_queryable' => false,
    'has_archive' => false,
    'rewrite' => false,
    'supports' => array('title'),
  );

  register_post_type('shoplite_delivery', $args);
}
add_action('init', 'sl_register_cpt_delivery', 5);

/**
 * Rewrite endpoint: /shoplite-download/<token>/
 * (In dev you can also use ?sl_download_token=... so rewrite is optional.)
 */
function sl_delivery_add_rewrite() {
  add_rewrite_rule(
    '^shoplite-download/([^/]+)/?$',
    'index.php?sl_download_token=$matches[1]',
    'top'
  );
}
add_action('init', 'sl_delivery_add_rewrite', 10);

function sl_delivery_query_vars($vars) {
  $vars[] = 'sl_download_token';
  return $vars;
}
add_filter('query_vars', 'sl_delivery_query_vars');

/**
 * Create a delivery record.
 * Returns token string or false.
 */
function sl_delivery_create($args) {
  $defaults = array(
    'order_id' => 0,
    'product_id' => 0,
    'attachment_id' => 0,
    'external_url' => '',
    'expires_at' => 0,
    'remaining' => 5, // default 5 downloads
    'buyer_email' => '',
  );
  $a = wp_parse_args($args, $defaults);

  // token as slug
  $token = wp_generate_password(32, false, false);
  $token = strtolower($token);

  $post_id = wp_insert_post(array(
    'post_type' => 'shoplite_delivery',
    'post_status' => 'publish',
    'post_title' => 'Delivery ' . ($a['order_id'] ? ('#' . (int)$a['order_id']) : ''),
    'post_name' => $token,
  ), true);

  if (is_wp_error($post_id)) {
    return false;
  }

  update_post_meta($post_id, 'sl_order_id', (int)$a['order_id']);
  update_post_meta($post_id, 'sl_product_id', (int)$a['product_id']);
  update_post_meta($post_id, 'sl_attachment_id', (int)$a['attachment_id']);
  update_post_meta($post_id, 'sl_external_url', sanitize_text_field($a['external_url']));
  update_post_meta($post_id, 'sl_expires_at', (int)$a['expires_at']);   // 0 = no expiry
  update_post_meta($post_id, 'sl_remaining', (int)$a['remaining']);     // downloads
  update_post_meta($post_id, 'sl_buyer_email', sanitize_email($a['buyer_email']));
  update_post_meta($post_id, 'sl_status', 'active');

  return $token;
}

/**
 * Get delivery post by token (slug).
 */
function sl_delivery_get_by_token($token) {
  $token = sanitize_title($token);
  if (!$token) return false;

  $post = get_page_by_path($token, OBJECT, 'shoplite_delivery');
  if (!$post || $post->post_type !== 'shoplite_delivery') return false;

  return $post;
}

/**
 * Consume one download attempt.
 */
function sl_delivery_consume($token) {
  $post = sl_delivery_get_by_token($token);
  if (!$post) return array('ok' => false, 'reason' => 'invalid_token');

  $post_id = (int)$post->ID;

  // lock to avoid double decrement
  $lock_key = 'sl_dl_lock_' . md5($token);
  if (get_transient($lock_key)) {
    return array('ok' => false, 'reason' => 'busy');
  }
  set_transient($lock_key, 1, 15);

  $status        = (string) get_post_meta($post_id, 'sl_status', true);
  $expires_at    = (int) get_post_meta($post_id, 'sl_expires_at', true);
  $remaining     = (int) get_post_meta($post_id, 'sl_remaining', true);
  $attachment_id = (int) get_post_meta($post_id, 'sl_attachment_id', true);

  // Expired?
  if ($expires_at > 0 && time() > $expires_at) {
    update_post_meta($post_id, 'sl_status', 'expired');
    delete_transient($lock_key);
    return array('ok' => false, 'reason' => 'expired');
  }

  // Status checks
  if ($status === 'used') {
    delete_transient($lock_key);
    return array('ok' => false, 'reason' => 'limit_reached');
  }
  if ($status === 'expired') {
    delete_transient($lock_key);
    return array('ok' => false, 'reason' => 'expired');
  }
  if ($status !== 'active') {
    delete_transient($lock_key);
    return array('ok' => false, 'reason' => 'inactive');
  }

  // Limit reached?
  if ($remaining <= 0) {
    update_post_meta($post_id, 'sl_status', 'used');
    delete_transient($lock_key);
    return array('ok' => false, 'reason' => 'limit_reached');
  }

  $remaining_new = max(0, $remaining - 1);
  update_post_meta($post_id, 'sl_remaining', $remaining_new);

  if ($remaining_new === 0) {
    update_post_meta($post_id, 'sl_status', 'used');
  }

  delete_transient($lock_key);

  return array(
    'ok' => true,
    'reason' => 'ok',
    'attachment_id' => $attachment_id,
    'post_id' => $post_id,
    'remaining' => $remaining_new,
  );
}

/**
 * Handle download endpoint.
 * Works with pretty URL (/shoplite-download/<token>/) AND querystring (?sl_download_token=<token>).
 */
function sl_delivery_template_redirect() {
  $token = get_query_var('sl_download_token');

  // Fallback: allow ?sl_download_token=...
  if (!$token && isset($_GET['sl_download_token'])) {
    $token = sanitize_text_field(wp_unslash($_GET['sl_download_token']));
  }

  if (!$token) return;

  $result = sl_delivery_consume($token);

  if (!$result['ok']) {
    status_header(403);
    nocache_headers();
    echo esc_html__( 'Download not available.', 'shoplite' ) . ' ' . esc_html($result['reason']);
    exit;
  }

  $attachment_id = (int)$result['attachment_id'];
  if ($attachment_id <= 0) {
    status_header(404);
    nocache_headers();
    echo esc_html__( 'File not found.', 'shoplite' );
    exit;
  }

  $file_path = get_attached_file($attachment_id);
  if (!$file_path || !file_exists($file_path)) {
    status_header(404);
    nocache_headers();
    echo esc_html__( 'File not found.', 'shoplite' );
    exit;
  }

  $mime = get_post_mime_type($attachment_id);
  if (!$mime) $mime = 'application/octet-stream';

  nocache_headers();
  header('Content-Type: ' . $mime);
  header('Content-Length: ' . filesize($file_path));
  header('Content-Disposition: attachment; filename="' . basename($file_path) . '"');
  header('X-Content-Type-Options: nosniff');

  while (ob_get_level()) { ob_end_clean(); }
  readfile($file_path);
  exit;
}
add_action('template_redirect', 'sl_delivery_template_redirect');

/**
 * Create digital delivery tokens when an order is paid/completed.
 *
 * Saves tokens in order meta: _shoplite_delivery_tokens
 * Adds a guard meta to avoid duplicates: _shoplite_delivery_tokens_created
 */
add_action('shoplite_order_paid', function($order_id) {

  $order_id = (int)$order_id;
  if ($order_id <= 0) return;

  // ✅ NEW: if stock module flagged issue, do NOT deliver digital goods
  if (get_post_meta($order_id, '_shoplite_stock_issue', true) === '1') {
    if (defined('WP_DEBUG') && WP_DEBUG && function_exists('error_log')) {
      error_log('[shoplite][DELIVERY] blocked due to stock issue. order=' . $order_id);
    }
    return;
  }

  // Guard: avoid duplicates if hook fires twice
  if (get_post_meta($order_id, '_shoplite_delivery_tokens_created', true) === '1') {
    return;
  }

  $raw = get_post_meta($order_id, '_shoplite_order_items', true);

  // Supports serialized array OR JSON OR array
  $items = maybe_unserialize($raw);
  if (is_string($items)) {
    $decoded = json_decode($items, true);
    if (is_array($decoded)) $items = $decoded;
  }
  if (!is_array($items)) $items = array();

  $customer_email = (string) get_post_meta($order_id, '_shoplite_customer_email', true);

  $created = array();

  foreach ($items as $key => $it) {
    if (!is_array($it)) continue;

    // Product id stored as 'id' OR inferred from key tbp_123
    $product_id = isset($it['id']) ? (int)$it['id'] : 0;
    if ($product_id <= 0 && is_string($key) && preg_match('/^tbp_(\d+)$/', (string)$key, $m)) {
      $product_id = (int)$m[1];
    }
    if ($product_id <= 0) continue;

    // Quantity: prefer 'qty', fallback to 'cantidad'
    $qty = 1;
    if (isset($it['qty'])) {
      $qty = (int)$it['qty'];
    } elseif (isset($it['cantidad'])) {
      $qty = (int)$it['cantidad'];
    }
    if ($qty < 1) $qty = 1;

    // Digital only = NOT physical (trust product meta)
    $is_physical = (get_post_meta($product_id, '_tb_is_physical', true) === '1');
    if ($is_physical) continue;

    // Must have an attachment configured
    $att_id = (int) get_post_meta($product_id, '_tb_download_attachment_id', true);
    if ($att_id <= 0) continue;

    // Policy: NO expiry + 5 downloads per token
    $expires_at = 0;
    $remaining_per_token = 5;

    // 1 token per unit
    for ($i = 0; $i < $qty; $i++) {
      $token = sl_delivery_create(array(
        'order_id'      => $order_id,
        'product_id'    => $product_id,
        'attachment_id' => $att_id,
        'expires_at'    => $expires_at,
        'remaining'     => $remaining_per_token,
        'buyer_email'   => $customer_email,
      ));

      if ($token) {
        $created[] = array(
          'product_id' => $product_id,
          'token'      => $token,
          'url'        => add_query_arg('sl_download_token', rawurlencode($token), home_url('/')),
        );
      }
    }
  }

  if (!empty($created)) {
    update_post_meta($order_id, '_shoplite_delivery_tokens', $created);
    update_post_meta($order_id, '_shoplite_delivery_tokens_created', '1');

    /**
     * ✅ Hook for Pro (or others) to sync tables / send emails.
     */
    do_action('shoplite_digital_tokens_created', $order_id, $created, $customer_email);
  }

}, 10, 1);
