<?php
// includes/class-shoplite-setup-wizard.php
if ( ! defined('ABSPATH') ) exit;

class Shoplite_Setup_Wizard {

  public static function init() {
    add_action('admin_menu', [__CLASS__, 'menu']);
  }

  public static function menu() {
    add_submenu_page(
      'shoplite',
      __('Setup wizard','shoplite'),
      __('Setup wizard','shoplite'),
      'manage_options',
      'shoplite-setup',
      [__CLASS__, 'render']
    );
  }

  /** Read current options */
  private static function get_mode() {
    $mode = get_option('shoplite_stripe_mode', 'test');
    return in_array($mode, ['test','live'], true) ? $mode : 'test';
  }

  private static function get_keys_for_mode($mode) {
    // 1) Combined option
    $all = get_option('shoplite_stripe_keys', []);
    if (isset($all[$mode]['pk'], $all[$mode]['sk'])) {
      return ['pk' => $all[$mode]['pk'], 'sk' => $all[$mode]['sk']];
    }
    // 2) Separate fields
    $map = [
      'test' => ['pk' => 'shoplite_stripe_pk_test', 'sk' => 'shoplite_stripe_sk_test'],
      'live' => ['pk' => 'shoplite_stripe_pk_live', 'sk' => 'shoplite_stripe_sk_live'],
    ];
    $pk = get_option($map[$mode]['pk'], '');
    $sk = get_option($map[$mode]['sk'], '');
    return ['pk' => $pk, 'sk' => $sk];
  }

  /** Wizard page */
  public static function render() {
    if ( ! current_user_can('manage_options') ) {
      return;
    }

    $mode = self::get_mode();
    $keys = self::get_keys_for_mode($mode);

    // ===== Auto-healing DEMO =====
    $demo_id = (int) get_option('shoplite_demo_product_id', 0);
    if ( $demo_id ) {
      $demo_post = get_post($demo_id);
      if ( ! $demo_post || get_post_status($demo_id) !== 'publish' ) {
        delete_option('shoplite_demo_product_id');
        $demo_id = 0;
      }
    }

    // ===== Auto-healing CHECKOUT (strict) =====
    $checkout_id  = (int) get_option('shoplite_checkout_page_id', 0);
    $checkout_url = '';
    if ( $checkout_id ) {
      $status = get_post_status($checkout_id);   // 'publish','draft','trash' or false
      if ( $status === 'publish' ) {
        $permalink = get_permalink($checkout_id);
        if ( $permalink ) {
          $checkout_url = $permalink;
        } else {
          delete_option('shoplite_checkout_page_id');
          $checkout_id = 0;
        }
      } else {
        delete_option('shoplite_checkout_page_id');
        $checkout_id = 0;
      }
    }

    // ===== Status flags =====
    $stripe_verified = get_option('shoplite_stripe_verified_'.$mode, []); // saved by verify_stripe()
    $stripe_ok       = is_array($stripe_verified) && !empty($stripe_verified['ok']);
    $demo_ok         = !empty($demo_id);
    $checkout_ok     = !empty($checkout_url);

    // ===== Quick styles =====
    echo '<style>
      .shoplite-badge{display:inline-block;padding:.15em .5em;border-radius:4px;font-size:12px;margin-left:.5em}
      .shoplite-badge.ok{background:#e6f4ea;color:#137333;border:1px solid #c8e6c9}
      .shoplite-badge.warn{background:#fff4e5;color:#8a6d3b;border:1px solid #ffe0b2}
      .step-title{display:flex;align-items:center;gap:.5em}
    </style>';
    ?>
    <div class="wrap">
      <h1><?php esc_html_e( 'Setup wizard', 'shoplite' ); ?></h1>
      <p><?php esc_html_e( 'Goal: Connect Stripe -> Create demo product -> Publish checkout page', 'shoplite' ); ?></p>

      <!-- Step 1 -->
      <h2 class="step-title">
        <?php
          echo ( $stripe_ok ? 'OK ' : 'X ' ) .
            esc_html__( 'Step 1 - Verify Stripe', 'shoplite' );
        ?>
        <?php if ( $stripe_ok ) : ?>
          <span class="shoplite-badge ok">
            <?php esc_html_e( 'Connected', 'shoplite' ); ?>
            <?php if ( ! empty( $stripe_verified['account']['email'] ) ) : ?>
              <?php echo ' - ' . esc_html( $stripe_verified['account']['email'] ); ?>
            <?php endif; ?>
          </span>
        <?php endif; ?>
      </h2>
      <p>
        <?php
          printf(
            esc_html__( 'Current mode: %s', 'shoplite' ),
            '<strong>' . esc_html( $mode ) . '</strong>'
          );
        ?>
        <br>
        <?php esc_html_e( 'Detected keys:', 'shoplite' ); ?>
        <code><?php echo esc_html( substr( $keys['pk'], 0, 12 ) ); ?>...</code> /
        <code><?php echo esc_html( substr( $keys['sk'], 0, 12 ) ); ?>...</code>
      </p>
      <p>
        <button class="button button-primary" id="slw-verify" <?php disabled( $stripe_ok ); ?>>
          <?php esc_html_e( 'Verify with Stripe', 'shoplite' ); ?>
        </button>
        <span id="slw-verify-msg">
          <?php echo $stripe_ok ? esc_html__( 'Stripe credentials are valid', 'shoplite' ) : ''; ?>
        </span>
        <?php if ( $stripe_ok && ! empty( $stripe_verified['time'] ) ) : ?>
          <span class="shoplite-badge ok">
            <?php
              echo esc_html__( 'Last verification: ', 'shoplite' ) .
                esc_html( $stripe_verified['time'] );
            ?>
          </span>
        <?php endif; ?>
      </p>

      <hr>

      <!-- Step 2 -->
      <h2 class="step-title">
        <?php
          echo ( $demo_ok ? 'OK ' : 'X ' ) .
            esc_html__( 'Step 2 - Create demo product', 'shoplite' );
        ?>
      </h2>
      <?php if ( $demo_ok ) : ?>
        <p>
          <?php
            printf(
              esc_html__( 'Existing demo product: ID %d', 'shoplite' ),
              $demo_id
            );
          ?>
          -
          <a href="<?php echo esc_url( get_edit_post_link( $demo_id ) ); ?>">
            <?php esc_html_e( 'Edit', 'shoplite' ); ?>
          </a>
          <button
            class="button button-link"
            id="slw-reset-demo"
            style="margin-left:8px"
          >
            <?php esc_html_e( 'Reset demo', 'shoplite' ); ?>
          </button>
          <span id="slw-reset-demo-msg" style="margin-left:6px;"></span>
        </p>
      <?php else : ?>
        <p>
          <button class="button" id="slw-create-demo">
            <?php esc_html_e( 'Create demo product', 'shoplite' ); ?>
          </button>
          <span id="slw-demo-msg"></span>
        </p>
      <?php endif; ?>

      <hr>

      <!-- Step 3 -->
      <h2 class="step-title">
        <?php
          echo ( $checkout_ok ? 'OK ' : 'X ' ) .
            esc_html__( 'Step 3 - Publish checkout page', 'shoplite' );
        ?>
      </h2>
      <?php if ( $checkout_ok ) : ?>
        <p>
          <?php esc_html_e( 'Checkout page ready:', 'shoplite' ); ?>
          <a href="<?php echo esc_url( $checkout_url ); ?>" target="_blank">
            <?php echo esc_html( $checkout_url ); ?>
          </a>
          <button
            class="button button-link"
            id="slw-reset-checkout"
            style="margin-left:8px"
          >
            <?php esc_html_e( 'Fix link', 'shoplite' ); ?>
          </button>
          <span id="slw-reset-msg" style="margin-left:6px;"></span>
        </p>
      <?php else : ?>
        <p>
          <button class="button" id="slw-publish-checkout">
            <?php esc_html_e( 'Create checkout page', 'shoplite' ); ?>
          </button>
          <span id="slw-checkout-msg"></span>
        </p>
      <?php endif; ?>

    </div>

    <script>
    (function(){
      const base  = '<?php echo esc_js( rest_url( 'shoplite/v1/setup/' ) ); ?>';
      const nonce = '<?php echo esc_js( wp_create_nonce( 'wp_rest' ) ); ?>';
      const mode  = '<?php echo esc_js( $mode ); ?>';

      async function post(url, body) {
        const r = await fetch(url, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': nonce
          },
          body: JSON.stringify(body || {})
        });
        const data = await r.json().catch(function(){ return {}; });
        if (!r.ok) {
          throw data;
        }
        return data;
      }

      // Step 1: Verify Stripe
      var verifyBtn = document.getElementById('slw-verify');
      if (verifyBtn) {
        verifyBtn.addEventListener('click', async function(ev) {
          var btn = ev.currentTarget;
          var msg = document.getElementById('slw-verify-msg');
          btn.disabled = true;
          msg.textContent = 'Verifying Stripe...';
          try {
            var res = await post(base + 'verify-stripe', { mode: mode });
            msg.textContent = 'Stripe credentials are valid';
            // Update badge on the fly
            var h2List = btn.closest('.wrap').querySelectorAll('h2.step-title');
            if (h2List.length > 0) {
              var h2 = h2List[0];
              if (!h2.querySelector('.shoplite-badge')) {
                var badge = document.createElement('span');
                badge.className = 'shoplite-badge ok';
                badge.textContent = 'Connected' + (res.account && res.account.email ? ' - ' + res.account.email : '');
                h2.appendChild(badge);
              }
            }
          } catch (e) {
            msg.textContent = 'Error verifying Stripe';
            btn.disabled = false;
          }
        });
      }

      // Step 2: Create demo
      var demoBtn = document.getElementById('slw-create-demo');
      if (demoBtn) {
        demoBtn.addEventListener('click', async function() {
          var msg = document.getElementById('slw-demo-msg');
          msg.textContent = 'Creating demo product...';
          try {
            var res = await post(base + 'create-demo');
            msg.textContent = 'Demo product created with ID ' + res.product_id;
            location.reload();
          } catch (e) {
            msg.textContent = 'Error creating demo product';
          }
        });
      }

      // Step 2: Reset demo
      var resetDemoBtn = document.getElementById('slw-reset-demo');
      if (resetDemoBtn) {
        resetDemoBtn.addEventListener('click', async function() {
          var el = document.getElementById('slw-reset-demo-msg');
          el.textContent = 'Resetting demo product...';
          try {
            await post(base + 'reset-demo');
            el.textContent = 'Demo product reset';
            location.reload();
          } catch (e) {
            el.textContent = 'Error resetting demo product';
          }
        });
      }

      // Step 3: Publish checkout
      var checkoutBtn = document.getElementById('slw-publish-checkout');
      if (checkoutBtn) {
        checkoutBtn.addEventListener('click', async function() {
          var msg = document.getElementById('slw-checkout-msg');
          msg.textContent = 'Creating checkout page...';
          try {
            var res = await post(base + 'publish-checkout');
            msg.innerHTML = 'Checkout page created: <a target="_blank" href="' + res.url + '">Open checkout</a>';
            location.reload();
          } catch (e) {
            msg.textContent = 'Error creating checkout page';
          }
        });
      }

      // Step 3: Reset checkout
      var resetCheckoutBtn = document.getElementById('slw-reset-checkout');
      if (resetCheckoutBtn) {
        resetCheckoutBtn.addEventListener('click', async function() {
          var msg = document.getElementById('slw-reset-msg');
          msg.textContent = 'Resetting checkout page...';
          try {
            await post(base + 'reset-checkout');
            msg.textContent = 'Checkout page reset';
            location.reload();
          } catch (e) {
            msg.textContent = 'Error resetting checkout page';
          }
        });
      }

    })();
    </script>
    <?php
  }
}

// Make sure to call ::init() from your main plugin file if you are not doing it yet.
// Shoplite_Setup_Wizard::init();


