<?php
// shoplite/includes/class-shoplite-orders.php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Shoplite_Orders {

    const POST_TYPE = 'shoplite_pedido';

    public static function init() {
        add_action( 'init', [ __CLASS__, 'register_cpt' ] );
    }

    public static function register_cpt() {

        $labels = [
            'name'          => __( 'Shoplite orders', 'shoplite' ),
            'singular_name' => __( 'Shoplite order', 'shoplite' ),
            'menu_name'     => __( 'Orders', 'shoplite' ),
        ];

        $args = [
            'labels'             => $labels,
            'public'             => false,
            'show_ui'            => true,
            'show_in_menu'       => 'shoplite', // attach to the Shoplite admin menu
            'capability_type'    => 'post',
            'hierarchical'       => false,
            'supports'           => [ 'title' ],
            'has_archive'        => false,
            'rewrite'            => false,
        ];

        register_post_type( self::POST_TYPE, $args );
    }

    /**
     * Create an order from checkout data.
     *
     * @param array $data {
     *   @type float  $total          Order total.
     *   @type string $currency       Currency code.
     *   @type string $email          Customer email.
     *   @type string $name           Customer name.
     *   @type array  $items          Order items.
     *   @type string $payment_method Payment method identifier.
     *   @type string $status         Order status.
     * }
     * @return int|WP_Error Order ID on success or WP_Error on failure.
     */
    public static function create_from_checkout( array $data ) {

        $order_title = sprintf(
            __( 'Shoplite order - %s', 'shoplite' ),
            current_time( 'Y-m-d H:i:s' )
        );

        $order_id = wp_insert_post( [
            'post_type'   => self::POST_TYPE,
            'post_status' => 'publish',
            'post_title'  => $order_title,
        ] );

        if ( is_wp_error( $order_id ) ) {
            return $order_id;
        }

        // Basic meta: used later in Analytics.
        $total          = (float) ( $data['total'] ?? 0 );
        $currency       = (string) ( $data['currency'] ?? 'EUR' );
        $status         = (string) ( $data['status'] ?? 'completed' );
        $email          = (string) ( $data['email'] ?? '' );
        $name           = (string) ( $data['name'] ?? '' );
        $payment_method = (string) ( $data['payment_method'] ?? '' );

        update_post_meta( $order_id, '_shoplite_order_total',    $total );
        update_post_meta( $order_id, '_shoplite_order_currency', $currency );
        update_post_meta( $order_id, '_shoplite_order_status',   $status );
        update_post_meta( $order_id, '_shoplite_customer_email', $email );
        update_post_meta( $order_id, '_shoplite_customer_name',  $name );
        update_post_meta( $order_id, '_shoplite_payment_method', $payment_method );

        // Order lines: always store as a serialized array.
        $items = $data['items'] ?? [];
        if ( ! is_array( $items ) ) {
            $items = [];
        }

        update_post_meta(
            $order_id,
            '_shoplite_order_items',
            maybe_serialize( $items )
        );

        /**
         * IMPORTANT:
         * Normalize the "order paid" event so all gateways (Stripe/PayPal/Bizum/...) trigger
         * the same downstream logic: digital deliveries, emails, analytics hooks, etc.
         *
         * We only fire it when status is completed (or similar paid statuses).
         */
        $paid_statuses = [ 'completed', 'paid', 'processing' ];
        if ( in_array( strtolower( $status ), $paid_statuses, true ) ) {
            /**
             * Fired when an order is paid/completed.
             *
             * @param int $order_id
             */
            do_action( 'shoplite_order_paid', (int) $order_id );
        }

        return $order_id;
    }
}
