<?php
// includes/class-shoplite-licensing.php
if ( ! defined('ABSPATH') ) exit;

class Shoplite_Licensing {

  public static function init(){
    add_action('admin_menu', [__CLASS__, 'menu']);
    add_action('admin_post_shoplite_activate_license', [__CLASS__, 'activate']);
    add_action('admin_post_shoplite_deactivate_license', [__CLASS__, 'deactivate']);
    add_action('admin_notices', [__CLASS__, 'admin_notice']);
  }

  public static function menu() {

    // If Shoplite Pro is active, do NOT add this menu.
    // License management is handled by Shoplite_Pro_Licensing.
    if ( function_exists('shoplite_is_pro') && shoplite_is_pro() ) {
      return;
    }

    add_submenu_page(
      TS_MENU_SLUG,
      __('License', 'shoplite'),
      __('License', 'shoplite'),
      'manage_options',
      'shoplite-license',
      [__CLASS__, 'render']
    );
  }

  public static function render(){
    if ( ! current_user_can('manage_options') ) return;

    $key    = (string) get_option('shoplite_license_key', '');
    $stat   = (string) get_option('shoplite_license_status', 'inactive');
    $meta   = json_decode((string) get_option('shoplite_license_meta','{}'), true);
    $meta   = is_array($meta) ? $meta : [];
    $masked = $key ? esc_html(substr($key, 0, 6) . '...' . substr($key, -4)) : '';

    ?>
    <div class="wrap">
      <h1><?php esc_html_e('Shoplite Pro License', 'shoplite'); ?></h1>
      <p><?php esc_html_e('Activate your license to receive updates and support.', 'shoplite'); ?></p>

      <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <?php wp_nonce_field('shoplite_license'); ?>
        <input type="hidden" name="action" value="shoplite_activate_license" />
        <table class="form-table">
          <tr>
            <th><label for="shoplite_key"><?php esc_html_e('License key', 'shoplite'); ?></label></th>
            <td>
              <input id="shoplite_key" type="text" name="key" class="regular-text"
                     value="<?php echo esc_attr($key); ?>"
                     placeholder="SLPRO-XXXX-XXXX-XXXX" />
              <?php if ( $key ) : ?>
                <p class="description">
                  <?php
                  printf(
                    esc_html__( 'Status: %s', 'shoplite' ),
                    '<strong>' . esc_html( $stat ) . '</strong>'
                  );

                  if ( ! empty( $meta['valid_until'] ) ) {
                    echo ' - ';
                    printf(
                      esc_html__( 'Valid until: %s', 'shoplite' ),
                      esc_html( (string) $meta['valid_until'] )
                    );
                  }

                  if ( $masked ) {
                    echo ' - ';
                    printf(
                      esc_html__( 'Key: %s', 'shoplite' ),
                      esc_html( $masked )
                    );
                  }
                  ?>
                </p>
              <?php endif; ?>
            </td>
          </tr>
        </table>

        <?php submit_button( __('Activate license', 'shoplite') ); ?>
      </form>

      <?php if ( $key && $stat === 'active' ) : ?>
        <hr>
        <h2><?php esc_html_e('Deactivate license on this site', 'shoplite'); ?></h2>
        <p><?php esc_html_e('Use this if you want to move your license to another domain.', 'shoplite'); ?></p>

        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="margin-top:1rem">
          <?php wp_nonce_field('shoplite_license'); ?>
          <input type="hidden" name="action" value="shoplite_deactivate_license" />
          <?php submit_button( __('Deactivate on this site', 'shoplite'), 'secondary' ); ?>
        </form>
      <?php endif; ?>
    </div>
    <?php
  }

  public static function activate(){
    if ( ! current_user_can('manage_options') ) wp_die('forbidden');
    check_admin_referer('shoplite_license');

    $posted_key = isset($_POST['key']) ? sanitize_text_field( wp_unslash($_POST['key']) ) : '';
    if ( $posted_key === '' ) {
      self::redirect(['sl_error' => 'missing_key']);
    }

    // If the user is trying a different key, reset meta to avoid stale UI.
    $current_key = (string) get_option('shoplite_license_key', '');
    if ( $current_key && $current_key !== $posted_key ) {
      delete_option('shoplite_license_meta');
      update_option('shoplite_license_status', 'inactive');
    }

    $site_token = (string) get_option('shoplite_site_token', '');
    if ( ! $site_token ) {
      $site_token = wp_generate_password(24, false, false);
      update_option('shoplite_site_token', $site_token, false);
    }

    $resp = wp_remote_post( SHOPLITE_LICENSE_API . 'activate', [
      'timeout' => 20,
      'body'    => [
        'key'        => $posted_key,
        'domain'     => home_url(),
        'site_token' => $site_token,
        'version'    => defined('SHOPLITE_VERSION') ? SHOPLITE_VERSION : '1.0.0',
      ],
    ]);

    if ( is_wp_error($resp) ) {
      self::redirect(['sl_error' => 'network']);
    }

    $code = (int) wp_remote_retrieve_response_code($resp);
    $body = (string) wp_remote_retrieve_body($resp);

    // Some servers may return HTML on error; treat as bad response.
    $data = json_decode($body, true);
    if ( $code !== 200 || ! is_array($data) ) {
      self::redirect(['sl_error' => 'bad_response']);
    }

    if ( ! empty($data['ok']) ) {
      update_option('shoplite_license_key', $posted_key);
      update_option('shoplite_license_status', 'active');
      update_option('shoplite_license_meta', wp_json_encode($data['meta'] ?? []), false);
      self::redirect(['sl_ok' => 'activated']);
    }

    $err = sanitize_key($data['error'] ?? 'invalid');

    // Normalize possible server error codes
    if ( $err === 'domain_limit' ) $err = 'sites_limit';

    self::redirect(['sl_error' => $err]);
  }

  public static function deactivate(){
    if ( ! current_user_can('manage_options') ) wp_die('forbidden');
    check_admin_referer('shoplite_license');

    $key   = (string) get_option('shoplite_license_key', '');
    $token = (string) get_option('shoplite_site_token', '');

    if ( ! $key ) {
      self::redirect(['sl_error' => 'missing_key']);
    }

    $ok = true;

    // If we have token, try to deactivate remotely.
    // If remote fails, we DO NOT pretend it worked.
    if ( $token ) {
      $resp = wp_remote_post( SHOPLITE_LICENSE_API . 'deactivate', [
        'timeout' => 20,
        'body'    => [
          'key'        => $key,
          'domain'     => home_url(),
          'site_token' => $token,
        ],
      ]);

      if ( is_wp_error($resp) ) {
        $ok = false;
      } else {
        $code = (int) wp_remote_retrieve_response_code($resp);
        $body = (string) wp_remote_retrieve_body($resp);
        $data = json_decode($body, true);

        if ( $code !== 200 || ! is_array($data) || empty($data['ok']) ) {
          $ok = false;
        }
      }
    }

    if ( ! $ok ) {
      self::redirect(['sl_error' => 'deactivate_failed']);
    }

    // Remote ok (or no token, local-only)
    delete_option('shoplite_license_key');
    update_option('shoplite_license_status', 'inactive');
    delete_option('shoplite_license_meta');

    self::redirect(['sl_ok' => 'deactivated']);
  }

  private static function redirect(array $args){
    $url = add_query_arg($args, admin_url('admin.php?page=shoplite-license'));
    shoplite_safe_redirect($url);
    exit;
  }

  public static function admin_notice(){
    if ( ! current_user_can('manage_options') ) return;
    $screen = get_current_screen();
    if ( ! $screen || $screen->id !== TS_MENU_SLUG . '_page_shoplite-license' ) return;

    if ( ! empty($_GET['sl_ok']) ) {
      $ok = sanitize_key($_GET['sl_ok']);
      $msg = ( $ok === 'activated' )
        ? __('License activated successfully.', 'shoplite')
        : __('License deactivated on this site.', 'shoplite');
      echo '<div class="notice notice-success"><p>' . esc_html($msg) . '</p></div>';
    }

    if ( ! empty($_GET['sl_error']) ) {
      $map = [
        'missing_key'       => __('Please enter a license key.', 'shoplite'),
        'network'           => __('Could not contact the license server.', 'shoplite'),
        'bad_response'      => __('Unexpected response from the licensing server.', 'shoplite'),
        'invalid'           => __('Invalid or unknown license key.', 'shoplite'),
        'expired'           => __('Your license has expired.', 'shoplite'),
        'sites_limit'       => __('You have reached the site limit for this license.', 'shoplite'),
        'deactivate_failed' => __('Could not deactivate the license on the server. Please try again.', 'shoplite'),
      ];
      $code = sanitize_key($_GET['sl_error']);
      $msg  = $map[$code] ?? __('Error while activating the license.', 'shoplite');
      echo '<div class="notice notice-error"><p>' . esc_html($msg) . '</p></div>';
    }
  }
}
