<?php
// includes/Rest/class-shoplite-rest-setup.php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Setup assistant REST endpoints for Shoplite.
 *
 * IMPORTANT:
 * - This class DOES NOT process the real checkout payment.
 * - It only exposes configuration endpoints: verify Stripe,
 *   create demo product, create/reset checkout page, etc.
 *
 * The creation of orders (shoplite_pedido) must be hooked
 * into the endpoint or callback that processes the real checkout payment,
 * not in this class.
 */
class Shoplite_REST_Setup {

    public static function init() {
        add_action(
            'rest_api_init',
            function() {

                register_rest_route(
                    'shoplite/v1',
                    '/setup/verify-stripe',
                    [
                        'methods'             => 'POST',
                        'callback'            => [ __CLASS__, 'verify_stripe' ],
                        'permission_callback' => [ __CLASS__, 'can_manage' ],
                    ]
                );

                register_rest_route(
                    'shoplite/v1',
                    '/setup/create-demo',
                    [
                        'methods'             => 'POST',
                        'callback'            => [ __CLASS__, 'create_demo' ],
                        'permission_callback' => [ __CLASS__, 'can_manage' ],
                    ]
                );

                register_rest_route(
                    'shoplite/v1',
                    '/setup/publish-checkout',
                    [
                        'methods'             => 'POST',
                        'callback'            => [ __CLASS__, 'publish_checkout' ],
                        'permission_callback' => [ __CLASS__, 'can_manage' ],
                    ]
                );

                register_rest_route(
                    'shoplite/v1',
                    '/setup/reset-checkout',
                    [
                        'methods'             => 'POST',
                        'callback'            => [ __CLASS__, 'reset_checkout' ],
                        'permission_callback' => [ __CLASS__, 'can_manage' ],
                    ]
                );

                register_rest_route(
                    'shoplite/v1',
                    '/setup/reset-demo',
                    [
                        'methods'             => 'POST',
                        'callback'            => [ __CLASS__, 'reset_demo' ],
                        'permission_callback' => [ __CLASS__, 'can_manage' ],
                    ]
                );

                // Optional: convert the demo product into a real product (remove is_demo).
                register_rest_route(
                    'shoplite/v1',
                    '/setup/demo-to-real',
                    [
                        'methods'             => 'POST',
                        'callback'            => [ __CLASS__, 'demo_to_real' ],
                        'permission_callback' => [ __CLASS__, 'can_manage' ],
                    ]
                );

            }
        );
    }

    /** Only administrators can manage setup. */
    public static function can_manage() {
        return current_user_can( 'manage_options' );
    }

    /** Read Stripe mode (test|live). */
    private static function get_mode() {
        $m = get_option( 'shoplite_stripe_mode', 'test' );
        return in_array( $m, [ 'test', 'live' ], true ) ? $m : 'test';
    }

    /**
     * Get Stripe keys (tolerant to different option names).
     *
     * @param string $mode Stripe mode (test|live).
     * @return array{pk:string,sk:string}
     */
    private static function get_keys_for_mode( $mode ) {
        $mode = in_array( $mode, [ 'test', 'live' ], true ) ? $mode : 'test';

        // 1) Combined option (shoplite_stripe_keys[mode][pk|sk]).
        $combo = get_option( 'shoplite_stripe_keys', [] );
        if (
            isset( $combo[ $mode ]['pk'], $combo[ $mode ]['sk'] ) &&
            $combo[ $mode ]['pk'] &&
            $combo[ $mode ]['sk']
        ) {
            return [
                'pk' => $combo[ $mode ]['pk'],
                'sk' => $combo[ $mode ]['sk'],
            ];
        }

        // 2) List of possible individual option names.
        $candidates = [
            'test' => [
                'pk' => [
                    'shoplite_stripe_pk_test',
                    'shoplite_pk_test',
                    'tb_stripe_pk_test',
                    'stripe_pk_test',
                ],
                'sk' => [
                    'shoplite_stripe_sk_test',
                    'shoplite_stripe_secret_test', // if you saved "secret" instead of "sk"
                    'shoplite_sk_test',
                    'tb_stripe_sk_test',
                    'stripe_sk_test',
                    'stripe_secret_test',
                ],
            ],
            'live' => [
                'pk' => [
                    'shoplite_stripe_pk_live',
                    'shoplite_pk_live',
                    'tb_stripe_pk_live',
                    'stripe_pk_live',
                ],
                'sk' => [
                    'shoplite_stripe_sk_live',
                    'shoplite_stripe_secret_live',
                    'shoplite_sk_live',
                    'tb_stripe_sk_live',
                    'stripe_sk_live',
                    'stripe_secret_live',
                ],
            ],
        ];

        $pk = '';
        foreach ( $candidates[ $mode ]['pk'] as $opt ) {
            $v = get_option( $opt, '' );
            if ( ! empty( $v ) ) {
                $pk = $v;
                break;
            }
        }

        $sk = '';
        foreach ( $candidates[ $mode ]['sk'] as $opt ) {
            $v = get_option( $opt, '' );
            if ( ! empty( $v ) ) {
                $sk = $v;
                break;
            }
        }

        // 3) Keys inside a settings array? (e.g. shoplite_pagos / shoplite_settings).
        if ( ! $pk || ! $sk ) {
            foreach ( [ 'shoplite_pagos', 'shoplite_settings', 'shoplite_options' ] as $bag ) {
                $opts = get_option( $bag, [] );
                if ( is_array( $opts ) && $opts ) {
                    if ( ! $pk ) {
                        $pk = $opts[ "stripe_pk_{$mode}" ]
                            ?? $opts[ "pk_{$mode}" ]
                            ?? $opts[ "shoplite_stripe_pk_{$mode}" ]
                            ?? $pk;
                    }
                    if ( ! $sk ) {
                        $sk = $opts[ "stripe_sk_{$mode}" ]
                            ?? $opts[ "stripe_secret_{$mode}" ]
                            ?? $opts[ "sk_{$mode}" ]
                            ?? $opts[ "shoplite_stripe_sk_{$mode}" ]
                            ?? $sk;
                    }
                }
            }
        }

        // 4) Last resort: scan all options and take the first value that matches by prefix.
        if ( ! $pk || ! $sk ) {
            $all = wp_load_alloptions();
            if ( is_array( $all ) ) {
                $pk_prefix = ( 'test' === $mode ) ? 'pk_test_' : 'pk_live_';
                $sk_prefix = ( 'test' === $mode ) ? 'sk_test_' : 'sk_live_';
                foreach ( $all as $name => $value ) {
                    if ( ! $pk && is_string( $value ) && 0 === strpos( $value, $pk_prefix ) ) {
                        $pk = $value;
                    }
                    if ( ! $sk && is_string( $value ) && 0 === strpos( $value, $sk_prefix ) ) {
                        $sk = $value;
                    }
                    if ( $pk && $sk ) {
                        break;
                    }
                }
            }
        }

        return [
            'pk' => (string) $pk,
            'sk' => (string) $sk,
        ];
    }

    /** POST /setup/verify-stripe */
    public static function verify_stripe( WP_REST_Request $r ) {
        $mode = self::get_mode();
        $keys = self::get_keys_for_mode( $mode );

        if ( empty( $keys['sk'] ) ) {
            return new WP_Error(
                'stripe_missing',
                __( 'Missing Stripe keys', 'shoplite' ),
                [ 'status' => 400 ]
            );
        }

        $resp = wp_remote_get(
            'https://api.stripe.com/v1/account',
            [
                'headers' => [
                    'Authorization' => 'Bearer ' . $keys['sk'],
                ],
                'timeout' => 15,
            ]
        );

        if ( is_wp_error( $resp ) ) {
            return new WP_Error(
                'stripe_network',
                __( 'Could not contact Stripe', 'shoplite' ),
                [ 'status' => 400 ]
            );
        }

        if ( 200 !== wp_remote_retrieve_response_code( $resp ) ) {
            return new WP_Error(
                'stripe_invalid',
                __( 'Invalid Stripe credentials', 'shoplite' ),
                [ 'status' => 400 ]
            );
        }

        $body = json_decode( wp_remote_retrieve_body( $resp ), true );

        // Persist verification state per mode.
        $meta = [
            'ok'   => true,
            'mode' => $mode,
            'time' => current_time( 'mysql' ),
            'account' => [
                'id'    => $body['id'] ?? null,
                'email' => $body['email'] ?? null,
            ],
        ];

        update_option( 'shoplite_stripe_verified_' . $mode, $meta );

        return [
            'ok'      => true,
            'mode'    => $mode,
            'account' => $meta['account'],
        ];
    }

    /** POST /setup/create-demo (idempotent). */
    public static function create_demo( WP_REST_Request $r ) {
        $post_type = 'shoplite_producto'; // your real CPT.

        // Reuse if a demo product is already registered.
        $existing_id = (int) get_option( 'shoplite_demo_product_id', 0 );
        if ( $existing_id && get_post( $existing_id ) ) {
            $edit = get_edit_post_link( $existing_id, '' )
                ?: admin_url( 'post.php?post=' . $existing_id . '&action=edit' );

            return [
                'ok'         => true,
                'reused'     => true,
                'product_id' => $existing_id,
                'edit_link'  => $edit,
            ];
        }

        // Create the demo product with basic data.
        $pid = wp_insert_post(
            [
                'post_type'    => $post_type,
                'post_status'  => 'publish',
                'post_title'   => __( 'Demo product', 'shoplite' ),
                'post_name'    => 'demo-product',
                'post_content' => __( 'Example product to test the checkout.', 'shoplite' ),
                'meta_input'   => [
                    'price'   => '1.00',     // adjust to your real price meta.
                    'sku'     => 'DEMO-001', // if you use SKU in meta.
                    'type'    => 'digital',
                    'is_demo' => 1,          // flag to hide it from the catalog.
                ],
            ]
        );

        if ( is_wp_error( $pid ) || ! $pid ) {
            return new WP_Error(
                'demo_fail',
                __( 'Could not create demo product', 'shoplite' ),
                [ 'status' => 500 ]
            );
        }

        update_option( 'shoplite_demo_product_id', (int) $pid );

        $edit_link = get_edit_post_link( $pid, '' )
            ?: admin_url( 'post.php?post=' . $pid . '&action=edit' );

        return [
            'ok'         => true,
            'created'    => true,
            'product_id' => (int) $pid,
            'edit_link'  => $edit_link,
        ];
    }

    /** POST /setup/publish-checkout (idempotent). */
    public static function publish_checkout( WP_REST_Request $r ) {
        // Canonical and compatible shortcodes.
        $canonical_sc = '[tienda_carrito]';
        $accepted     = [ $canonical_sc, '[shoplite_checkout]' ];

        // 1) Option already points to a valid page.
        $opt_id = (int) get_option( 'shoplite_checkout_page_id', 0 );
        if ( $opt_id && 'publish' === get_post_status( $opt_id ) ) {
            return [
                'ok'      => true,
                'reused'  => true,
                'page_id' => $opt_id,
                'url'     => get_permalink( $opt_id ),
                'message' => __( 'Existing checkout page reused (from option).', 'shoplite' ),
            ];
        }

        // 2) Look for an existing page that contains the shortcode.
        $found_id = 0;
        $q        = new WP_Query(
            [
                'post_type'      => 'page',
                'post_status'    => [ 'publish', 'draft', 'trash' ],
                's'              => 'checkout',
                'posts_per_page' => 20,
                'fields'         => 'ids',
            ]
        );

        if ( $q->have_posts() ) {
            foreach ( $q->posts as $pid ) {
                $content = get_post_field( 'post_content', $pid );
                foreach ( $accepted as $sc ) {
                    if ( false !== strpos( $content, $sc ) ) {
                        $found_id = (int) $pid;
                        break 2;
                    }
                }
            }
        }

        if ( $found_id ) {
            if ( 'publish' !== get_post_status( $found_id ) ) {
                wp_update_post(
                    [
                        'ID'          => $found_id,
                        'post_status' => 'publish',
                    ]
                );
            }

            update_option( 'shoplite_checkout_page_id', $found_id );

            return [
                'ok'      => true,
                'reused'  => true,
                'page_id' => $found_id,
                'url'     => get_permalink( $found_id ),
                'message' => __( 'Existing checkout page reused (by content).', 'shoplite' ),
            ];
        }

        // 3) Create a new checkout page.
        $page_id = wp_insert_post(
            [
                'post_type'    => 'page',
                'post_status'  => 'publish',
                'post_title'   => __( 'Checkout', 'shoplite' ),
                'post_content' => $canonical_sc,
            ]
        );

        if ( is_wp_error( $page_id ) || ! $page_id ) {
            return new WP_Error(
                'page_fail',
                __( 'Could not create checkout page', 'shoplite' ),
                [ 'status' => 500 ]
            );
        }

        update_option( 'shoplite_checkout_page_id', (int) $page_id );

        return [
            'ok'      => true,
            'created' => true,
            'page_id' => (int) $page_id,
            'url'     => get_permalink( $page_id ),
            'message' => __( 'New checkout page created.', 'shoplite' ),
        ];
    }

    /** POST /setup/reset-checkout */
    public static function reset_checkout( WP_REST_Request $r ) {
        delete_option( 'shoplite_checkout_page_id' );

        return [
            'ok' => true,
        ];
    }

    /** POST /setup/reset-demo */
    public static function reset_demo( WP_REST_Request $r ) {
        delete_option( 'shoplite_demo_product_id' );

        return [
            'ok' => true,
        ];
    }

    /** POST /setup/demo-to-real (optional). */
    public static function demo_to_real( WP_REST_Request $r ) {
        $id = (int) get_option( 'shoplite_demo_product_id', 0 );

        if ( ! $id || ! get_post( $id ) ) {
            return new WP_Error(
                'no_demo',
                __( 'No demo product found', 'shoplite' ),
                [ 'status' => 400 ]
            );
        }

        delete_post_meta( $id, 'is_demo' );

        return [
            'ok'         => true,
            'product_id' => $id,
            'edit_link'  => get_edit_post_link( $id, '' ),
        ];
    }
}


