<?php
/*
Plugin Name: Talariasoft Shoplite Pro
Plugin URI: https://talariasoft.com/shoplite-pro
Description: Funcionalidades avanzadas para Shoplite (cupones, informes, licencias, etc.).
Author: Talariasoft
Version: 1.0.4
Requires Plugins: shoplite
Text Domain: shoplite-pro
Domain Path: /languages
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Versión interna (útil para migraciones / db_version)
if ( ! defined( 'SHOPLITE_PRO_VERSION' ) ) {
    define( 'SHOPLITE_PRO_VERSION', '1.0.4' );
}

/**
 * Carga de traducciones (i18n) para Shoplite Pro.
 */
function shoplite_pro_load_textdomain() {
    load_plugin_textdomain(
        'shoplite-pro',
        false,
        dirname( plugin_basename( __FILE__ ) ) . '/languages/'
    );
}
add_action( 'plugins_loaded', 'shoplite_pro_load_textdomain' );

// Marca global: el sitio está en modo Pro.
if ( ! defined( 'SHOPLITE_PRO' ) ) {
    define( 'SHOPLITE_PRO', true );
}

// Constantes de ruta/URL del plugin Pro (para require_once limpios).
if ( ! defined( 'SHOPLITE_PRO_PLUGIN_FILE' ) ) {
    define( 'SHOPLITE_PRO_PLUGIN_FILE', __FILE__ );
}
if ( ! defined( 'SHOPLITE_PRO_DIR' ) ) {
    define( 'SHOPLITE_PRO_DIR', plugin_dir_path( __FILE__ ) );
}
if ( ! defined( 'SHOPLITE_PRO_URL' ) ) {
    define( 'SHOPLITE_PRO_URL', plugin_dir_url( __FILE__ ) );
}

/**
 * Installer / migraciones DB (incluye deliveries).
 * Nota: debe existir includes/install.php
 */
$shoplite_pro_install_file = SHOPLITE_PRO_DIR . 'includes/install.php';
if ( file_exists( $shoplite_pro_install_file ) ) {
    require_once $shoplite_pro_install_file;
}

/**
 * Activación de Shoplite Pro:
 *  - Comprueba que el core está activo.
 *  - Crea/actualiza tablas (cupones, usos, abandonados, deliveries, etc.).
 */
register_activation_hook( __FILE__, 'shoplite_pro_on_activate' );

function shoplite_pro_on_activate() {
    // 1) Asegurar que el core (gratuito) está activo
    if ( ! function_exists( 'shoplite_is_pro' ) ) {
        deactivate_plugins( plugin_basename( __FILE__ ) );
        wp_die(
            'Debes activar <strong>Talariasoft Shoplite</strong> (versión gratuita) antes de activar Shoplite Pro.',
            'Shoplite Pro',
            [ 'back_link' => true ]
        );
    }

    // 2) Crear/actualizar tablas desde el installer central
    if ( function_exists( 'shoplite_pro_install' ) ) {
        shoplite_pro_install();
        return;
    }

    /**
     * Fallback (por si falta includes/install.php):
     * mantenemos tu instalación antigua para no romper activaciones.
     */
    global $wpdb;
    require_once ABSPATH . 'wp-admin/includes/upgrade.php';

    $charset_collate = $wpdb->get_charset_collate();

    $table_coupons   = $wpdb->prefix . 'shoplite_coupons';
    $table_uses      = $wpdb->prefix . 'shoplite_coupon_uses';
    $table_abandoned = $wpdb->prefix . 'shoplite_abandoned_carts';

    // Tabla de cupones
    $sql_coupons = "CREATE TABLE $table_coupons (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        internal_name VARCHAR(190) NOT NULL,
        code VARCHAR(64) NOT NULL,
        type VARCHAR(20) NOT NULL,
        amount DECIMAL(10,2) NOT NULL DEFAULT 0,
        starts_at DATETIME NULL,
        ends_at DATETIME NULL,
        max_uses INT NULL,
        max_uses_per_user INT NULL,
        min_cart_total DECIMAL(10,2) NULL,
        product_ids TEXT NULL,
        first_purchase TINYINT(1) NOT NULL DEFAULT 0,
        status VARCHAR(20) NOT NULL DEFAULT 'active',
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE KEY code_unique (code)
    ) $charset_collate;";

    // Tabla de usos de cupones
    $sql_uses = "CREATE TABLE $table_uses (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        coupon_id BIGINT UNSIGNED NOT NULL,
        user_id BIGINT UNSIGNED NULL,
        order_id BIGINT UNSIGNED NULL,
        used_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY coupon_id_idx (coupon_id),
        KEY user_id_idx (user_id)
    ) $charset_collate;";

    // Tabla de carritos abandonados
    $sql_abandoned = "CREATE TABLE $table_abandoned (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        session_id VARCHAR(191) NOT NULL,
        user_id BIGINT UNSIGNED NULL,
        email VARCHAR(190) NULL,
        cart_json LONGTEXT NULL,
        cart_total DECIMAL(10,2) NOT NULL DEFAULT 0,
        last_update DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        email_sent TINYINT(1) NOT NULL DEFAULT 0,
        email_sent_at DATETIME NULL,
        PRIMARY KEY (id),
        KEY session_id_idx (session_id),
        KEY user_idx (user_id),
        KEY email_idx (email)
    ) $charset_collate;";

    dbDelta( $sql_coupons );
    dbDelta( $sql_uses );
    dbDelta( $sql_abandoned );
}

/**
 * Require seguro (evita fatal si falta un archivo durante despliegues/updates).
 */
if ( ! function_exists( 'shoplite_pro_require_once' ) ) {
    function shoplite_pro_require_once( $path, $label = '' ) {
        if ( file_exists( $path ) ) {
            require_once $path;
            return true;
        }

        if ( function_exists( 'error_log' ) ) {
            error_log( '[shoplite-pro] Missing file: ' . $path . ( $label ? ' (' . $label . ')' : '' ) );
        }

        return false;
    }
}

/**
 * Informes generales de ventas (tarjetas, totales, etc.).
 * Esta clase ya registra su propio submenú "Informes de ventas (Pro)".
 */
$sales_file = SHOPLITE_PRO_DIR . 'includes/class-shoplite-pro-sales.php';
if ( shoplite_pro_require_once( $sales_file, 'sales' ) && class_exists( 'Shoplite_Pro_Sales' ) ) {
    Shoplite_Pro_Sales::init();
}

/**
 * Evolución temporal de ventas (gráficas día/semana/mes).
 * Registramos aquí el submenú "Evolución temporal" para no duplicarlo.
 */
$evo_file = SHOPLITE_PRO_DIR . 'includes/class-shoplite-pro-evolution.php';
if ( shoplite_pro_require_once( $evo_file, 'evolution' ) && class_exists( 'Shoplite_Pro_Evolution' ) ) {
    Shoplite_Pro_Evolution::init();
}

/**
 * Helper de seguridad: recrea la tabla de carritos abandonados si
 * por cualquier motivo ha desaparecido después de la activación.
 */
function shoplite_pro_maybe_install_abandoned_carts_table() {
    global $wpdb;

    $table_abandoned = $wpdb->prefix . 'shoplite_abandoned_carts';

    $exists = $wpdb->get_var(
        $wpdb->prepare( "SHOW TABLES LIKE %s", $table_abandoned )
    );

    if ( $exists === $table_abandoned ) {
        return;
    }

    /**
     * ✅ Preferido: usar el installer central (evita duplicar SQL).
     * Esto requiere que includes/install.php esté cargado y tenga esta función.
     */
    if ( function_exists( 'shoplite_pro_install_abandoned_carts_table' ) ) {
        shoplite_pro_install_abandoned_carts_table();

        if ( defined( 'WP_DEBUG' ) && WP_DEBUG && function_exists( 'error_log' ) ) {
            error_log( '[shoplite][AC] Tabla shoplite_abandoned_carts recreada desde installer.' );
        }
        return;
    }

    /**
     * Fallback: si por alguna razón no existe el installer,
     * recreamos aquí la tabla como antes.
     */
    if ( ! function_exists( 'dbDelta' ) ) {
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    }

    $charset_collate = $wpdb->get_charset_collate();

    $sql_abandoned = "CREATE TABLE $table_abandoned (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        session_id VARCHAR(191) NOT NULL,
        user_id BIGINT UNSIGNED NULL,
        email VARCHAR(190) NULL,
        cart_json LONGTEXT NULL,
        cart_total DECIMAL(10,2) NOT NULL DEFAULT 0,
        last_update DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        email_sent TINYINT(1) NOT NULL DEFAULT 0,
        email_sent_at DATETIME NULL,
        PRIMARY KEY (id),
        KEY session_id_idx (session_id),
        KEY user_idx (user_id),
        KEY email_idx (email)
    ) $charset_collate;";

    dbDelta( $sql_abandoned );

    if ( defined( 'WP_DEBUG' ) && WP_DEBUG && function_exists( 'error_log' ) ) {
        error_log( '[shoplite][AC] Tabla shoplite_abandoned_carts recreada automáticamente (fallback).' );
    }
}



/**
 * Helper de seguridad: recrea la tabla de entregas digitales si
 * por cualquier motivo ha desaparecido después de la activación.
 */
function shoplite_pro_maybe_install_deliveries_table() {
    global $wpdb;

    $table_deliveries = $wpdb->prefix . 'shoplite_deliveries';

    $exists = $wpdb->get_var(
        $wpdb->prepare( "SHOW TABLES LIKE %s", $table_deliveries )
    );

    if ( $exists === $table_deliveries ) {
        return;
    }

    // ✅ Preferido: usar el installer central
    if ( function_exists( 'shoplite_pro_install_deliveries_table' ) ) {
        shoplite_pro_install_deliveries_table();

        if ( defined( 'WP_DEBUG' ) && WP_DEBUG && function_exists( 'error_log' ) ) {
            error_log( '[shoplite][deliveries] Tabla shoplite_deliveries recreada desde installer.' );
        }
        return;
    }

    // Fallback
    if ( ! function_exists( 'dbDelta' ) ) {
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    }

    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_deliveries (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        order_id BIGINT UNSIGNED NOT NULL,
        product_id BIGINT UNSIGNED NOT NULL,
        customer_email VARCHAR(190) NOT NULL,
        delivery_type VARCHAR(30) NOT NULL,
        status VARCHAR(20) NOT NULL DEFAULT 'pending',
        token VARCHAR(64) NULL,
        delivery_data LONGTEXT NULL,
        downloads_used INT UNSIGNED NOT NULL DEFAULT 0,
        downloads_limit INT UNSIGNED NULL,
        expires_at DATETIME NULL,
        created_at DATETIME NOT NULL,
        updated_at DATETIME NOT NULL,
        PRIMARY KEY (id),
        KEY order_id (order_id),
        KEY product_id (product_id),
        KEY customer_email (customer_email),
        KEY status (status),
        KEY token (token)
    ) $charset_collate;";

    dbDelta( $sql );

    if ( defined( 'WP_DEBUG' ) && WP_DEBUG && function_exists( 'error_log' ) ) {
        error_log( '[shoplite][deliveries] Tabla shoplite_deliveries recreada automáticamente (fallback).' );
    }
}


add_action( 'admin_init', 'shoplite_pro_maybe_install_abandoned_carts_table' );
add_action( 'wp_loaded',  'shoplite_pro_maybe_install_abandoned_carts_table' );

add_action( 'admin_init', 'shoplite_pro_maybe_install_deliveries_table' );
add_action( 'wp_loaded',  'shoplite_pro_maybe_install_deliveries_table' );


/**
 * Bootstrap de funcionalidades Pro cuando el core ha cargado.
 */
add_action( 'shoplite_loaded', 'shoplite_pro_boot' );

function shoplite_pro_boot() {

    // Evita doble bootstrap si el hook se dispara más de una vez
    static $booted = false;
    if ( $booted ) {
        return;
    }
    $booted = true;

    // Rutas de clases/archivos Pro adicionales
    $files = [
        'coupons'           => SHOPLITE_PRO_DIR . 'includes/class-shoplite-pro-coupons.php',
        'licensing'         => SHOPLITE_PRO_DIR . 'includes/class-shoplite-pro-licensing.php',
        'updater'           => SHOPLITE_PRO_DIR . 'includes/class-shoplite-pro-updater.php',
        'ac_ui'             => SHOPLITE_PRO_DIR . 'includes/class-shoplite-pro-abandoned-carts.php',
        'ac_engine'         => SHOPLITE_PRO_DIR . 'includes/class-shoplite-pro-abandoned-engine.php',
        'dashboard'         => SHOPLITE_PRO_DIR . 'includes/class-shoplite-pro-dashboard.php',

        // ✅ Email de descargas digitales (Pro)
        'digital_email'     => SHOPLITE_PRO_DIR . 'includes/class-shoplite-pro-digital-delivery-email.php',

        // ✅ NUEVO: Deliveries (Pro) - Actions (Resend/Revoke/Regen token)
        'deliveries_actions'=> SHOPLITE_PRO_DIR . 'includes/pro/deliveries-actions.php',

        // ✅ Deliveries (Pro) - Admin (menú/pantalla)
        'deliveries_admin'  => SHOPLITE_PRO_DIR . 'includes/pro/deliveries-admin.php',
        'deliveries_sync'  => SHOPLITE_PRO_DIR . 'includes/pro/deliveries-sync.php',

        // (Opcional) cuando lo crees: modelo CRUD (ahora lo carga deliveries-admin.php directamente)
        // 'deliveries_model'  => SHOPLITE_PRO_DIR . 'includes/pro/deliveries-model.php',

        // OJO: Evolution y Sales ya se cargan arriba, no se listan aquí
    ];

    foreach ( $files as $key => $path ) {
        if ( file_exists( $path ) ) {
            require_once $path;
        } else {
            if ( is_admin() && current_user_can( 'manage_options' ) && function_exists( 'error_log' ) ) {
                error_log( '[shoplite-pro] Falta el archivo: ' . $path );
            }
        }
    }

    // Inicializar solo las clases que existan
    if ( class_exists( 'Shoplite_Pro_Coupons' ) ) {
        Shoplite_Pro_Coupons::init();
    }
    if ( class_exists( 'Shoplite_Pro_Licensing' ) ) {
        Shoplite_Pro_Licensing::init();
    }
    if ( class_exists( 'Shoplite_Pro_Updater' ) ) {
        Shoplite_Pro_Updater::init();
    }
    if ( class_exists( 'Shoplite_Pro_Abandoned_Carts' ) ) {
        Shoplite_Pro_Abandoned_Carts::init();
    }
    if ( class_exists( 'Shoplite_Pro_Abandoned_Engine' ) ) {
        Shoplite_Pro_Abandoned_Engine::init();
    }
    if ( class_exists( 'Shoplite_Pro_Dashboard' ) ) {
        Shoplite_Pro_Dashboard::init();
    }

    // Aviso (lo puedes comentar/quitar cuando te apetezca)
    add_action( 'admin_notices', function () {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        echo '<div class="notice notice-success is-dismissible"><p>'
             . esc_html__( 'Shoplite Pro está activo. Funciones premium disponibles.', 'shoplite-pro' )
             . '</p></div>';
    } );
}

// ✅ DEBUG temporal: mostrar meta de pedidos con ?shoplite_debug_meta=1
if ( is_admin() ) {
    $debug_file = SHOPLITE_PRO_DIR . 'includes/pro/debug-order-meta.php';
    if ( file_exists( $debug_file ) ) {
        require_once $debug_file;
    }
}




/**
 * Estilos + JS de administración para las pantallas de Shoplite / Shoplite Pro.
 */
function shoplite_pro_admin_assets( $hook ) {

    if ( ! is_admin() ) {
        return;
    }

    // Detectamos si estamos en una pantalla relacionada con Shoplite
    $is_shoplite_screen = false;

    if ( function_exists( 'get_current_screen' ) ) {
        $screen = get_current_screen();
        if ( $screen && strpos( $screen->id, 'shoplite' ) !== false ) {
            $is_shoplite_screen = true;
        }
    }

    // Fallback por si aún no hay screen: mirar el parámetro ?page=
    if ( ! $is_shoplite_screen ) {
        if ( ! empty( $_GET['page'] ) && strpos( sanitize_text_field( wp_unslash( $_GET['page'] ) ), 'shoplite' ) !== false ) {
            $is_shoplite_screen = true;
        }
    }

    if ( ! $is_shoplite_screen ) {
        return;
    }

    // ========== CSS Admin Pro ==========
    wp_enqueue_style(
        'shoplite-pro-admin',
        SHOPLITE_PRO_URL . 'assets/css/shoplite-pro-admin.css',
        [],
        '1.0.0'
    );

    // ========== JS Gate de licencia ==========
    $handle = 'shoplite-pro-license-gate';

    wp_enqueue_script(
        $handle,
        plugins_url( 'assets/js/shoplite-pro-license-gate.js', __FILE__ ),
        [ 'jquery' ],
        '1.0.0',
        true
    );

    // Estado de la licencia
    $license_active = false;
    if ( class_exists( 'Shoplite_Pro_Licensing' ) && method_exists( 'Shoplite_Pro_Licensing', 'is_active' ) ) {
        $license_active = Shoplite_Pro_Licensing::is_active();
    }

    wp_localize_script(
        $handle,
        'ShopliteProLicenseData',
        [
            'license_active' => (bool) $license_active,
            'license_url'    => admin_url( 'admin.php?page=shoplite-pro-license' ),
            'i18n'           => [
                'title'       => __( 'Pro feature', 'shoplite-pro' ),
                'message'     => __( 'This feature is part of Shoplite Pro. Activate your license to use it.', 'shoplite-pro' ),
                'go_to_page'  => __( 'Go to the license page', 'shoplite-pro' ),
                'maybe_later' => __( 'Maybe later', 'shoplite-pro' ),
            ],
        ]
    );
}
add_action( 'admin_enqueue_scripts', 'shoplite_pro_admin_assets' );

/**
 * Enlace rápido "Carritos abandonados" en la lista de plugins (fila de Shoplite Pro)
 */
add_filter(
    'plugin_action_links_' . plugin_basename( __FILE__ ),
    function ( $links ) {
        $url = admin_url( 'options-general.php?page=shoplite_pro_abandoned_carts' );

        $links[] = '<a href="' . esc_url( $url ) . '">'
                 . esc_html__( 'Abandoned carts', 'shoplite-pro' )
                 . '</a>';

        return $links;
    }
);
