<?php
if ( ! defined( 'ABSPATH' ) ) exit;

function shoplite_deliveries_table() {
    global $wpdb;
    return $wpdb->prefix . 'shoplite_deliveries';
}

function shoplite_delivery_count( $where_sql = '1=1', $params = [] ) {
    global $wpdb;
    $table = shoplite_deliveries_table();

    if ( ! empty( $params ) ) {
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        return (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM $table WHERE $where_sql",
            $params
        ) );
    }

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.PreparedSQL.NotPrepared
    return (int) $wpdb->get_var( "SELECT COUNT(*) FROM $table WHERE $where_sql" );
}

function shoplite_delivery_list( $args = [] ) {
    global $wpdb;
    $table = shoplite_deliveries_table();

    $defaults = [
        'paged'    => 1,
        'per_page' => 20,
        'status'   => '',
        'search'   => '',
        'orderby'  => 'id',
        'order'    => 'DESC',
    ];
    $args = array_merge( $defaults, is_array( $args ) ? $args : [] );

    $paged    = max( 1, (int) $args['paged'] );
    $per_page = max( 1, min( 100, (int) $args['per_page'] ) );
    $offset   = ( $paged - 1 ) * $per_page;

    $where  = "1=1";
    $params = [];

    if ( $args['status'] !== '' ) {
        $where .= " AND status = %s";
        $params[] = (string) $args['status'];
    }

    if ( $args['search'] !== '' ) {
        $like = '%' . $wpdb->esc_like( (string) $args['search'] ) . '%';
        $where .= " AND (customer_email LIKE %s OR CAST(order_id AS CHAR) LIKE %s OR CAST(product_id AS CHAR) LIKE %s)";
        $params[] = $like;
        $params[] = $like;
        $params[] = $like;
    }

    $allowed_orderby = [ 'id','order_id','product_id','customer_email','status','created_at','expires_at' ];
    $orderby = in_array( $args['orderby'], $allowed_orderby, true ) ? $args['orderby'] : 'id';

    $order = strtoupper( (string) $args['order'] );
    $order = in_array( $order, [ 'ASC','DESC' ], true ) ? $order : 'DESC';

    // Items
    $sql = "SELECT * FROM $table WHERE $where ORDER BY $orderby $order LIMIT %d OFFSET %d";
    $params_items = array_merge( $params, [ $per_page, $offset ] );

    if ( ! empty( $params ) ) {
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $items = $wpdb->get_results( $wpdb->prepare( $sql, $params_items ), ARRAY_A );
    } else {
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        $items = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table WHERE 1=1 ORDER BY $orderby $order LIMIT %d OFFSET %d",
                $per_page,
                $offset
            ),
            ARRAY_A
        );
    }

    // Total
    $total = shoplite_delivery_count( $where, $params );

    return [ is_array( $items ) ? $items : [], (int) $total ];
}

/**
 * Get one delivery row by ID
 */
function shoplite_delivery_get( $id ) {
    global $wpdb;

    $id = (int) $id;
    if ( $id <= 0 ) {
        return null;
    }

    $table = shoplite_deliveries_table();

    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
    $row = $wpdb->get_row(
        $wpdb->prepare( "SELECT * FROM $table WHERE id = %d", $id ),
        ARRAY_A
    );

    return is_array( $row ) ? $row : null;
}

/**
 * Update a delivery row by ID (safe allowed columns)
 */
function shoplite_delivery_update( $id, $data ) {
    global $wpdb;

    $id = (int) $id;
    if ( $id <= 0 ) return false;
    if ( ! is_array( $data ) || empty( $data ) ) return false;

    $table = shoplite_deliveries_table();

    // Allowed columns only
    $allowed = [
        'order_id', 'product_id', 'customer_email', 'delivery_type',
        'status', 'token', 'delivery_data',
        'downloads_used', 'downloads_limit',
        'expires_at', 'created_at', 'updated_at'
    ];

    $update = [];
    foreach ( $data as $k => $v ) {
        if ( in_array( $k, $allowed, true ) ) {
            $update[ $k ] = $v; // allow NULL values too
        }
    }

    if ( empty( $update ) ) return false;

    // Build formats (wpdb can handle NULL values; format still ok)
    $format = [];
    foreach ( $update as $k => $v ) {
        if ( in_array( $k, [ 'order_id','product_id','downloads_used','downloads_limit' ], true ) ) {
            $format[] = '%d';
        } else {
            $format[] = '%s';
        }
    }

    $res = $wpdb->update(
        $table,
        $update,
        [ 'id' => $id ],
        $format,
        [ '%d' ]
    );

    return ( $res !== false );
}
