<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Render admin notices based on ?msg=...
 */
function shoplite_pro_deliveries_render_notice() {
    $msg = isset( $_GET['msg'] ) ? sanitize_key( wp_unslash( $_GET['msg'] ) ) : '';
    if ( $msg === '' ) {
        return;
    }

    $map = [
        'revoked'               => [ 'success', __( 'Delivery revoked.', 'shoplite-pro' ) ],
        'token_regen'           => [ 'success', __( 'Token regenerated.', 'shoplite-pro' ) ],
        'resent'                => [ 'success', __( 'Delivery email resent.', 'shoplite-pro' ) ],
        'resend_not_configured' => [ 'warning', __( 'Resend is not configured yet (email sender method missing).', 'shoplite-pro' ) ],
        'not_found'             => [ 'error',   __( 'Delivery not found.', 'shoplite-pro' ) ],
        'invalid_id'            => [ 'error',   __( 'Invalid delivery ID.', 'shoplite-pro' ) ],
        'db_error'              => [ 'error',   __( 'Database error. Please check logs.', 'shoplite-pro' ) ],
        'nonce'                 => [ 'error',   __( 'Security check failed. Please try again.', 'shoplite-pro' ) ],
        'no_permissions'        => [ 'error',   __( 'No permissions.', 'shoplite-pro' ) ],
    ];

    if ( ! isset( $map[ $msg ] ) ) {
        return;
    }

    [ $type, $text ] = $map[ $msg ];

    $class = 'notice';
    if ( $type === 'success' ) {
        $class .= ' notice-success';
    } elseif ( $type === 'warning' ) {
        $class .= ' notice-warning';
    } else {
        $class .= ' notice-error';
    }

    echo '<div class="' . esc_attr( $class ) . ' is-dismissible"><p>' . esc_html( $text ) . '</p></div>';
}

add_action(
    'admin_menu',
    function () {
        add_submenu_page(
            'shoplite',
            __( 'Deliveries', 'shoplite-pro' ),
            __( 'Deliveries (Pro)', 'shoplite-pro' ),
            'manage_options',
            'shoplite-deliveries',
            'shoplite_pro_render_deliveries_page'
        );
    },
    50
);

function shoplite_pro_deliveries_detail_url( $id ) {
    return admin_url( 'admin.php?page=shoplite-deliveries&view=detail&id=' . (int) $id );
}

function shoplite_pro_render_deliveries_page() {

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'No permissions.', 'shoplite-pro' ) );
    }

    // Ensure actions are loaded (admin-post handlers)
    $actions_file = SHOPLITE_PRO_DIR . 'includes/pro/deliveries-actions.php';
    if ( file_exists( $actions_file ) ) {
        require_once $actions_file;
    }

    require_once SHOPLITE_PRO_DIR . 'includes/pro/deliveries-model.php';
    require_once SHOPLITE_PRO_DIR . 'includes/pro/class-shoplite-deliveries-table.php';

    $view = isset( $_GET['view'] ) ? sanitize_key( wp_unslash( $_GET['view'] ) ) : 'list';

    // Safety net por si tu CSS admin oculta tablas de WP_List_Table
    echo '<style>
        .wrap .wp-list-table { display: table !important; visibility: visible !important; }
        .wrap .tablenav { display: block !important; }
    </style>';

    echo '<div class="wrap">';

    // Notices (list + detail)
    shoplite_pro_deliveries_render_notice();

    // =========================
    // DETAIL VIEW
    // =========================
    if ( $view === 'detail' ) {

        $id = isset( $_GET['id'] ) ? (int) $_GET['id'] : 0;
        if ( $id <= 0 ) {
            echo '<h1>' . esc_html__( 'Delivery details', 'shoplite-pro' ) . '</h1>';
            echo '<p>' . esc_html__( 'Invalid delivery ID.', 'shoplite-pro' ) . '</p>';
            echo '</div>';
            return;
        }

        $delivery = function_exists( 'shoplite_delivery_get' ) ? shoplite_delivery_get( $id ) : null;

        echo '<h1>' . esc_html__( 'Delivery details', 'shoplite-pro' ) . '</h1>';
        echo '<p><a href="' . esc_url( admin_url( 'admin.php?page=shoplite-deliveries' ) ) . '">&larr; ' . esc_html__( 'Back to list', 'shoplite-pro' ) . '</a></p>';

        if ( ! $delivery || ! is_array( $delivery ) ) {
            echo '<p>' . esc_html__( 'Delivery not found.', 'shoplite-pro' ) . '</p>';
            echo '</div>';
            return;
        }

        // Action buttons (Regenerate token / Revoke / Resend)
        $return = 'detail';

        $regen_url = wp_nonce_url(
            admin_url( 'admin-post.php?action=shoplite_pro_delivery_regen_token&id=' . $id . '&return=' . $return ),
            'shoplite_pro_delivery_regen_token_' . $id
        );

        $revoke_url = wp_nonce_url(
            admin_url( 'admin-post.php?action=shoplite_pro_delivery_revoke&id=' . $id . '&return=' . $return ),
            'shoplite_pro_delivery_revoke_' . $id
        );

        $resend_url = wp_nonce_url(
            admin_url( 'admin-post.php?action=shoplite_pro_delivery_resend&id=' . $id . '&return=' . $return ),
            'shoplite_pro_delivery_resend_' . $id
        );

        $status     = isset( $delivery['status'] ) ? (string) $delivery['status'] : '';
        $is_revoked = ( $status === 'revoked' );

        echo '<div style="margin:12px 0; padding:12px; background:#fff; border:1px solid #dcdcde; border-radius:6px;">';

        echo '<a class="button button-primary" href="' . esc_url( $regen_url ) . '">'
            . esc_html__( 'Regenerate token', 'shoplite-pro' )
            . '</a> ';

        echo '<a class="button" href="' . esc_url( $revoke_url ) . '"'
            . ' onclick="return confirm(\'' . esc_js( __( 'Revoke this delivery? The token will be removed and the download will stop working.', 'shoplite-pro' ) ) . '\');"'
            . ' style="border-color:#b91c1c;color:#b91c1c;">'
            . esc_html__( 'Revoke', 'shoplite-pro' )
            . '</a> ';

        echo '<a class="button" href="' . esc_url( $resend_url ) . '">'
            . esc_html__( 'Resend email', 'shoplite-pro' )
            . '</a>';

        if ( $is_revoked ) {
            echo '<p style="margin:10px 0 0; color:#b91c1c;">'
                . esc_html__( 'This delivery is revoked. Regenerate token to reactivate (status may return to pending).', 'shoplite-pro' )
                . '</p>';
        }

        echo '</div>';

        // Render details table (pretty JSON for delivery_data)
        echo '<table class="widefat striped" style="max-width: 980px">';
        echo '<tbody>';

        foreach ( $delivery as $k => $v ) {

            $val_html = '';

            if ( $k === 'delivery_data' && is_string( $v ) && $v !== '' ) {
                $decoded = json_decode( $v, true );
                if ( json_last_error() === JSON_ERROR_NONE ) {
                    $pretty   = wp_json_encode( $decoded, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
                    $val_html = '<pre style="margin:0; white-space:pre-wrap;">' . esc_html( $pretty ) . '</pre>';
                } else {
                    $val_html = '<pre style="margin:0; white-space:pre-wrap;">' . esc_html( (string) $v ) . '</pre>';
                }
            } else {
                $val      = is_null( $v ) ? '' : (string) $v;
                $val_html = esc_html( $val );
            }

            echo '<tr>';
            echo '<th style="width:220px">' . esc_html( $k ) . '</th>';
            echo '<td>' . $val_html . '</td>';
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';

        echo '</div>';
        return;
    }

    // =========================
    // LIST VIEW
    // =========================

    $table = new Shoplite_Deliveries_Table();
    $table->prepare_items();

    $current_status = isset( $_GET['status'] ) ? sanitize_key( wp_unslash( $_GET['status'] ) ) : '';
    $statuses       = [
        ''          => __( 'All', 'shoplite-pro' ),
        'pending'   => __( 'Pending', 'shoplite-pro' ),
        'delivered' => __( 'Delivered', 'shoplite-pro' ),
        'expired'   => __( 'Expired', 'shoplite-pro' ),
        'revoked'   => __( 'Revoked', 'shoplite-pro' ),
        'failed'    => __( 'Failed', 'shoplite-pro' ),
    ];

    echo '<h1>' . esc_html__( 'Deliveries (Pro)', 'shoplite-pro' ) . '</h1>';

    // Filtros rápidos por estado
    echo '<ul class="subsubsub">';
    $i     = 0;
    $count = count( $statuses );
    foreach ( $statuses as $k => $label ) {
        $url   = admin_url( 'admin.php?page=shoplite-deliveries' . ( $k !== '' ? '&status=' . rawurlencode( $k ) : '' ) );
        $class = ( $k === $current_status ) ? 'class="current"' : '';
        echo '<li><a ' . $class . ' href="' . esc_url( $url ) . '">' . esc_html( $label ) . '</a>' . ( ++$i < $count ? ' | ' : '' ) . '</li>';
    }
    echo '</ul>';

    echo '<form method="get">';
    echo '<input type="hidden" name="page" value="shoplite-deliveries" />';

    if ( $current_status !== '' ) {
        echo '<input type="hidden" name="status" value="' . esc_attr( $current_status ) . '" />';
    }

    // Preserve view if present (should be list, but keep it clean)
    echo '<input type="hidden" name="view" value="list" />';

    $table->search_box( __( 'Search', 'shoplite-pro' ), 'shoplite-deliveries' );
    $table->display();

    echo '</form>';
    echo '</div>';
}
