<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Actions handlers for Deliveries (Pro)
 *
 * admin-post.php?action=shoplite_pro_delivery_resend&id=123
 * admin-post.php?action=shoplite_pro_delivery_revoke&id=123
 * admin-post.php?action=shoplite_pro_delivery_regen_token&id=123
 */

// Ensure model is available
require_once SHOPLITE_PRO_DIR . 'includes/pro/deliveries-model.php';

/**
 * Small helper: redirect back to Deliveries list with message
 */
function shoplite_pro_deliveries_redirect( $args = [] ) {

    $return = sanitize_text_field( $_GET['return'] ?? '' );
    $id     = (int) ( $_GET['id'] ?? 0 );

    // Default: back to list
    $url = admin_url( 'admin.php?page=shoplite-deliveries' );

    // If called from detail view, return to that detail
    if ( $return === 'detail' && $id > 0 ) {
        $url = admin_url( 'admin.php?page=shoplite-deliveries&view=detail&id=' . $id );
    }

    if ( ! empty( $args ) ) {
        $url = add_query_arg( $args, $url );
    }

    wp_safe_redirect( $url );
    exit;
}


/**
 * Small helper: common guards
 */
function shoplite_pro_deliveries_guard( $action, $id ) {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'No permissions.', 'shoplite-pro' ) );
    }

    $id = (int) $id;
    if ( $id <= 0 ) {
        shoplite_pro_deliveries_redirect( [ 'msg' => 'invalid_id' ] );
    }

    check_admin_referer( $action . '_' . $id );

    return $id;
}

/**
 * REVOKE: set status=revoked, token=NULL
 */
add_action( 'admin_post_shoplite_pro_delivery_revoke', function () {

    $id = shoplite_pro_deliveries_guard( 'shoplite_pro_delivery_revoke', $_GET['id'] ?? 0 );

    $delivery = shoplite_delivery_get( $id );
    if ( ! $delivery ) {
        shoplite_pro_deliveries_redirect( [ 'msg' => 'not_found' ] );
    }

    $ok = shoplite_delivery_update( $id, [
        'status'     => 'revoked',
        'token'      => null,
        'updated_at' => current_time( 'mysql' ),
    ] );

    shoplite_pro_deliveries_redirect( [
        'msg' => $ok ? 'revoked' : 'db_error',
    ] );
} );

/**
 * REGEN TOKEN: generate a new random token (and set status to pending if revoked/expired)
 */
add_action( 'admin_post_shoplite_pro_delivery_regen_token', function () {

    $id = shoplite_pro_deliveries_guard( 'shoplite_pro_delivery_regen_token', $_GET['id'] ?? 0 );

    $delivery = shoplite_delivery_get( $id );
    if ( ! $delivery ) {
        shoplite_pro_deliveries_redirect( [ 'msg' => 'not_found' ] );
    }

    // 32 bytes -> 64 hex chars
    $new_token = bin2hex( random_bytes( 32 ) );

    $new_status = $delivery['status'] ?? '';
    if ( in_array( $new_status, [ 'revoked', 'expired', 'failed' ], true ) ) {
        $new_status = 'pending';
    }

    $ok = shoplite_delivery_update( $id, [
        'token'      => $new_token,
        'status'     => $new_status,
        'updated_at' => current_time( 'mysql' ),
    ] );

    shoplite_pro_deliveries_redirect( [
        'msg' => $ok ? 'token_regen' : 'db_error',
    ] );
} );

/**
 * RESEND: trigger your existing email delivery system
 * - We call a method if it exists
 * - If the delivery has no token, we regenerate one first
 */
add_action( 'admin_post_shoplite_pro_delivery_resend', function () {

    $id = shoplite_pro_deliveries_guard( 'shoplite_pro_delivery_resend', $_GET['id'] ?? 0 );

    $delivery = shoplite_delivery_get( $id );
    if ( ! $delivery ) {
        shoplite_pro_deliveries_redirect( [ 'msg' => 'not_found' ] );
    }

    // Ensure token exists
    $token = (string) ( $delivery['token'] ?? '' );
    if ( $token === '' ) {
        $token = bin2hex( random_bytes( 32 ) );
        shoplite_delivery_update( $id, [
            'token'      => $token,
            'status'     => 'pending',
            'updated_at' => current_time( 'mysql' ),
        ] );
        $delivery['token'] = $token;
    }

    /**
     * Try to send email using your Pro email class.
     * We don’t assume method names; we try a couple of common ones.
     * Adjust ONE line below once you confirm the real method in your class.
     */
    $sent = false;

    if ( class_exists( 'Shoplite_Pro_Digital_Delivery_Email' ) ) {

        // Option 1: static method
        if ( method_exists( 'Shoplite_Pro_Digital_Delivery_Email', 'send_delivery_email' ) ) {
            $sent = (bool) Shoplite_Pro_Digital_Delivery_Email::send_delivery_email( $delivery );
        }

        // Option 2: instance method
        if ( ! $sent && method_exists( 'Shoplite_Pro_Digital_Delivery_Email', 'send' ) ) {
            $obj = new Shoplite_Pro_Digital_Delivery_Email();
            $sent = (bool) $obj->send( $delivery );
        }
    }

    // If no sender available, don’t fail hard
    if ( ! $sent ) {
        shoplite_pro_deliveries_redirect( [ 'msg' => 'resend_not_configured' ] );
    }

    // Update status to delivered (optional; depends on your logic)
    shoplite_delivery_update( $id, [
        'status'     => 'delivered',
        'updated_at' => current_time( 'mysql' ),
    ] );

    shoplite_pro_deliveries_redirect( [ 'msg' => 'resent' ] );
} );
