<?php
if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class Shoplite_Deliveries_Table extends WP_List_Table {

    public function __construct() {
        parent::__construct( [
            'singular' => 'delivery',
            'plural'   => 'deliveries',
            'ajax'     => false,
        ] );
    }

    public function get_columns() {
        return [
            'id'             => 'ID',
            'order_id'       => __( 'Order', 'shoplite-pro' ),
            'product_id'     => __( 'Product', 'shoplite-pro' ),
            'customer_email' => __( 'Customer', 'shoplite-pro' ),
            'delivery_type'  => __( 'Type', 'shoplite-pro' ),
            'status'         => __( 'Status', 'shoplite-pro' ),
            'downloads'      => __( 'Downloads', 'shoplite-pro' ),
            'expires_at'     => __( 'Expires', 'shoplite-pro' ),
            'created_at'     => __( 'Created', 'shoplite-pro' ),
            'actions'        => __( 'Actions', 'shoplite-pro' ),
        ];
    }

    public function get_sortable_columns() {
        return [
            'id'         => [ 'id', true ],
            'order_id'   => [ 'order_id', false ],
            'product_id' => [ 'product_id', false ],
            'status'     => [ 'status', false ],
            'created_at' => [ 'created_at', false ],
            'expires_at' => [ 'expires_at', false ],
        ];
    }

    /**
     * ID -> enlace a detalle
     */
    public function column_id( $item ) {
        $id = (int) ( $item['id'] ?? 0 );
        $url = admin_url( 'admin.php?page=shoplite-deliveries&view=detail&id=' . $id );
        return '<a href="' . esc_url( $url ) . '"><strong>' . $id . '</strong></a>';
    }

    /**
     * Order -> si existe un "pedido" interno, aquí lo enlazamos.
     * Si no, al menos muestra el número.
     */
    public function column_order_id( $item ) {
        $order_id = (int) ( $item['order_id'] ?? 0 );

        // Ajusta este link si tienes una pantalla de pedidos distinta.
        // En tu menú veo "Pedidos", así que el slug podría ser "shoplite_orders" o similar.
        // Si no coincide, el enlace igual no funcionará, pero no rompe: lo dejamos como texto.
        $possible = [
            admin_url( 'admin.php?page=shoplite_orders&order_id=' . $order_id ),
            admin_url( 'admin.php?page=shoplite-orders&order_id=' . $order_id ),
            admin_url( 'admin.php?page=shoplite_orders&view=detail&id=' . $order_id ),
            admin_url( 'admin.php?page=shoplite-orders&view=detail&id=' . $order_id ),
        ];

        // Mostramos el primero (si luego lo confirmas, lo fijamos definitivo)
        $url = $possible[0];

        return '<a href="' . esc_url( $url ) . '">' . esc_html( (string) $order_id ) . '</a>';
    }

    /**
     * Product -> por ahora ID. Si tienes una función para obtener nombre,
     * aquí es donde se enchufa.
     */
    public function column_product_id( $item ) {
        $product_id = (int) ( $item['product_id'] ?? 0 );

        // Si Shoplite core tiene helper tipo shoplite_get_product($id), puedes usarlo.
        // Ejemplo:
        // if ( function_exists('shoplite_get_product') ) {
        //     $p = shoplite_get_product($product_id);
        //     if ($p && !empty($p['name'])) return esc_html($p['name']) . ' (#' . $product_id . ')';
        // }

        return esc_html( (string) $product_id );
    }

    public function column_customer_email( $item ) {
        $email = (string) ( $item['customer_email'] ?? '' );
        return esc_html( $email );
    }

    public function column_delivery_type( $item ) {
        $type = (string) ( $item['delivery_type'] ?? '' );
        return esc_html( $type );
    }

    /**
     * Status -> badge simple
     */
    public function column_status( $item ) {
        $status = (string) ( $item['status'] ?? '' );

        $map = [
            'pending'   => [ __( 'Pending', 'shoplite-pro' ),  '#b45309', '#fef3c7' ],
            'delivered' => [ __( 'Delivered', 'shoplite-pro' ), '#047857', '#d1fae5' ],
            'expired'   => [ __( 'Expired', 'shoplite-pro' ),  '#6b7280', '#f3f4f6' ],
            'revoked'   => [ __( 'Revoked', 'shoplite-pro' ),  '#b91c1c', '#fee2e2' ],
            'failed'    => [ __( 'Failed', 'shoplite-pro' ),   '#b91c1c', '#fee2e2' ],
        ];

        if ( isset( $map[ $status ] ) ) {
            list( $label, $color, $bg ) = $map[ $status ];
        } else {
            $label = $status !== '' ? $status : __( 'Unknown', 'shoplite-pro' );
            $color = '#374151';
            $bg    = '#e5e7eb';
        }

        return '<span style="display:inline-block;padding:2px 8px;border-radius:999px;font-size:12px;line-height:18px;color:' . esc_attr( $color ) . ';background:' . esc_attr( $bg ) . ';">'
            . esc_html( $label )
            . '</span>';
    }

    public function column_downloads( $item ) {
        $used = (int) ( $item['downloads_used'] ?? 0 );
        $limit_raw = $item['downloads_limit'] ?? null;
        $limit = is_null( $limit_raw ) ? '∞' : (int) $limit_raw;
        return esc_html( $used . ' / ' . $limit );
    }

    public function column_expires_at( $item ) {
        $v = $item['expires_at'] ?? '';
        return esc_html( (string) $v );
    }

    public function column_created_at( $item ) {
        $v = $item['created_at'] ?? '';
        return esc_html( (string) $v );
    }

    /**
     * Actions reales: View / Resend / Revoke / Regen token
     * (Solo links por ahora; handlers en siguiente paso)
     */
    public function column_actions( $item ) {
        $id = (int) ( $item['id'] ?? 0 );

        $view_url = admin_url( 'admin.php?page=shoplite-deliveries&view=detail&id=' . $id );

        $resend_url = wp_nonce_url(
            admin_url( 'admin-post.php?action=shoplite_pro_delivery_resend&id=' . $id ),
            'shoplite_pro_delivery_resend_' . $id
        );

        $revoke_url = wp_nonce_url(
            admin_url( 'admin-post.php?action=shoplite_pro_delivery_revoke&id=' . $id ),
            'shoplite_pro_delivery_revoke_' . $id
        );

        $regen_url = wp_nonce_url(
            admin_url( 'admin-post.php?action=shoplite_pro_delivery_regen_token&id=' . $id ),
            'shoplite_pro_delivery_regen_token_' . $id
        );

        $links = [];
        $links[] = '<a href="' . esc_url( $view_url ) . '">' . esc_html__( 'View', 'shoplite-pro' ) . '</a>';
        $links[] = '<a href="' . esc_url( $resend_url ) . '">' . esc_html__( 'Resend', 'shoplite-pro' ) . '</a>';
        $links[] = '<a href="' . esc_url( $regen_url ) . '">' . esc_html__( 'Regen token', 'shoplite-pro' ) . '</a>';
        $links[] = '<a href="' . esc_url( $revoke_url ) . '" style="color:#b91c1c;">' . esc_html__( 'Revoke', 'shoplite-pro' ) . '</a>';

        return implode( ' | ', $links );
    }

    public function column_default( $item, $column_name ) {
        if ( isset( $item[ $column_name ] ) ) {
            return esc_html( (string) $item[ $column_name ] );
        }
        return '';
    }

    public function prepare_items() {

        // Evita renders raros: define headers explícitamente
        $this->_column_headers = [ $this->get_columns(), [], $this->get_sortable_columns(), 'id' ];

        $per_page = 20;
        $paged = max( 1, (int) ( $_GET['paged'] ?? 1 ) );

        $status = sanitize_text_field( $_GET['status'] ?? '' );
        $search = sanitize_text_field( $_GET['s'] ?? '' );

        $orderby = sanitize_text_field( $_GET['orderby'] ?? 'id' );
        $order   = sanitize_text_field( $_GET['order'] ?? 'DESC' );

        list( $items, $total ) = shoplite_delivery_list( [
            'paged'    => $paged,
            'per_page' => $per_page,
            'status'   => $status,
            'search'   => $search,
            'orderby'  => $orderby,
            'order'    => $order,
        ] );

        $this->items = $items;

        $this->set_pagination_args( [
            'total_items' => $total,
            'per_page'    => $per_page,
            'total_pages' => ( $per_page > 0 ) ? ceil( $total / $per_page ) : 1,
        ] );
    }
}
