<?php
// shoplite-pro/includes/class-shoplite-pro-updater.php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Shoplite_Pro_Updater {

    // Endpoint that will return version, changelog and more
    const UPDATE_ENDPOINT = 'https://talariasoft.com/api/shoplite-pro/update.php';
    const TD              = 'shoplite-pro'; // Text domain (Pro)

    public static function init() {

        // Check for automatic updates
        add_filter( 'pre_set_site_transient_update_plugins', [ __CLASS__, 'check_update' ] );

        // “View details” popup
        add_filter( 'plugins_api', [ __CLASS__, 'plugins_api' ], 20, 3 );
    }

    /**
     * Check if there is an update available for Shoplite Pro.
     */
    public static function check_update( $transient ) {

        // Nothing to do if WP has not collected plugin data yet
        if ( empty( $transient->checked ) ) {
            return $transient;
        }

        // Only look for updates if the license is active
        if ( ! class_exists( 'Shoplite_Pro_Licensing' ) || ! Shoplite_Pro_Licensing::is_active() ) {
            return $transient;
        }

        $plugin_file = 'shoplite-pro/shoplite-pro.php';

        // Current installed version
        if ( empty( $transient->checked[ $plugin_file ] ) ) {
            return $transient;
        }

        $current_version = $transient->checked[ $plugin_file ];

        // Request to the remote server
        $response = wp_remote_get(
            self::UPDATE_ENDPOINT,
            [
                'timeout' => 15,
                'headers' => [
                    'Accept' => 'application/json',
                ],
            ]
        );

        if ( is_wp_error( $response ) ) {
            return $transient;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( 200 !== $code ) {
            return $transient;
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( ! is_array( $body ) || empty( $body['version'] ) ) {
            return $transient;
        }

        // Is it a newer version?
        if ( version_compare( $body['version'], $current_version, '<=' ) ) {
            return $transient;
        }

        // We require a valid license to get the protected download
        $license = get_option( 'shoplite_pro_license_key', '' );
        if ( empty( $license ) ) {
            return $transient;
        }

        // Protected download URL
        $download_url = add_query_arg(
            [
                'license_key' => rawurlencode( $license ),
                'domain'      => rawurlencode( home_url() ),
            ],
            'https://talariasoft.com/api/shoplite-pro/download.php'
        );

        // Structure that WordPress understands as an update
        $update              = new stdClass();
        $update->slug        = 'shoplite-pro';
        $update->plugin      = $plugin_file;
        $update->new_version = sanitize_text_field( $body['version'] );
        $update->url         = 'https://talariasoft.com/shoplite-pro';
        $update->package     = esc_url_raw( $download_url );

        // Register the update
        $transient->response[ $plugin_file ] = $update;

        return $transient;
    }

    /**
     * Controls the “Plugin details” window.
     */
    public static function plugins_api( $res, $action, $args ) {

        // Only act if WP is requesting information about this plugin
        if ( 'plugin_information' !== $action ) {
            return $res;
        }

        if ( empty( $args->slug ) || 'shoplite-pro' !== $args->slug ) {
            return $res;
        }

        $response = wp_remote_get(
            self::UPDATE_ENDPOINT . '?info=1',
            [
                'timeout' => 15,
                'headers' => [
                    'Accept' => 'application/json',
                ],
            ]
        );

        if ( is_wp_error( $response ) ) {
            return $res;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( 200 !== $code ) {
            return $res;
        }

        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        if ( ! is_array( $body ) ) {
            return $res;
        }

        // Build the object required by WP
        $info                = new stdClass();
        $info->name          = __( 'Talariasoft Shoplite Pro', self::TD );
        $info->slug          = 'shoplite-pro';
        $info->version       = sanitize_text_field( $body['version'] ?? '1.0.0' );
        $info->author        = '<a href="https://talariasoft.com">Talariasoft</a>';
        $info->homepage      = 'https://talariasoft.com/shoplite-pro';
        $info->download_link = esc_url_raw( $body['download_url'] ?? '' );

        // Popup sections
        $info->sections = [
            'description' => wp_kses_post(
                $body['description'] ?? __( 'Pro add-on for Talariasoft Shoplite.', self::TD )
            ),
            'changelog'   => wp_kses_post(
                $body['changelog'] ?? __( 'No changelog entries available.', self::TD )
            ),
        ];

        return $info;
    }

}

