<?php
// shoplite-pro/includes/class-shoplite-pro-sales.php
if ( ! defined( 'ABSPATH' ) ) exit;

class Shoplite_Pro_Sales {

    const MENU_SLUG = 'shoplite_pro_sales';
    const TD        = 'shoplite-pro'; // Text domain for Pro

    /**
     * Initial hook: register the submenu.
     */
    public static function init() {
        // Pro reports menu
        add_action( 'admin_menu', [ __CLASS__, 'register_menu' ], 30 );

        // Export TOP customers
        add_action(
            'admin_post_shoplite_export_top_customers',
            [ __CLASS__, 'handle_export_top_customers_csv' ]
        );

        // Export ALL customers
        add_action(
            'admin_post_shoplite_export_all_customers',
            [ __CLASS__, 'handle_export_all_customers_csv' ]
        );

        // Export TOP products
        add_action(
            'admin_post_shoplite_export_top_products',
            [ __CLASS__, 'handle_export_top_products_csv' ]
        );
    }

    /**
     * Register the main reports screen.
     */
    public static function register_menu() {
        add_submenu_page(
            'shoplite',
            __( 'Sales reports (Pro)', self::TD ),
            __( 'Sales reports (Pro)', self::TD ),
            'manage_options',
            self::MENU_SLUG,
            [ __CLASS__, 'render_page' ]
        );
    }

    /**
     * ===== MAIN RENDER (with tabs) =====
     */
    public static function render_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have sufficient permissions.', self::TD ) );
        }

        // Chart.js for Pro reports tabs
        wp_enqueue_script(
            'shoplite-chartjs',
            'https://cdn.jsdelivr.net/npm/chart.js',
            [],
            '4.4.0',
            true
        );

        // Active tab
        $tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : 'summary';
        ?>
        <div class="wrap shoplite-pro-sales">

            <?php
            // Global brand header
            if ( function_exists( 'shoplite_render_brand_header' ) ) {
                shoplite_render_brand_header();
            }
            ?>

            <h1 style="margin-bottom:4px;">
                <?php esc_html_e( 'Sales reports (Pro)', self::TD ); ?>
            </h1>

            <!-- NAV TABS -->
            <h2 class="nav-tab-wrapper shoplite-pro-sales-tabs">
                <?php self::render_tab_link( 'summary',       __( 'Overview', self::TD ),          $tab ); ?>
                <?php self::render_tab_link( 'products',      __( 'Top products', self::TD ),      $tab ); ?>
                <?php self::render_tab_link( 'top_customers', __( 'Top customers', self::TD ),     $tab ); ?>
                <?php self::render_tab_link( 'all_customers', __( 'All customers', self::TD ),     $tab ); ?>
                <?php self::render_tab_link( 'payments',      __( 'Payment methods', self::TD ),   $tab ); ?>
            </h2>

            <div class="shoplite-pro-sales-content">
                <?php
                switch ( $tab ) {
                    case 'products':
                        self::render_tab_products();
                        break;

                    case 'top_customers':
                        self::render_tab_top_customers();
                        break;

                    case 'all_customers':
                        self::render_tab_all_customers();
                        break;

                    case 'payments':
                        self::render_tab_payment_methods();
                        break;

                    case 'summary':
                    default:
                        self::render_tab_summary();
                        break;
                }
                ?>
            </div>
        </div>

        <?php self::render_tabs_css(); ?>
        <?php
    }

    /**
     * Render a tab link.
     */
    protected static function render_tab_link( $slug, $label, $active ) {
        $class = ( $slug === $active ) ? 'nav-tab nav-tab-active' : 'nav-tab';
        $url   = admin_url( 'admin.php?page=' . self::MENU_SLUG . '&tab=' . $slug );
        echo '<a href="' . esc_url( $url ) . '" class="' . esc_attr( $class ) . '">' . esc_html( $label ) . '</a>';
    }



    /* ===================================================================
 * FUNCIONES DE DATOS
 * =================================================================== */

/**
 * Resumen de pedidos completados.
 *
 * @param int|null $days  Últimos X días o null para todo.
 * @return array { orders, revenue, avg }
 */
protected static function get_orders_summary( $days = null ) {

    $args = [
        'post_type'      => 'shoplite_pedido',
        'post_status'    => 'publish',
        'posts_per_page' => -1,
        'meta_query'     => [
            [
                'key'   => '_shoplite_order_status',
                'value' => 'completed',
            ],
        ],
    ];

    if ( is_int( $days ) && $days > 0 ) {
        $args['date_query'] = [
            [
                'after'  => gmdate( 'Y-m-d', time() - $days * DAY_IN_SECONDS ),
                'column' => 'post_date',
            ],
        ];
    }

    $orders = get_posts( $args );

    $total = 0.0;
    $count = 0;

    foreach ( $orders as $order ) {
        $t = get_post_meta( $order->ID, '_shoplite_order_total', true );
        if ( is_numeric( $t ) ) {
            $total += (float) $t;
        }
        $count++;
    }

    return [
        'orders'  => $count,
        'revenue' => $total,
        'avg'     => ( $count > 0 ) ? ( $total / $count ) : 0.0,
    ];
}

/**
 * Devuelve los productos más vendidos en un rango de fechas.
 *
 * @param string      $start   Fecha de inicio (Y-m-d) o (Y-m-d H:i:s).
 * @param string      $end     Fecha de fin   (Y-m-d) o (Y-m-d H:i:s).
 * @param int         $limit   Máximo de productos a devolver.
 * @param string|null $gateway Método de pago ('stripe', 'paypal', etc.) o null para todos.
 * @return array[] {
 *   @type int    product_id  ID del producto.
 *   @type string name        Nombre del producto.
 *   @type int    qty         Unidades vendidas.
 *   @type float  revenue     Ingresos generados por ese producto.
 * }
 */
 public static function get_top_products( $start, $end, $limit = 10, $gateway = null ) {

     // Normalizar fechas (usamos post_date igual que en las estadísticas)
     $start_dt = date( 'Y-m-d 00:00:00', strtotime( $start ) );
     $end_dt   = date( 'Y-m-d 23:59:59', strtotime( $end ) );

     // Meta query base: solo pedidos completados
     $meta_query = [
         [
             'key'   => '_shoplite_order_status',
             'value' => 'completed',
         ],
     ];

     // Filtro opcional por pasarela / método de pago
     if ( ! empty( $gateway ) ) {
         $meta_query[] = [
             'relation' => 'OR',
             [
                 'key'   => '_shoplite_payment_method',
                 'value' => $gateway,
             ],
             [
                 'key'   => '_shoplite_order_gateway',
                 'value' => $gateway,
             ],
         ];
     }

     $args = [
         'post_type'      => 'shoplite_pedido',
         'post_status'    => 'publish',
         'posts_per_page' => -1,
         'no_found_rows'  => true,
         'orderby'        => 'date',
         'order'          => 'ASC',
         'date_query'     => [
             [
                 'after'     => $start_dt,
                 'before'    => $end_dt,
                 'inclusive' => true,
                 'column'    => 'post_date',
             ],
         ],
         'meta_query'     => $meta_query,
     ];

     $orders = get_posts( $args );

     if ( empty( $orders ) ) {
         return [];
     }

     $products = [];

     foreach ( $orders as $order ) {
         $order_id = $order->ID;

         // Meta donde guardamos las líneas de pedido (_shoplite_order_items)
         $items = get_post_meta( $order_id, '_shoplite_order_items', true );

         if ( empty( $items ) ) {
             continue;
         }

         // 1) Puede venir como string serializada (a veces anidada) o JSON
         if ( is_string( $items ) ) {
             $un1 = @unserialize( $items );

             if ( $un1 !== false || $items === 'b:0;' ) {
                 $items = $un1;

                 // Si sigue siendo string tipo "a:1:{...}" => deserializar una segunda vez
                 if ( is_string( $items ) && preg_match( '/^a:\d+:/', $items ) ) {
                     $un2 = @unserialize( $items );
                     if ( $un2 !== false || $items === 'b:0;' ) {
                         $items = $un2;
                     }
                 }
             } else {
                 // Plan B: JSON
                 $decoded = json_decode( $items, true );
                 if ( json_last_error() === JSON_ERROR_NONE ) {
                     $items = $decoded;
                 }
             }
         }

         if ( ! is_array( $items ) ) {
             continue;
         }

         foreach ( $items as $item_key => $item ) {
             if ( ! is_array( $item ) ) {
                 continue;
             }

             // 1) ID de producto
             $pid = 0;
             if ( isset( $item['product_id'] ) ) {
                 $pid = (int) $item['product_id'];
             } elseif ( isset( $item['id'] ) ) {
                 $pid = (int) $item['id'];
             } elseif ( isset( $item['product'] ) ) {
                 $pid = (int) $item['product'];
             } elseif ( isset( $item['post_id'] ) ) {
                 $pid = (int) $item['post_id'];
             }

             // Como último recurso, si la clave es algo tipo "tbp_22"
             if ( $pid <= 0 && is_string( $item_key ) && preg_match( '/(\d+)$/', $item_key, $m ) ) {
                 $pid = (int) $m[1];
             }

             // 2) Cantidad
             $qty = 0;
             if ( isset( $item['cantidad'] ) ) {
                 $qty = (int) $item['cantidad'];
             } elseif ( isset( $item['qty'] ) ) {
                 $qty = (int) $item['qty'];
             } elseif ( isset( $item['quantity'] ) ) {
                 $qty = (int) $item['quantity'];
             }

             if ( $pid <= 0 || $qty <= 0 ) {
                 continue;
             }

             // 3) Precio unitario
             $price = 0.0;
             if ( isset( $item['precio'] ) ) {
                 $price = (float) $item['precio'];
             } elseif ( isset( $item['price'] ) ) {
                 $price = (float) $item['price'];
             } elseif ( isset( $item['unit_price'] ) ) {
                 $price = (float) $item['unit_price'];
             } elseif ( isset( $item['importe_unitario'] ) ) {
                 $price = (float) $item['importe_unitario'];
             }

             if ( $price <= 0 ) {
                 continue;
             }

             $line_revenue = $price * $qty;

             // 4) Nombre del producto
             $name = '';
             if ( isset( $item['nombre'] ) ) {
                 $name = (string) $item['nombre'];
             } elseif ( isset( $item['name'] ) ) {
                 $name = (string) $item['name'];
             } elseif ( $pid > 0 ) {
                 $name = get_the_title( $pid );
             }

             if ( ! isset( $products[ $pid ] ) ) {
                 $products[ $pid ] = [
                     'product_id' => $pid,
                     'name'       => $name,
                     'qty'        => 0,
                     'revenue'    => 0.0,
                 ];
             }

             $products[ $pid ]['qty']     += $qty;
             $products[ $pid ]['revenue'] += $line_revenue;
         }
     }

     if ( empty( $products ) ) {
         return [];
     }

     // Convertir a array indexado y ordenar (por defecto, ingresos desc + unidades desc)
     $products = array_values( $products );

     usort(
         $products,
         function( $a, $b ) {
             $revA = isset( $a['revenue'] ) ? (float) $a['revenue'] : 0.0;
             $revB = isset( $b['revenue'] ) ? (float) $b['revenue'] : 0.0;

             if ( $revA === $revB ) {
                 $qtyA = isset( $a['qty'] ) ? (int) $a['qty'] : 0;
                 $qtyB = isset( $b['qty'] ) ? (int) $b['qty'] : 0;
                 return $qtyB <=> $qtyA;
             }

             return $revB <=> $revA;
         }
     );

     if ( $limit > 0 ) {
         $products = array_slice( $products, 0, $limit );
     }

     return $products;
 }


     /**
     * Top clientes por gasto.
     */
    protected static function get_top_customers( $limit = 50 ) {

        $args = [
            'post_type'      => 'shoplite_pedido',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'meta_query'     => [
                [
                    'key'   => '_shoplite_order_status',
                    'value' => 'completed',
                ],
            ],
        ];

        $posts     = get_posts( $args );
        $customers = [];

        foreach ( $posts as $p ) {

            $email = get_post_meta( $p->ID, '_shoplite_customer_email', true );
            if ( ! $email ) {
                continue;
            }

            $name_raw = get_post_meta( $p->ID, '_shoplite_customer_name', true );
            $name     = $name_raw ? $name_raw : '-';
            $total    = (float) get_post_meta( $p->ID, '_shoplite_order_total', true );

            if ( ! isset( $customers[ $email ] ) ) {
                $customers[ $email ] = [
                    'name'   => $name,
                    'email'  => $email,
                    'orders' => 0,
                    'spent'  => 0.0,
                ];
            }

            $customers[ $email ]['orders']++;
            $customers[ $email ]['spent'] += $total;
        }

        foreach ( $customers as $email => $c ) {
            $customers[ $email ]['avg'] = $c['orders'] > 0 ? ( $c['spent'] / $c['orders'] ) : 0.0;
        }

        usort(
            $customers,
            function( $a, $b ) {
                return $b['spent'] <=> $a['spent'];
            }
        );

        if ( $limit > 0 ) {
            $customers = array_slice( $customers, 0, $limit );
        }

        return $customers;
    }

    /**
     * Todos los clientes (sin límite).
     */
    protected static function get_all_customers() {

        $args = [
            'post_type'      => 'shoplite_pedido',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'meta_query'     => [
                [
                    'key'   => '_shoplite_order_status',
                    'value' => 'completed',
                ],
            ],
        ];

        $posts     = get_posts( $args );
        $customers = [];

        foreach ( $posts as $p ) {

            $email = get_post_meta( $p->ID, '_shoplite_customer_email', true );
            if ( ! $email ) {
                continue;
            }

            $name_raw = get_post_meta( $p->ID, '_shoplite_customer_name', true );
            $name     = $name_raw ? $name_raw : '-';
            $total    = (float) get_post_meta( $p->ID, '_shoplite_order_total', true );

            if ( ! isset( $customers[ $email ] ) ) {
                $customers[ $email ] = [
                    'name'   => $name,
                    'email'  => $email,
                    'orders' => 0,
                    'spent'  => 0.0,
                ];
            }

            $customers[ $email ]['orders']++;
            $customers[ $email ]['spent'] += $total;
        }

        usort(
            $customers,
            function( $a, $b ) {
                return strcmp( strtolower( $a['name'] ), strtolower( $b['name'] ) );
            }
        );

        return $customers;
    }

    /* ======================================================
     * TAB 1: GLOBAL OVERVIEW
     * ====================================================== */

    protected static function render_tab_summary() {

        $summary_all     = self::get_orders_summary();      // Full history
        $summary_30d     = self::get_orders_summary( 30 );  // Last 30 days
        $currency_symbol = get_option( 'shoplite_moneda_simbolo', '€' );
        ?>

        <div class="shoplite-card">
            <h2><?php esc_html_e( 'Global overview', self::TD ); ?></h2>

            <div class="shoplite-kpi-grid">
                <div class="shoplite-kpi">
                    <div class="shoplite-kpi__label">
                        <?php esc_html_e( 'Total revenue', self::TD ); ?>
                    </div>
                    <div class="shoplite-kpi__value">
                        <?php echo esc_html( number_format_i18n( $summary_all['revenue'], 2 ) ); ?>
                        <span class="shoplite-kpi__meta">
                            <?php echo esc_html( $currency_symbol ); ?>
                        </span>
                    </div>
                </div>

                <div class="shoplite-kpi">
                    <div class="shoplite-kpi__label">
                        <?php esc_html_e( 'Completed orders', self::TD ); ?>
                    </div>
                    <div class="shoplite-kpi__value">
                        <?php echo esc_html( number_format_i18n( $summary_all['orders'] ) ); ?>
                    </div>
                </div>

                <div class="shoplite-kpi">
                    <div class="shoplite-kpi__label">
                        <?php esc_html_e( 'Average order value', self::TD ); ?>
                    </div>
                    <div class="shoplite-kpi__value">
                        <?php echo esc_html( number_format_i18n( $summary_all['avg'], 2 ) ); ?>
                        <span class="shoplite-kpi__meta">
                            <?php echo esc_html( $currency_symbol ); ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <div class="shoplite-card">
            <h2><?php esc_html_e( 'Last 30 days', self::TD ); ?></h2>

            <div class="shoplite-kpi-grid">
                <div class="shoplite-kpi">
                    <div class="shoplite-kpi__label">
                        <?php esc_html_e( 'Revenue (30 days)', self::TD ); ?>
                    </div>
                    <div class="shoplite-kpi__value">
                        <?php echo esc_html( number_format_i18n( $summary_30d['revenue'], 2 ) ); ?>
                        <span class="shoplite-kpi__meta">
                            <?php echo esc_html( $currency_symbol ); ?>
                        </span>
                    </div>
                </div>

                <div class="shoplite-kpi">
                    <div class="shoplite-kpi__label">
                        <?php esc_html_e( 'Completed orders (30 days)', self::TD ); ?>
                    </div>
                    <div class="shoplite-kpi__value">
                        <?php echo esc_html( number_format_i18n( $summary_30d['orders'] ) ); ?>
                    </div>
                </div>

                <div class="shoplite-kpi">
                    <div class="shoplite-kpi__label">
                        <?php esc_html_e( 'Average order value (30 days)', self::TD ); ?>
                    </div>
                    <div class="shoplite-kpi__value">
                        <?php echo esc_html( number_format_i18n( $summary_30d['avg'], 2 ) ); ?>
                        <span class="shoplite-kpi__meta">
                            <?php echo esc_html( $currency_symbol ); ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <?php
    }


    /* ======================================================
     * TAB 2: TOP PRODUCTS
     * ====================================================== */
    protected static function render_tab_products() {

        $currency_symbol = get_option( 'shoplite_moneda_simbolo', '€' );

        // Allowed periods (in days)
        $periods = [
            30  => __( 'Last 30 days', self::TD ),
            90  => __( 'Last 90 days', self::TD ),
            365 => __( 'Last 365 days', self::TD ),
            730 => __( 'Last 2 years', self::TD ),
        ];

        // Sort modes
        $sort_modes = [
            'revenue' => __( 'By revenue', self::TD ),
            'qty'     => __( 'By units sold', self::TD ),
        ];

        // Filter by payment method
        $gateway_modes = [
            'all'    => __( 'All methods', self::TD ),
            'stripe' => __( 'Stripe only', self::TD ),
            'paypal' => __( 'PayPal only', self::TD ),
        ];

        // Read filters from URL
        $period = isset( $_GET['prd_period'] ) ? (int) $_GET['prd_period'] : 90;
        if ( ! isset( $periods[ $period ] ) ) {
            $period = 90;
        }

        $sort = isset( $_GET['prd_sort'] ) ? sanitize_key( $_GET['prd_sort'] ) : 'revenue';
        if ( ! isset( $sort_modes[ $sort ] ) ) {
            $sort = 'revenue';
        }

        $gateway = isset( $_GET['prd_gateway'] ) ? sanitize_key( $_GET['prd_gateway'] ) : 'all';
        if ( ! isset( $gateway_modes[ $gateway ] ) ) {
            $gateway = 'all';
        }

        // Compute actual date range
        $now_ts     = current_time( 'timestamp' );
        $range_days = max( 1, (int) $period );
        $start_ts   = $now_ts - ( $range_days - 1 ) * DAY_IN_SECONDS;

        $range_start = gmdate( 'Y-m-d', $start_ts );
        $range_end   = gmdate( 'Y-m-d', $now_ts );

        // Optional payment method filter (null = all)
        $gateway_filter = ( 'all' === $gateway ) ? null : $gateway;

        // Get top products (max 50)
        $products = self::get_top_products( $range_start, $range_end, 50, $gateway_filter );
        if ( ! is_array( $products ) ) {
            $products = [];
        }

        // Reorder if user wants to sort by units
        if ( 'qty' === $sort && ! empty( $products ) ) {
            usort(
                $products,
                function( $a, $b ) {
                    $qtyA = isset( $a['qty'] ) ? (int) $a['qty'] : 0;
                    $qtyB = isset( $b['qty'] ) ? (int) $b['qty'] : 0;
                    if ( $qtyA === $qtyB ) {
                        $revA = isset( $a['revenue'] ) ? (float) $a['revenue'] : 0.0;
                        $revB = isset( $b['revenue'] ) ? (float) $b['revenue'] : 0.0;
                        return $revB <=> $revA;
                    }
                    // More units first
                    return $qtyB <=> $qtyA;
                }
            );
        }

        // CSV export URL preserving filters
        $export_url = add_query_arg(
            [
                'action'      => 'shoplite_export_top_products',
                'prd_period'  => $period,
                'prd_sort'    => $sort,
                'prd_gateway' => $gateway,
            ],
            admin_url( 'admin-post.php' )
        );
        $export_url = wp_nonce_url(
            $export_url,
            'shoplite_export_top_products',
            '_wpnonce'
        );
        ?>

        <div class="shoplite-card">

            <div style="display:flex;align-items:center;justify-content:space-between;gap:10px;margin-bottom:10px;">
                <div>
                    <h2 style="margin:0;">
                        <?php esc_html_e( 'Top products', self::TD ); ?>
                    </h2>
                    <p class="shoplite-note" style="margin:4px 0 0 0;">
                        <?php
                        printf(
                            /* translators: 1: start date, 2: end date */
                            esc_html__( 'Products with the highest sales between %1$s and %2$s.', self::TD ),
                            esc_html( date_i18n( get_option( 'date_format' ), strtotime( $range_start . ' 00:00:00' ) ) ),
                            esc_html( date_i18n( get_option( 'date_format' ), strtotime( $range_end . ' 00:00:00' ) ) )
                        );
                        ?>
                    </p>
                </div>

                <?php if ( ! empty( $products ) ) : ?>
                    <a href="<?php echo esc_url( $export_url ); ?>" class="button button-secondary">
                        <?php esc_html_e( 'Export CSV', self::TD ); ?>
                    </a>
                <?php endif; ?>
            </div>

            <!-- Filters bar -->
            <form method="get" class="shoplite-filters-bar" style="margin-top:4px;margin-bottom:12px;">
                <input type="hidden" name="page" value="<?php echo esc_attr( self::MENU_SLUG ); ?>" />
                <input type="hidden" name="tab"  value="products" />

                <label for="shoplite-prd-period">
                    <?php esc_html_e( 'Period:', self::TD ); ?>
                </label>
                <select id="shoplite-prd-period" name="prd_period">
                    <?php foreach ( $periods as $days => $label ) : ?>
                        <option value="<?php echo esc_attr( $days ); ?>" <?php selected( $period, $days ); ?>>
                            <?php echo esc_html( $label ); ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <label for="shoplite-prd-sort">
                    <?php esc_html_e( 'Order by:', self::TD ); ?>
                </label>
                <select id="shoplite-prd-sort" name="prd_sort">
                    <?php foreach ( $sort_modes as $key => $label ) : ?>
                        <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $sort, $key ); ?>>
                            <?php echo esc_html( $label ); ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <label for="shoplite-prd-gateway">
                    <?php esc_html_e( 'Payment method:', self::TD ); ?>
                </label>
                <select id="shoplite-prd-gateway" name="prd_gateway">
                    <?php foreach ( $gateway_modes as $key => $label ) : ?>
                        <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $gateway, $key ); ?>>
                            <?php echo esc_html( $label ); ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <button type="submit" class="button button-primary">
                    <?php esc_html_e( 'Filter', self::TD ); ?>
                </button>
            </form>

            <table class="widefat striped">
                <thead>
                    <tr>
                        <th style="width:3em;">#</th>
                        <th><?php esc_html_e( 'Product', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Units', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Revenue', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Average price', self::TD ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ( empty( $products ) ) : ?>
                        <tr>
                            <td colspan="5">
                                <?php esc_html_e( 'There is no product data for the selected range.', self::TD ); ?>
                            </td>
                        </tr>
                    <?php else : ?>
                        <?php
                        $pos = 1;
                        foreach ( $products as $prod ) :
                            $name = isset( $prod['name'] ) ? $prod['name'] : '';
                            $qty  = isset( $prod['qty'] ) ? (int) $prod['qty'] : 0;
                            $rev  = isset( $prod['revenue'] ) ? (float) $prod['revenue'] : 0.0;
                            $avg  = $qty > 0 ? ( $rev / $qty ) : 0.0;
                            ?>
                            <tr>
                                <td><?php echo esc_html( $pos ); ?></td>
                                <td><?php echo esc_html( $name ); ?></td>
                                <td style="text-align:right;"><?php echo esc_html( number_format_i18n( $qty ) ); ?></td>
                                <td style="text-align:right;">
                                    <?php echo esc_html( number_format_i18n( $rev, 2 ) ) . ' ' . esc_html( $currency_symbol ); ?>
                                </td>
                                <td style="text-align:right;">
                                    <?php echo esc_html( number_format_i18n( $avg, 2 ) ) . ' ' . esc_html( $currency_symbol ); ?>
                                </td>
                            </tr>
                            <?php
                            $pos++;
                        endforeach;
                        ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <?php
    }


        /* ======================================================
     * TAB 3: TOP CUSTOMERS
     * ====================================================== */

    protected static function render_tab_top_customers() {

        $currency_symbol = get_option( 'shoplite_moneda_simbolo', '€' );
        $rows            = self::get_top_customers( 50 );

        // Secure URL to export TOP customers to CSV
        $export_top_customers_url = wp_nonce_url(
            admin_url( 'admin-post.php?action=shoplite_export_top_customers' ),
            'shoplite_export_top_customers',
            '_wpnonce'
        );
        ?>

        <div class="shoplite-card">

            <div style="display:flex;align-items:center;justify-content:space-between;gap:10px;margin-bottom:8px;">
                <h2 style="margin:0;">
                    <?php esc_html_e( 'Top customers', self::TD ); ?>
                </h2>

                <?php if ( ! empty( $rows ) ) : ?>
                    <a href="<?php echo esc_url( $export_top_customers_url ); ?>"
                       class="button button-secondary shoplite-pro-need-license">
                        <?php esc_html_e( 'Export CSV', self::TD ); ?>
                    </a>
                <?php endif; ?>
            </div>

            <p class="shoplite-note" style="margin-top:0;margin-bottom:10px;">
                <?php esc_html_e( 'You can export this list to CSV for internal analysis or to use it in your email marketing tool.', self::TD ); ?>
            </p>

            <table class="widefat striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'Customer', self::TD ); ?></th>
                        <th><?php esc_html_e( 'Email', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Orders', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Total spent', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Average order value', self::TD ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ( empty( $rows ) ) : ?>
                        <tr>
                            <td colspan="5">
                                <?php esc_html_e( 'There is no customer data yet.', self::TD ); ?>
                            </td>
                        </tr>
                    <?php else : ?>
                        <?php foreach ( $rows as $c ) : ?>
                            <tr>
                                <td><?php echo esc_html( $c['name'] ); ?></td>
                                <td><?php echo esc_html( $c['email'] ); ?></td>
                                <td style="text-align:right;"><?php echo esc_html( $c['orders'] ); ?></td>
                                <td style="text-align:right;">
                                    <?php echo esc_html( number_format_i18n( $c['spent'], 2 ) ) . ' ' . esc_html( $currency_symbol ); ?>
                                </td>
                                <td style="text-align:right;">
                                    <?php echo esc_html( number_format_i18n( $c['avg'], 2 ) ) . ' ' . esc_html( $currency_symbol ); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <?php
    }


    /* ======================================================
     * TAB 4: ALL CUSTOMERS
     * ====================================================== */

    protected static function render_tab_all_customers() {

        $currency_symbol = get_option( 'shoplite_moneda_simbolo', '€' );
        $rows            = self::get_all_customers();

        // Secure URL to export ALL customers to CSV
        $export_all_customers_url = wp_nonce_url(
            admin_url( 'admin-post.php?action=shoplite_export_all_customers' ),
            'shoplite_export_all_customers',
            '_wpnonce'
        );
        ?>

        <div class="shoplite-card">

            <div style="display:flex;align-items:center;justify-content:space-between;gap:10px;margin-bottom:8px;">
                <h2 style="margin:0;">
                    <?php esc_html_e( 'All customers', self::TD ); ?>
                </h2>

                <?php if ( ! empty( $rows ) ) : ?>
                    <a href="<?php echo esc_url( $export_all_customers_url ); ?>"
                       class="button button-secondary shoplite-pro-need-license">
                        <?php esc_html_e( 'Export CSV', self::TD ); ?>
                    </a>
                <?php endif; ?>
            </div>

            <p class="shoplite-note" style="margin-top:0;margin-bottom:10px;">
                <?php esc_html_e( 'Export the full customer list to CSV for internal use and to analyse your customer base.', self::TD ); ?>
            </p>

            <table class="widefat striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'Customer', self::TD ); ?></th>
                        <th><?php esc_html_e( 'Email', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Orders', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Total spent', self::TD ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ( empty( $rows ) ) : ?>
                        <tr>
                            <td colspan="4">
                                <?php esc_html_e( 'No customers found.', self::TD ); ?>
                            </td>
                        </tr>
                    <?php else : ?>
                        <?php foreach ( $rows as $c ) : ?>
                            <tr>
                                <td><?php echo esc_html( $c['name'] ); ?></td>
                                <td><?php echo esc_html( $c['email'] ); ?></td>
                                <td style="text-align:right;"><?php echo esc_html( $c['orders'] ); ?></td>
                                <td style="text-align:right;">
                                    <?php echo esc_html( number_format_i18n( $c['spent'], 2 ) ) . ' ' . esc_html( $currency_symbol ); ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

        <?php
    }


    /* ======================================================
     * TAB 5: PAYMENT METHODS (table + chart)
     * ====================================================== */
    protected static function render_tab_payment_methods() {

        $currency_symbol = get_option( 'shoplite_moneda_simbolo', '€' );
        $rows            = self::get_payment_methods_stats();

        // Prepare data for the chart
        $labels       = [];
        $revenue_data = [];

        if ( ! empty( $rows ) ) {
            foreach ( $rows as $row ) {
                $labels[]       = $row['label'];
                $revenue_data[] = (float) $row['revenue'];
            }
        }
        ?>
        <div class="shoplite-card">
            <h2 style="margin-top:0;">
                <?php esc_html_e( 'Payment methods', self::TD ); ?>
            </h2>
            <p style="font-size:12px;color:#6b7280;margin-top:0;">
                <?php esc_html_e( 'Summary of orders and revenue grouped by payment method.', self::TD ); ?>
            </p>

            <?php if ( empty( $rows ) ) : ?>

                <p><?php esc_html_e( 'There is no payment method data yet.', self::TD ); ?></p>

            <?php else : ?>

                <div style="display:flex;flex-wrap:wrap;gap:24px;align-items:flex-start;">

                    <!-- Column 1: chart -->
                    <div style="flex:1 1 260px;max-width:380px;">
                        <canvas id="shoplite-payments-chart" height="220"></canvas>
                    </div>

                    <!-- Column 2: table -->
                    <div style="flex:2 1 320px;min-width:280px;">
                        <table class="widefat striped">
                            <thead>
                                <tr>
                                    <th><?php esc_html_e( 'Method', self::TD ); ?></th>
                                    <th style="text-align:right;"><?php esc_html_e( 'Orders', self::TD ); ?></th>
                                    <th style="text-align:right;"><?php esc_html_e( 'Revenue', self::TD ); ?></th>
                                    <th style="text-align:right;"><?php esc_html_e( 'Average order value', self::TD ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $rows as $row ) : ?>
                                    <tr>
                                        <td><?php echo esc_html( $row['label'] ); ?></td>
                                        <td style="text-align:right;"><?php echo esc_html( $row['orders'] ); ?></td>
                                        <td style="text-align:right;">
                                            <?php
                                            echo esc_html(
                                                number_format_i18n( $row['revenue'], 2 )
                                            ) . ' ' . esc_html( $currency_symbol );
                                            ?>
                                        </td>
                                        <td style="text-align:right;">
                                            <?php
                                            echo esc_html(
                                                number_format_i18n( $row['avg'], 2 )
                                            ) . ' ' . esc_html( $currency_symbol );
                                            ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                </div><!-- /flex -->

                <!-- Chart script -->
                <script>
                document.addEventListener('DOMContentLoaded', function () {
                    const canvas = document.getElementById('shoplite-payments-chart');
                    if (!canvas || typeof Chart === 'undefined') {
                        return;
                    }

                    const data = {
                        labels: <?php echo wp_json_encode( $labels ); ?>,
                        datasets: [{
                            label: '<?php echo esc_js( __( 'Revenue by payment method', self::TD ) ); ?>',
                            data: <?php echo wp_json_encode( $revenue_data ); ?>,
                            backgroundColor: [
                                'rgba(59, 130, 246, 0.85)',
                                'rgba(16, 185, 129, 0.85)',
                                'rgba(251, 191, 36, 0.85)',
                                'rgba(239, 68, 68, 0.85)',
                                'rgba(139, 92, 246, 0.85)'
                            ],
                            borderWidth: 0
                        }]
                    };

                    const options = {
                        responsive: true,
                        plugins: {
                            legend: {
                                position: 'bottom',
                                labels: {
                                    boxWidth: 14,
                                    boxHeight: 14,
                                    usePointStyle: true
                                }
                            },
                            tooltip: {
                                callbacks: {
                                    label: function (context) {
                                        const value = context.parsed;
                                        return value.toLocaleString(undefined, {
                                            minimumFractionDigits: 2,
                                            maximumFractionDigits: 2
                                        }) + ' <?php echo esc_js( $currency_symbol ); ?>';
                                    }
                                }
                            }
                        },
                        cutout: '55%'
                    };

                    new Chart(canvas.getContext('2d'), {
                        type: 'doughnut',
                        data: data,
                        options: options
                    });
                });
                </script>

            <?php endif; ?>

        </div><!-- .shoplite-card -->
        <?php
    }


    /**
     * Payment method statistics.
     *
     * Returns an indexed array with:
     * [
     *   'method'  => 'stripe',
     *   'label'   => 'Stripe',
     *   'orders'  => 3,
     *   'revenue' => 120.50,
     *   'avg'     => 40.17,
     * ]
     */
    protected static function get_payment_methods_stats() {

        $args = [
            'post_type'      => 'shoplite_pedido',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'no_found_rows'  => true,
            'meta_query'     => [
                [
                    'key'   => '_shoplite_order_status',
                    'value' => 'completed',
                ],
            ],
        ];

        $orders = get_posts( $args );

        if ( empty( $orders ) ) {
            return [];
        }

        $methods = [];

        foreach ( $orders as $order ) {
            $order_id = $order->ID;

            // Try several possible meta keys for payment method
            $method = get_post_meta( $order_id, '_shoplite_order_payment_method', true );
            if ( ! $method ) {
                $method = get_post_meta( $order_id, '_shoplite_payment_method', true );
            }
            if ( ! $method ) {
                $method = get_post_meta( $order_id, '_shoplite_gateway', true );
            }

            if ( ! $method ) {
                $method = 'unknown';
            }

            $total_raw = get_post_meta( $order_id, '_shoplite_order_total', true );

            // Normalize decimal "41,05" or "41.05"
            $total_str = str_replace( ' ', '', (string) $total_raw );
            $total_str = str_replace( ',', '.', $total_str );
            $total     = is_numeric( $total_str ) ? (float) $total_str : 0.0;

            if ( ! isset( $methods[ $method ] ) ) {
                $methods[ $method ] = [
                    'method'  => $method,
                    'label'   => self::humanize_payment_method( $method ),
                    'orders'  => 0,
                    'revenue' => 0.0,
                    'avg'     => 0.0,
                ];
            }

            $methods[ $method ]['orders']++;
            $methods[ $method ]['revenue'] += $total;
        }

        // Average order value per method
        foreach ( $methods as $key => $data ) {
            if ( $data['orders'] > 0 ) {
                $methods[ $key ]['avg'] = $data['revenue'] / $data['orders'];
            }
        }

        // Sort by revenue descending
        usort(
            $methods,
            function ( $a, $b ) {
                return $b['revenue'] <=> $a['revenue'];
            }
        );

        return $methods;
    }

    /**
     * Convert a payment method slug to a human-readable label.
     */
    protected static function humanize_payment_method( $method ) {

        $method = strtolower( (string) $method );

        switch ( $method ) {
            case 'stripe':
            case 'stripe_card':
                return 'Stripe';

            case 'paypal':
            case 'pay_pal':
            case 'paypal_standard':
                return 'PayPal';

            case 'unknown':
            case '':
            default:
                return __( 'Unknown', self::TD );
        }
    }

    /**
     * Export TOP customers list to CSV.
     */
    public static function handle_export_top_customers_csv() {

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to export data.', self::TD ) );
        }

        // Verify nonce
        check_admin_referer( 'shoplite_export_top_customers' );

        if ( ! method_exists( __CLASS__, 'get_top_customers' ) ) {
            wp_die( esc_html__( 'The get_top_customers() function is not available.', self::TD ) );
        }

        // Top N customers (you can adjust the limit)
        $customers = self::get_top_customers( 200 );

        if ( ! is_array( $customers ) ) {
            $customers = [];
        }

        self::output_customers_csv( $customers, 'shoplite-top-customers' );
    }

    /**
     * Export ALL customers list to CSV.
     */
    public static function handle_export_all_customers_csv() {

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to export data.', self::TD ) );
        }

        // Verify nonce
        check_admin_referer( 'shoplite_export_all_customers' );

        if ( ! method_exists( __CLASS__, 'get_all_customers' ) ) {
            wp_die( esc_html__( 'The get_all_customers() function is not available.', self::TD ) );
        }

        $customers = self::get_all_customers();
        if ( ! is_array( $customers ) ) {
            $customers = [];
        }

        self::output_customers_csv( $customers, 'shoplite-all-customers' );
    }

    /**
     * Common helper: send a CSV to the browser from a customers array.
     */
    protected static function output_customers_csv( array $customers, $filename_prefix ) {

        if ( headers_sent() ) {
            wp_die( esc_html__( 'Headers have already been sent. The CSV cannot be generated.', self::TD ) );
        }

        $date     = gmdate( 'Ymd-His' );
        $filename = $filename_prefix . '-' . $date . '.csv';

        nocache_headers();
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename="' . $filename . '"' );

        $output = fopen( 'php://output', 'w' );

        if ( ! $output ) {
            wp_die( esc_html__( 'Could not open the output stream for the CSV.', self::TD ) );
        }

        // UTF-8 BOM so Excel displays characters correctly
        fprintf( $output, chr(0xEF) . chr(0xBB) . chr(0xBF) );

        // Column header
        fputcsv(
            $output,
            [
                'Email',
                'Name',
                'Orders',
                'Revenue',
                'First order',
                'Last order',
            ],
            ';'
        );

        foreach ( $customers as $row ) {

            $email = isset( $row['email'] ) ? (string) $row['email'] : '';
            $name  = isset( $row['name'] )  ? (string) $row['name']  : '';

            if ( isset( $row['orders_count'] ) ) {
                $orders = (int) $row['orders_count'];
            } elseif ( isset( $row['orders'] ) ) {
                $orders = (int) $row['orders'];
            } else {
                $orders = 0;
            }

            if ( isset( $row['revenue'] ) ) {
                $revenue = (float) $row['revenue'];
            } elseif ( isset( $row['spent'] ) ) {
                $revenue = (float) $row['spent'];
            } else {
                $revenue = 0.0;
            }

            $first = isset( $row['first_order'] ) ? (string) $row['first_order'] : '';
            $last  = isset( $row['last_order'] )  ? (string) $row['last_order']  : '';

            fputcsv(
                $output,
                [
                    $email,
                    $name,
                    $orders,
                    number_format( $revenue, 2, '.', '' ),
                    $first,
                    $last,
                ],
                ';'
            );
        }

        fclose( $output );
        exit;
    }

    /**
     * Lightweight internal CSS for tabs.
     */
    protected static function render_tabs_css() {
        ?>
        <style>
            .shoplite-pro-sales-tabs .nav-tab {
                font-size: 13px;
                padding: 6px 12px;
            }
            .shoplite-pro-sales-tabs .nav-tab-active {
                font-weight: 600;
            }
            .shoplite-pro-sales-content {
                margin-top: 18px;
            }
        </style>
        <?php
    }

    /**
     * Export TOP products list to CSV using current filters.
     */
    public static function handle_export_top_products_csv() {

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to export data.', self::TD ) );
        }

        check_admin_referer( 'shoplite_export_top_products' );

        $periods = [
            30  => __( 'Last 30 days', self::TD ),
            90  => __( 'Last 90 days', self::TD ),
            365 => __( 'Last 365 days', self::TD ),
            730 => __( 'Last 2 years', self::TD ),
        ];

        $period = isset( $_GET['prd_period'] ) ? (int) $_GET['prd_period'] : 90;
        if ( ! isset( $periods[ $period ] ) ) {
            $period = 90;
        }

        $sort = isset( $_GET['prd_sort'] ) ? sanitize_key( $_GET['prd_sort'] ) : 'revenue';
        if ( ! in_array( $sort, [ 'revenue', 'qty' ], true ) ) {
            $sort = 'revenue';
        }

        $gateway = isset( $_GET['prd_gateway'] ) ? sanitize_key( $_GET['prd_gateway'] ) : 'all';
        if ( ! in_array( $gateway, [ 'all', 'stripe', 'paypal' ], true ) ) {
            $gateway = 'all';
        }

        $now_ts     = current_time( 'timestamp' );
        $range_days = max( 1, (int) $period );
        $start_ts   = $now_ts - ( $range_days - 1 ) * DAY_IN_SECONDS;

        $range_start    = gmdate( 'Y-m-d', $start_ts );
        $range_end      = gmdate( 'Y-m-d', $now_ts );
        $gateway_filter = ( 'all' === $gateway ) ? null : $gateway;

        $products = self::get_top_products( $range_start, $range_end, 500, $gateway_filter );
        if ( ! is_array( $products ) ) {
            $products = [];
        }

        if ( 'qty' === $sort && ! empty( $products ) ) {
            usort(
                $products,
                function( $a, $b ) {
                    $qtyA = isset( $a['qty'] ) ? (int) $a['qty'] : 0;
                    $qtyB = isset( $b['qty'] ) ? (int) $b['qty'] : 0;
                    if ( $qtyA === $qtyB ) {
                        $revA = isset( $a['revenue'] ) ? (float) $a['revenue'] : 0.0;
                        $revB = isset( $b['revenue'] ) ? (float) $b['revenue'] : 0.0;
                        return $revB <=> $revA;
                    }
                    return $qtyB <=> $qtyA;
                }
            );
        }

        self::output_products_csv( $products, 'shoplite-top-products' );
    }

    /**
     * Helper: send a products CSV to the browser.
     */
    protected static function output_products_csv( array $products, $filename_prefix ) {

        if ( headers_sent() ) {
            wp_die( esc_html__( 'Headers have already been sent. The CSV cannot be generated.', self::TD ) );
        }

        $date     = gmdate( 'Ymd-His' );
        $filename = $filename_prefix . '-' . $date . '.csv';

        nocache_headers();
        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename="' . $filename . '"' );

        $output = fopen( 'php://output', 'w' );
        if ( ! $output ) {
            wp_die( esc_html__( 'Could not open the output stream for the CSV.', self::TD ) );
        }

        // UTF-8 BOM
        fprintf( $output, chr(0xEF) . chr(0xBB) . chr(0xBF) );

        // Column header
        fputcsv(
            $output,
            [ 'product_id', 'name', 'qty', 'revenue' ],
            ';'
        );

        foreach ( $products as $prod ) {
            $pid  = isset( $prod['product_id'] ) ? (int) $prod['product_id'] : 0;
            $name = isset( $prod['name'] )       ? (string) $prod['name']       : '';
            $qty  = isset( $prod['qty'] )        ? (int) $prod['qty']           : 0;
            $rev  = isset( $prod['revenue'] )    ? (float) $prod['revenue']     : 0.0;

            fputcsv(
                $output,
                [
                    $pid,
                    $name,
                    $qty,
                    number_format( $rev, 2, '.', '' ),
                ],
                ';'
            );
        }

        fclose( $output );
        exit;
    }
}