<?php
// shoplite-pro/includes/class-shoplite-pro-licensing.php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

error_log('SHOPLITE PRO LICENSING LOADED: ' . __FILE__);

class Shoplite_Pro_Licensing {

    const OPTION_KEY     = 'shoplite_pro_license_key';
    const OPTION_STATE   = 'shoplite_pro_license_state';   // active | error_* | inactive
    const OPTION_EXPIRES = 'shoplite_pro_license_expires';

    const API_BASE = 'https://talariasoft.com/api/shoplite-pro/';
    const TD       = 'shoplite-pro'; // Text domain (Pro)

    public static function init() {

        // 1) Menu only for administrators
        add_action( 'admin_menu', [ __CLASS__, 'menu' ], 20 );

        // 2) Handle activation
        add_action( 'admin_post_shoplite_pro_activate_license',   [ __CLASS__, 'handle_activate' ] );

        // 3) Handle deactivation
        add_action( 'admin_post_shoplite_pro_deactivate_license', [ __CLASS__, 'handle_deactivate' ] );
    }

    /**
     * Add a submenu under "Talariasoft Shoplite".
     */
    public static function menu() {

        // IMPORTANT: parent menu slug
        // In Shoplite the parent menu is always "shoplite"
        $parent_slug = 'shoplite';

        add_submenu_page(
            $parent_slug,
            __( 'Shoplite Pro license', self::TD ),
            __( 'License', self::TD ),
            'manage_options',
            'shoplite-pro-license',
            [ __CLASS__, 'render' ]
        );
    }

    /**
     * Render the license page.
     */
    public static function render() {

        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $key     = (string) get_option( self::OPTION_KEY, '' );
        $state   = (string) get_option( self::OPTION_STATE, 'inactive' );
        $expires = (string) get_option( self::OPTION_EXPIRES, '' );
        $code    = isset( $_GET['slpro'] ) ? sanitize_text_field( wp_unslash( $_GET['slpro'] ) ) : '';

        ?>
        <div class="wrap shoplite-pro-licensing">

            <?php
            // Talariasoft + Shoplite brand header
            if ( function_exists( 'shoplite_render_brand_header' ) ) {
                shoplite_render_brand_header();
            }
            ?>
            <h1><?php esc_html_e( 'Shoplite Pro license', self::TD ); ?></h1>

            <p>
                <?php
                esc_html_e(
                    'Activate your license to receive updates, premium features and technical support.',
                    self::TD
                );
                ?>
            </p>

            <?php self::render_notices( $state, $code, $expires ); ?>

            <h2><?php esc_html_e( 'Activate license', self::TD ); ?></h2>

            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                <?php wp_nonce_field( 'shoplite_pro_license' ); ?>
                <input type="hidden" name="action" value="shoplite_pro_activate_license" />

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row">
                            <label for="shoplite_pro_license_key">
                                <?php esc_html_e( 'License key', self::TD ); ?>
                            </label>
                        </th>
                        <td>
                            <input type="text"
                                   id="shoplite_pro_license_key"
                                   name="license_key"
                                   class="regular-text"
                                   placeholder="SLPRO-XXXX-XXXX-XXXX"
                                   value="<?php echo esc_attr( $key ); ?>" />

                            <p class="description">
                                <?php
                                esc_html_e(
                                    'Paste the key you received when you purchased Shoplite Pro.',
                                    self::TD
                                );
                                ?>
                            </p>

                            <?php if ( self::is_active() && $key ) : ?>
                                <p class="description">
                                    <strong><?php esc_html_e( 'Currently active on this site.', self::TD ); ?></strong>
                                    <?php if ( $expires ) : ?>
                                        <?php echo ' '; ?>
                                        <?php
                                        printf(
                                            /* translators: %s: expiration date */
                                            esc_html__( 'Expires: %s', self::TD ),
                                            esc_html( $expires )
                                        );
                                        ?>
                                    <?php endif; ?>
                                </p>
                                <p class="description">
                                    <?php esc_html_e( 'To use another key, deactivate the current license first.', self::TD ); ?>
                                </p>
                            <?php endif; ?>
                        </td>
                    </tr>
                </table>

                <?php submit_button( __( 'Activate license', self::TD ) ); ?>
            </form>

           
            <?php if ( self::is_active() ) : ?>
                <hr />

                <h2><?php esc_html_e( 'Deactivate license on this site', self::TD ); ?></h2>

                <p><?php esc_html_e( 'Use this if you want to move your license to another domain.', self::TD ); ?></p>

                <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                    <?php wp_nonce_field( 'shoplite_pro_license_deactivate' ); ?>
                    <input type="hidden" name="action" value="shoplite_pro_deactivate_license" />
                    <?php submit_button(
                        __( 'Deactivate license on this domain', self::TD ),
                        'secondary'
                    ); ?>
                </form>

            <?php endif; ?>

        </div>
        <?php
    }

    /**
     * Show notices on screen according to state/code.
     */
    private static function render_notices( $state, $code, $expires ) {

        $msg  = '';
        $type = 'info';

        switch ( true ) {

            case ( $state === 'active' ):
                $type = 'success';
                if ( $expires ) {
                    /* translators: %s: license expiration date. */
                    $msg = sprintf(
                        __( 'Your license is active. It expires on %s.', self::TD ),
                        esc_html( $expires )
                    );
                } else {
                    $msg = __( 'Your license is active.', self::TD );
                }
                break;

            case ( $state === 'error_empty' || $code === 'empty' ):
                $type = 'error';
                $msg  = __( 'You must enter a license key.', self::TD );
                break;

            case ( $state === 'error_invalid' || $code === 'invalid' ):
                $type = 'error';
                $msg  = __( 'The license key is not valid. Please check for typos.', self::TD );
                break;

            case ( $state === 'error_expired' || $code === 'expired' ):
                $type = 'error';
                $msg  = __( 'Your license has expired. Renew it to keep receiving updates.', self::TD );
                break;

            case ( $state === 'error_domain_limit' || $code === 'domain_limit' || $code === 'sites_limit' ):
                $type = 'error';
                $msg  = __( 'You have reached the maximum number of domains allowed by your license.', self::TD );
                break;

            case ( $state === 'error_not_activated_for_domain' || $code === 'not_activated_for_domain' ):
                $type = 'error';
                $msg  = __( 'This license is not activated for this domain. Activate it here first.', self::TD );
                break;

            case ( $state === 'error_http' || $code === 'http_error' ):
                $type = 'error';
                $msg  = __( 'Could not connect to the licensing server. Please try again later.', self::TD );
                break;

            case ( $state === 'error_response' || $code === 'bad_response' ):
                $type = 'error';
                $msg  = __( 'Unexpected response from the licensing server.', self::TD );
                break;

            case ( $code === 'deactivated' ):
                $type = 'success';
                $msg  = __( 'The license has been successfully deactivated on this domain.', self::TD );
                break;

            case ( $code === 'deactivate_failed' ):
                $type = 'error';
                $msg  = __( 'Could not deactivate the license on the server. Please try again.', self::TD );
                break;

            case ( $code === 'ok' ):
                $type = 'success';
                $msg  = __( 'License activated successfully.', self::TD );
                break;
        }

        if ( ! $msg ) {
            return;
        }

        printf(
            '<div class="notice notice-%s"><p>%s</p></div>',
            esc_attr( $type ),
            esc_html( $msg )
        );
    }

    /**
     * Clear local license state (always).
     */
    private static function clear_local_license_state() : void {
        delete_option( self::OPTION_KEY );
        delete_option( self::OPTION_EXPIRES );
        update_option( self::OPTION_STATE, 'inactive' );
    }

    /**
     * Handle license activation.
     */
    public static function handle_activate() {

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have sufficient permissions.', self::TD ) );
        }

        check_admin_referer( 'shoplite_pro_license' );

        $key = isset( $_POST['license_key'] ) ? sanitize_text_field( wp_unslash( $_POST['license_key'] ) ) : '';

        if ( empty( $key ) ) {
            update_option( self::OPTION_STATE, 'error_empty' );
            wp_redirect( admin_url( 'admin.php?page=shoplite-pro-license&slpro=empty' ) );
            exit;
        }

        // If the user changed the key, clear previous expires to avoid stale UI.
        $current_key = (string) get_option( self::OPTION_KEY, '' );
        if ( $current_key && $current_key !== $key ) {
            delete_option( self::OPTION_EXPIRES );
            update_option( self::OPTION_STATE, 'inactive' );
        }

        $domain = home_url();

        $response = wp_remote_post(
            self::API_BASE . 'activate.php',
            [
                'timeout' => 20,
                'headers' => [ 'Content-Type' => 'application/json' ],
                'body'    => wp_json_encode(
                    [
                        'license_key' => $key,
                        'domain'      => $domain,
                    ]
                ),
            ]
        );

        if ( is_wp_error( $response ) ) {
            update_option( self::OPTION_STATE, 'error_http' );
            wp_redirect( admin_url( 'admin.php?page=shoplite-pro-license&slpro=http_error' ) );
            exit;
        }

        $http_code = (int) wp_remote_retrieve_response_code( $response );
        $raw_body  = (string) wp_remote_retrieve_body( $response );
        $body      = json_decode( $raw_body, true );

        if ( 200 !== $http_code || ! is_array( $body ) ) {
            update_option( self::OPTION_STATE, 'error_response' );
            wp_redirect( admin_url( 'admin.php?page=shoplite-pro-license&slpro=bad_response' ) );
            exit;
        }

        if ( empty( $body['ok'] ) ) {
            $error = isset( $body['error'] ) ? (string) $body['error'] : 'invalid';

            // Normalize possible errors coming from the server
            if ( $error === 'sites_limit' ) {
                $error = 'domain_limit';
            }

            update_option( self::OPTION_STATE, 'error_' . sanitize_key( $error ) );
            wp_redirect( admin_url( 'admin.php?page=shoplite-pro-license&slpro=' . urlencode( $error ) ) );
            exit;
        }

        $expires = isset( $body['expires'] ) ? (string) $body['expires'] : '';

        update_option( self::OPTION_KEY,   $key );
        update_option( self::OPTION_STATE, 'active' );

        // Always update expires (clean stale value if empty)
        if ( $expires ) {
            update_option( self::OPTION_EXPIRES, $expires );
        } else {
            delete_option( self::OPTION_EXPIRES );
        }

        wp_redirect( admin_url( 'admin.php?page=shoplite-pro-license&slpro=ok' ) );
        exit;
    }

    /**
     * Handle license deactivation.
     * NOTE: Always clear local state to immediately disable Pro on this site.
     * The server call is best-effort to release the domain.
     */
    public static function handle_deactivate() {

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'You do not have sufficient permissions.', self::TD ) );
        }

        check_admin_referer( 'shoplite_pro_license_deactivate' );

        $key = (string) get_option( self::OPTION_KEY, '' );

        if ( empty( $key ) ) {
            // Already inactive locally
            self::clear_local_license_state();

            wp_redirect( admin_url( 'admin.php?page=shoplite-pro-license&slpro=deactivated' ) );
            exit;
        }

        $domain = home_url();

        $response = wp_remote_post(
    self::API_BASE . 'deactivate.php',
    [
        'timeout' => 20,
        'headers' => [ 'Content-Type' => 'application/json' ],
        'body'    => wp_json_encode(
            [
                'license_key' => $key,
                'domain'      => $domain,
            ]
        ),
    ]
);


        if ( is_wp_error( $response ) ) {
            // ✅ Clear local state anyway
            self::clear_local_license_state();
            wp_redirect( admin_url( 'admin.php?page=shoplite-pro-license&slpro=deactivate_failed' ) );
            exit;
        }

        $http_code = (int) wp_remote_retrieve_response_code( $response );
        $raw_body  = (string) wp_remote_retrieve_body( $response );
        $body      = json_decode( $raw_body, true );

        if ( 200 !== $http_code || ! is_array( $body ) || empty( $body['ok'] ) ) {
            // ✅ Clear local state anyway
            self::clear_local_license_state();
            wp_redirect( admin_url( 'admin.php?page=shoplite-pro-license&slpro=deactivate_failed' ) );
            exit;
        }

        // Server confirmed: clear local state
        self::clear_local_license_state();

        wp_redirect( admin_url( 'admin.php?page=shoplite-pro-license&slpro=deactivated' ) );
        exit;
    }

    /**
     * Check if the license is active.
     */
    public static function is_active() : bool {
        return (string) get_option( self::OPTION_STATE, 'inactive' ) === 'active';
    }
}
