<?php
// wp-content/plugins/shoplite-pro/includes/class-shoplite-pro-evolution.php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Shoplite_Pro_Evolution {

    const MENU_SLUG     = 'shoplite_pro_evolution';
    const DEFAULT_RANGE = 30; // default days
    const TD            = 'shoplite-pro';

    /**
     * Initial hook.
     */
    public static function init() {
        add_action( 'admin_menu', [ __CLASS__, 'register_menu' ], 30 );
    }

    /**
     * Register the "Time-based evolution" submenu.
     */
    public static function register_menu() {
        add_submenu_page(
            'shoplite',
            __( 'Sales evolution (Pro)', self::TD ),
            __( 'Time-based evolution', self::TD ),
            'manage_options',
            self::MENU_SLUG,
            [ __CLASS__, 'render_page' ]
        );
    }

    /**
     * Main page for time-based evolution.
     */
    public static function render_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Allowed ranges (in days)
        $ranges = [
            7   => __( 'Last 7 days', self::TD ),
            30  => __( 'Last 30 days', self::TD ),
            60  => __( 'Last 60 days', self::TD ),
            90  => __( 'Last 90 days', self::TD ),
            365 => __( 'Last 365 days', self::TD ),
            730 => __( 'Last 730 days', self::TD ),
        ];

        // Grouping modes
        $group_modes = [
            'day'   => __( 'Day by day', self::TD ),
            'week'  => __( 'Week by week', self::TD ),
            'month' => __( 'Month by month', self::TD ),
        ];

        // Read range from URL (?range=...)
        $range = isset( $_GET['range'] ) ? (int) $_GET['range'] : self::DEFAULT_RANGE;
        if ( ! array_key_exists( $range, $ranges ) ) {
            $range = self::DEFAULT_RANGE;
        }

        // Read grouping from URL (?group=...)
        $group = isset( $_GET['group'] ) ? sanitize_key( $_GET['group'] ) : 'day';
        if ( ! array_key_exists( $group, $group_modes ) ) {
            $group = 'day';
        }

        // URL to export CSV with the same filters
        $export_url = add_query_arg(
            [
                'page'            => self::MENU_SLUG,
                'shoplite_export' => 'csv',
                'range'           => $range,
                'group'           => $group,
                '_wpnonce'        => wp_create_nonce( 'shoplite_export_csv' ),
            ],
            admin_url( 'admin.php' )
        );

        // Get stats according to grouping
        switch ( $group ) {
            case 'week':
                $rows = self::get_weekly_stats( $range );
                break;
            case 'month':
                $rows = self::get_monthly_stats( $range );
                break;
            case 'day':
            default:
                $rows = self::get_daily_stats( $range );
                break;
        }

        // Prepare data for the chart (chronological: oldest → newest)
        // $rows is expected from newest to oldest, so we reverse it
        $chart_source = array_reverse( $rows );

        $labels   = [];
        $revenues = [];
        $orders   = [];

        foreach ( $chart_source as $row ) {
            switch ( $group ) {
                case 'day':
                    $ts       = strtotime( $row['date'] . ' 00:00:00' );
                    $labels[] = date_i18n( get_option( 'date_format' ), $ts );
                    break;

                case 'week':
                    $start_ts = strtotime( $row['start'] . ' 00:00:00' );
                    $end_ts   = strtotime( $row['end'] . ' 00:00:00' );
                    $week_no  = date_i18n( 'W', $start_ts );
                    $label    = sprintf(
                        /* translators: 1: week number, 2: start date, 3: end date */
                        __( 'Week %1$s (%2$s – %3$s)', self::TD ),
                        $week_no,
                        date_i18n( 'j M', $start_ts ),
                        date_i18n( 'j M', $end_ts )
                    );
                    $labels[] = $label;
                    break;

                case 'month':
                    $start_ts = strtotime( $row['start'] . ' 00:00:00' );
                    $labels[] = date_i18n( 'M Y', $start_ts );
                    break;
            }

            $revenues[] = (float) $row['revenue'];
            $orders[]   = (int) $row['orders'];
        }

        // Currency symbol
        $currency_symbol = get_option( 'shoplite_moneda_simbolo', '€' );

        // JSON for JS
        $js_labels  = wp_json_encode( $labels );
        $js_revenue = wp_json_encode( $revenues );
        $js_orders  = wp_json_encode( $orders );

        // =========================================
        // Top products in the selected range
        // =========================================

        // Sorting selector for the top products (GET)
        $top_orderby = isset( $_GET['top_orderby'] ) ? sanitize_key( $_GET['top_orderby'] ) : 'revenue';
        if ( ! in_array( $top_orderby, [ 'revenue', 'qty' ], true ) ) {
            $top_orderby = 'revenue';
        }

        $top_products        = [];
        $total_range_revenue = 0.0;
        $best_product_id     = 0;   // To mark the "Best seller" chip

        // Compute the real date range (today backwards "range" days)
        $now_ts     = current_time( 'timestamp' );
        $range_days = max( 1, (int) $range );
        $start_ts   = $now_ts - ( $range_days - 1 ) * DAY_IN_SECONDS;

        $range_start = gmdate( 'Y-m-d', $start_ts );
        $range_end   = gmdate( 'Y-m-d', $now_ts );

        if ( class_exists( 'Shoplite_Pro_Sales' ) && method_exists( 'Shoplite_Pro_Sales', 'get_top_products' ) ) {

            // Request more than 5 products, then slice
            $top_products = Shoplite_Pro_Sales::get_top_products( $range_start, $range_end, 50 );

            if ( ! empty( $top_products ) && is_array( $top_products ) ) {

                // Total revenue in the range (for percentages)
                $total_range_revenue = array_sum( wp_list_pluck( $top_products, 'revenue' ) );

                // Sort according to the selector: by revenue or by units
                usort(
                    $top_products,
                    function( $a, $b ) use ( $top_orderby ) {
                        $qtyA = isset( $a['qty'] ) ? (int) $a['qty'] : 0;
                        $qtyB = isset( $b['qty'] ) ? (int) $b['qty'] : 0;
                        $revA = isset( $a['revenue'] ) ? (float) $a['revenue'] : 0.0;
                        $revB = isset( $b['revenue'] ) ? (float) $b['revenue'] : 0.0;

                        if ( 'qty' === $top_orderby ) {
                            // First by units, tie-breaker by revenue
                            if ( $qtyA === $qtyB ) {
                                return $revB <=> $revA;
                            }
                            return $qtyB <=> $qtyA;
                        }

                        // Default: by revenue, tie-breaker by units
                        if ( $revA === $revB ) {
                            return $qtyB <=> $qtyA;
                        }
                        return $revB <=> $revA;
                    }
                );

                // Keep only Top 5 products
                $top_products = array_slice( $top_products, 0, 5 );

                // Detect the top product for the chip
                $best_main      = -1;
                $best_secondary = -1;

                foreach ( $top_products as $prod_tmp ) {
                    $pid_tmp = isset( $prod_tmp['product_id'] ) ? (int) $prod_tmp['product_id'] : 0;
                    $qty_tmp = isset( $prod_tmp['qty'] ) ? (int) $prod_tmp['qty'] : 0;
                    $rev_tmp = isset( $prod_tmp['revenue'] ) ? (float) $prod_tmp['revenue'] : 0.0;

                    $metric    = ( 'qty' === $top_orderby ) ? $qty_tmp : $rev_tmp;
                    $secondary = ( 'qty' === $top_orderby ) ? $rev_tmp : $qty_tmp;

                    if ( $metric > $best_main || ( $metric === $best_main && $secondary > $best_secondary ) ) {
                        $best_main       = $metric;
                        $best_secondary  = $secondary;
                        $best_product_id = $pid_tmp;
                    }
                }
            }
        }
        ?>
        <div class="wrap shoplite-pro-evolution">
            <?php
            // Global Pro brand header
            if ( function_exists( 'shoplite_render_brand_header' ) ) {
                shoplite_render_brand_header();
            }
            ?>
            <h1><?php esc_html_e( 'Time-based sales evolution', self::TD ); ?></h1>

            <p style="color:#4b5563;margin-top:4px;margin-bottom:10px;font-size:13px;">
                <?php esc_html_e(
                    'Summary of revenue and orders aggregated by day, week or month based on your completed orders.',
                    self::TD
                ); ?>
            </p>

            <form method="get" style="margin-bottom:16px;display:flex;flex-wrap:wrap;align-items:center;gap:8px;">
                <input type="hidden" name="page" value="<?php echo esc_attr( self::MENU_SLUG ); ?>" />

                <label for="shoplite-range-select">
                    <?php esc_html_e( 'Period:', self::TD ); ?>
                </label>
                <select id="shoplite-range-select" name="range">
                    <?php foreach ( $ranges as $days => $label ) : ?>
                        <option value="<?php echo esc_attr( $days ); ?>" <?php selected( $range, $days ); ?>>
                            <?php echo esc_html( $label ); ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <label for="shoplite-group-select">
                    <?php esc_html_e( 'Group by:', self::TD ); ?>
                </label>
                <select id="shoplite-group-select" name="group">
                    <?php foreach ( $group_modes as $key => $label ) : ?>
                        <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $group, $key ); ?>>
                            <?php echo esc_html( $label ); ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <!-- Sorting selector for the Top products panel -->
                <label for="shoplite-top-orderby">
                    <?php esc_html_e( 'Order top by:', self::TD ); ?>
                </label>
                <select id="shoplite-top-orderby" name="top_orderby">
                    <option value="revenue" <?php selected( $top_orderby, 'revenue' ); ?>>
                        <?php esc_html_e( 'Revenue', self::TD ); ?>
                    </option>
                    <option value="qty" <?php selected( $top_orderby, 'qty' ); ?>>
                        <?php esc_html_e( 'Units sold', self::TD ); ?>
                    </option>
                </select>

                <button type="submit" class="button button-primary">
                    <?php esc_html_e( 'Update', self::TD ); ?>
                </button>

                <!-- Export CSV button -->
                <a href="<?php echo esc_url( $export_url ); ?>" class="button">
                    <?php esc_html_e( 'Export CSV', self::TD ); ?>
                </a>
            </form>

            <h2 style="margin-top:0;">
                <?php
                echo esc_html( $ranges[ $range ] . ' · ' . $group_modes[ $group ] );
                ?>
            </h2>

            <?php if ( ! empty( $rows ) ) : ?>
                <div class="shoplite-pro-card shoplite-pro-card--chart" style="max-width:1100px;margin-bottom:20px;background:#ffffff;border-radius:16px;padding:14px 16px;box-shadow:0 10px 28px rgba(15,23,42,0.06);border:1px solid rgba(148,163,184,0.4);">
                    <h3 class="shoplite-pro-card__title" style="margin-top:0;font-size:15px;margin-bottom:6px;">
                        <?php esc_html_e( 'Revenue and orders chart', self::TD ); ?>
                    </h3>
                    <p class="shoplite-pro-card__subtitle" style="margin-top:0;margin-bottom:10px;font-size:12px;color:#6b7280;">
                        <?php esc_html_e( 'Main line: revenue. Dashed line: number of orders.', self::TD ); ?>
                    </p>
                    <canvas id="shoplite-evolution-chart" height="90"></canvas>
                </div>
            <?php endif; ?>





    <?php
// Total revenue in the range (for percentage calculations)
if ( ! empty( $top_products ) && is_array( $top_products ) ) {
    $total_range_revenue = array_sum( wp_list_pluck( $top_products, 'revenue' ) );
} else {
    $total_range_revenue = 0;
}

// Variables to detect the top product
$best_revenue = -1;
$best_qty     = -1;

if ( ! empty( $top_products ) && is_array( $top_products ) ) {

    foreach ( $top_products as $prod_tmp ) {
        $rev_tmp = isset( $prod_tmp['revenue'] ) ? (float) $prod_tmp['revenue'] : 0.0;
        $qty_tmp = isset( $prod_tmp['qty'] ) ? (int) $prod_tmp['qty'] : 0;

        // Choose the product with the highest revenue,
        // and if tied, the one with the most units
        if ( $rev_tmp > $best_revenue || ( $rev_tmp === $best_revenue && $qty_tmp > $best_qty ) ) {
            $best_revenue = $rev_tmp;
            $best_qty     = $qty_tmp;
        }
    }
}
?>

        <table class="widefat fixed striped" style="max-width:1100px;">
            <thead>
                <tr>
                    <th>
                        <?php
                        switch ( $group ) {
                            case 'week':
                                esc_html_e( 'Week', self::TD );
                                break;
                            case 'month':
                                esc_html_e( 'Month', self::TD );
                                break;
                            case 'day':
                            default:
                                esc_html_e( 'Date', self::TD );
                                break;
                        }
                        ?>
                    </th>
                    <th style="text-align:right;"><?php esc_html_e( 'Completed orders', self::TD ); ?></th>
                    <th style="text-align:right;"><?php esc_html_e( 'Period revenue', self::TD ); ?></th>
                    <th style="text-align:right;"><?php esc_html_e( 'Average order value', self::TD ); ?></th>
                </tr>
            </thead>
            <tbody>
            <?php if ( empty( $rows ) ) : ?>
                <tr>
                    <td colspan="4">
                        <?php esc_html_e( 'There is no order data for the selected range.', self::TD ); ?>
                    </td>
                </tr>
            <?php else : ?>
                <?php foreach ( $rows as $row ) : ?>
                    <tr>
                        <td>
                            <?php
                            if ( 'day' === $group ) {
                                $ts = strtotime( $row['date'] . ' 00:00:00' );
                                echo esc_html( date_i18n( get_option( 'date_format' ), $ts ) );
                            } elseif ( 'week' === $group ) {
                                $start_ts = strtotime( $row['start'] . ' 00:00:00' );
                                $end_ts   = strtotime( $row['end'] . ' 00:00:00' );
                                $week_no  = date_i18n( 'W', $start_ts );

                                printf(
                                    '<strong>%s</strong><br><span style="font-size:11px;color:#6b7280;">%s – %s</span>',
                                    esc_html( sprintf( __( 'Week %s', self::TD ), $week_no ) ),
                                    esc_html( date_i18n( get_option( 'date_format' ), $start_ts ) ),
                                    esc_html( date_i18n( get_option( 'date_format' ), $end_ts ) )
                                );
                            } else { // month
                                $start_ts = strtotime( $row['start'] . ' 00:00:00' );
                                echo esc_html( date_i18n( 'F Y', $start_ts ) );
                            }
                            ?>
                        </td>
                        <td style="text-align:right;">
                            <?php echo esc_html( number_format_i18n( $row['orders'] ) ); ?>
                        </td>
                        <td style="text-align:right;">
                            <?php
                            echo esc_html( number_format_i18n( $row['revenue'], 2 ) );
                            echo ' ' . esc_html( $currency_symbol );
                            ?>
                        </td>
                        <td style="text-align:right;">
                            <?php
                            echo esc_html( number_format_i18n( $row['avg'], 2 ) );
                            echo ' ' . esc_html( $currency_symbol );
                            ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
            </tbody>
        </table>

        <p style="margin-top:10px;color:#6b7280;font-size:12px;max-width:1100px;">
            <?php esc_html_e( 'Data is calculated from orders with status "completed" and post type "shoplite_pedido".', self::TD ); ?>
        </p>

        <!-- Panel: Top products for the selected range -->
        <div class="shoplite-pro-card shoplite-pro-card--tops" style="max-width:1100px;margin-top:22px;background:#ffffff;border-radius:16px;padding:14px 16px;box-shadow:0 10px 28px rgba(15,23,42,0.06);border:1px solid rgba(148,163,184,0.4);">
            <h3 class="shoplite-pro-card__title" style="margin-top:0;font-size:15px;margin-bottom:6px;">
                <?php esc_html_e( 'Top products in the selected period', self::TD ); ?>
            </h3>
            <p style="margin-top:0;margin-bottom:10px;font-size:12px;color:#6b7280;">
                <?php
                printf(
                    /* translators: 1: start date, 2: end date */
                    esc_html__( 'Products with the highest revenue between %1$s and %2$s.', self::TD ),
                    esc_html( date_i18n( get_option( 'date_format' ), strtotime( $range_start . ' 00:00:00' ) ) ),
                    esc_html( date_i18n( get_option( 'date_format' ), strtotime( $range_end . ' 00:00:00' ) ) )
                );
                ?>
            </p>

            <table class="widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width:3em;">#</th>
                        <th><?php esc_html_e( 'Product', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Units sold', self::TD ); ?></th>
                        <th style="text-align:right;"><?php esc_html_e( 'Revenue', self::TD ); ?></th>
                    </tr>
                </thead>
                <tbody>
                <?php if ( empty( $top_products ) ) : ?>
                    <tr>
                        <td colspan="4">
                            <?php esc_html_e( 'There are no sales registered in this range yet, or line items are not available.', self::TD ); ?>
                        </td>
                    </tr>
                <?php else : ?>
    <?php
    $pos = 1;

    foreach ( $top_products as $prod ) :
        $name = isset( $prod['name'] ) ? $prod['name'] : '';
        $qty  = isset( $prod['qty'] ) ? (int) $prod['qty'] : 0;
        $rev  = isset( $prod['revenue'] ) ? (float) $prod['revenue'] : 0.0;

        // Percentage of this product over the total revenue in the range
        $percent = ( $total_range_revenue > 0 )
            ? ( $rev * 100 / $total_range_revenue )
            : 0;

        // Is this the "top" product? (highest revenue and, if tied, most units)
        $is_best = ( $rev === $best_revenue && $qty === $best_qty );
        ?>
        <tr>
            <td>
                <?php echo esc_html( $pos ); ?>
            </td>
            <td>
                <?php echo esc_html( $name ); ?>
                <?php if ( $is_best ) : ?>
                    <span style="display:inline-block;margin-left:6px;padding:1px 6px;border-radius:9999px;font-size:10px;background:#0f766e;color:#fff;">
                        <?php esc_html_e( 'Best seller', self::TD ); ?>
                    </span>
                <?php endif; ?>
            </td>
            <td style="text-align:right;">
                <?php echo esc_html( number_format_i18n( $qty ) ); ?>
            </td>
            <td style="text-align:right;">
                <?php
                echo esc_html( number_format_i18n( $rev, 2 ) ) . ' ' . esc_html( $currency_symbol );
                if ( $percent > 0 ) {
                    echo ' ';
                    printf(
                        '<span style="font-size:11px;color:#6b7280;">(%.1f%%)</span>',
                        $percent
                    );
                }
                ?>
            </td>
        </tr>
        <?php
        $pos++;
    endforeach;
    ?>
<?php endif; ?>
                </tbody>
            </table>

        </div>

    </div><!-- /.wrap -->

    <?php if ( ! empty( $rows ) ) : ?>
        <!-- Chart.js from CDN -->
        <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
        <script>
            (function() {
                document.addEventListener('DOMContentLoaded', function () {
                    var canvas = document.getElementById('shoplite-evolution-chart');
                    if (!canvas || typeof Chart === 'undefined') {
                        return;
                    }

                    var labels  = <?php echo $js_labels; ?>;
                    var revenue = <?php echo $js_revenue; ?>;
                    var orders  = <?php echo $js_orders; ?>;

                    var ctx = canvas.getContext('2d');

                    new Chart(ctx, {
                        type: 'line',
                        data: {
                            labels: labels,
                            datasets: [
                                {
                                    label: '<?php echo esc_js( __( 'Revenue', self::TD ) . ' (' . $currency_symbol . ')' ); ?>',
                                    data: revenue,
                                    yAxisID: 'y',
                                    tension: 0.3,
                                    borderWidth: 2,
                                    pointRadius: 2
                                },
                                {
                                    label: '<?php echo esc_js( __( 'Orders', self::TD ) ); ?>',
                                    data: orders,
                                    yAxisID: 'y1',
                                    tension: 0.3,
                                    borderWidth: 2,
                                    pointRadius: 2,
                                    borderDash: [5, 5]
                                }
                            ]
                        },
                        options: {
                            responsive: true,
                            interaction: {
                                mode: 'index',
                                intersect: false
                            },
                            stacked: false,
                            plugins: {
                                legend: {
                                    position: 'top'
                                },
                                tooltip: {
                                    callbacks: {
                                        label: function(context) {
                                            var label = context.dataset.label || '';
                                            if (label) {
                                                label += ': ';
                                            }
                                            if (context.dataset.yAxisID === 'y') {
                                                label += context.formattedValue + ' <?php echo esc_js( $currency_symbol ); ?>';
                                            } else {
                                                label += context.formattedValue;
                                            }
                                            return label;
                                        }
                                    }
                                }
                            },
                            scales: {
                                y: {
                                    type: 'linear',
                                    position: 'left',
                                    beginAtZero: true
                                },
                                y1: {
                                    type: 'linear',
                                    position: 'right',
                                    beginAtZero: true,
                                    grid: {
                                        drawOnChartArea: false
                                    }
                                }
                            }
                        }
                    });
                });
            })();
        </script>
    <?php endif; ?>
    <?php
}


    /**
     * Daily statistics for the last $days days.
     *
     * @param int $days
     * @return array[] Each row: [ 'date' => 'Y-m-d', 'orders' => int, 'revenue' => float, 'avg' => float ]
     */
    protected static function get_daily_stats( $days ) {
        global $wpdb;

        $days = max( 1, (int) $days );

        // Shoplite orders configuration
        $post_type        = 'shoplite_pedido';
        $meta_key_total   = '_shoplite_order_total';
        $meta_key_status  = '_shoplite_order_status';
        $completed_status = 'completed';

        $posts_table    = $wpdb->posts;
        $postmeta_table = $wpdb->postmeta;

        // Start date (start of the day $days-1 days ago)
        $today_ts   = current_time( 'timestamp' ); // WP timezone
        $start_ts   = $today_ts - ( $days - 1 ) * DAY_IN_SECONDS;
        $start_date = gmdate( 'Y-m-d 00:00:00', $start_ts );

        $sql = "
            SELECT 
                DATE(p.post_date) AS order_date,
                COUNT(DISTINCT p.ID) AS orders_count,
                COALESCE(SUM(CAST(pm_total.meta_value AS DECIMAL(10,2))), 0) AS revenue
            FROM {$posts_table} p
            INNER JOIN {$postmeta_table} pm_total
                ON pm_total.post_id = p.ID
               AND pm_total.meta_key = %s
            INNER JOIN {$postmeta_table} pm_status
                ON pm_status.post_id = p.ID
               AND pm_status.meta_key = %s
               AND pm_status.meta_value = %s
            WHERE p.post_type = %s
              AND p.post_status = 'publish'
              AND p.post_date >= %s
            GROUP BY DATE(p.post_date)
            ORDER BY order_date DESC
        ";

        $prepared = $wpdb->prepare(
            $sql,
            $meta_key_total,
            $meta_key_status,
            $completed_status,
            $post_type,
            $start_date
        );

        $results = $wpdb->get_results( $prepared );

        // Map results by date
        $by_date = [];
        if ( $results ) {
            foreach ( $results as $row ) {
                $date_key = $row->order_date; // Y-m-d
                $orders   = (int) $row->orders_count;
                $revenue  = (float) $row->revenue;
                $avg      = $orders > 0 ? ( $revenue / $orders ) : 0.0;

                $by_date[ $date_key ] = [
                    'orders'  => $orders,
                    'revenue' => $revenue,
                    'avg'     => $avg,
                ];
            }
        }

        // Fill every day in the range (including days with 0 orders)
        $rows = [];
        for ( $i = 0; $i < $days; $i++ ) {
            $ts       = $today_ts - $i * DAY_IN_SECONDS;
            $date_key = gmdate( 'Y-m-d', $ts );

            if ( isset( $by_date[ $date_key ] ) ) {
                $data = $by_date[ $date_key ];
            } else {
                $data = [
                    'orders'  => 0,
                    'revenue' => 0.0,
                    'avg'     => 0.0,
                ];
            }

            $rows[] = [
                'date'    => $date_key,
                'orders'  => $data['orders'],
                'revenue' => $data['revenue'],
                'avg'     => $data['avg'],
            ];
        }

        return $rows;
    }

    /**
     * Weekly statistics for the last $days days.
     *
     * @param int $days
     * @return array[] Each row: [ 'start' => 'Y-m-d', 'end' => 'Y-m-d', 'orders' => int, 'revenue' => float, 'avg' => float ]
     */
    protected static function get_weekly_stats( $days ) {
        global $wpdb;

        $days = max( 1, (int) $days );

        $post_type        = 'shoplite_pedido';
        $meta_key_total   = '_shoplite_order_total';
        $meta_key_status  = '_shoplite_order_status';
        $completed_status = 'completed';

        $posts_table    = $wpdb->posts;
        $postmeta_table = $wpdb->postmeta;

        $today_ts   = current_time( 'timestamp' );
        $start_ts   = $today_ts - ( $days - 1 ) * DAY_IN_SECONDS;
        $start_date = gmdate( 'Y-m-d 00:00:00', $start_ts );

        $sql = "
            SELECT 
                YEARWEEK(p.post_date, 1) AS yearweek,
                MIN(DATE(p.post_date))   AS period_start,
                MAX(DATE(p.post_date))   AS period_end,
                COUNT(DISTINCT p.ID)     AS orders_count,
                COALESCE(SUM(CAST(pm_total.meta_value AS DECIMAL(10,2))), 0) AS revenue
            FROM {$posts_table} p
            INNER JOIN {$postmeta_table} pm_total
                ON pm_total.post_id = p.ID
               AND pm_total.meta_key = %s
            INNER JOIN {$postmeta_table} pm_status
                ON pm_status.post_id = p.ID
               AND pm_status.meta_key = %s
               AND pm_status.meta_value = %s
            WHERE p.post_type = %s
              AND p.post_status = 'publish'
              AND p.post_date >= %s
            GROUP BY YEARWEEK(p.post_date, 1)
            ORDER BY period_start DESC
        ";

        $prepared = $wpdb->prepare(
            $sql,
            $meta_key_total,
            $meta_key_status,
            $completed_status,
            $post_type,
            $start_date
        );

        $results = $wpdb->get_results( $prepared );

        $rows = [];
        if ( $results ) {
            foreach ( $results as $row ) {
                $orders  = (int) $row->orders_count;
                $revenue = (float) $row->revenue;
                $avg     = $orders > 0 ? ( $revenue / $orders ) : 0.0;

                $rows[] = [
                    'start'   => $row->period_start, // Y-m-d
                    'end'     => $row->period_end,   // Y-m-d
                    'orders'  => $orders,
                    'revenue' => $revenue,
                    'avg'     => $avg,
                ];
            }
        }

        return $rows;
    }

    /**
     * Generates the CSV export for the time-based evolution.
     * It will be called from admin_init before HTML headers are sent.
     */
    public static function handle_export_csv() {

        // Only for admins
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have permission to export data.', self::TD ) );
        }

        // Make sure CSV export was actually requested
        if ( ! isset( $_GET['shoplite_export'] ) || 'csv' !== $_GET['shoplite_export'] ) {
            return;
        }

        // Verify nonce
        if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'shoplite_export_csv' ) ) {
            wp_die( esc_html__( 'Invalid nonce. The data cannot be exported.', self::TD ) );
        }

        // === Filters: range and grouping (same as in render_page) ===
        $ranges = [
            7   => __( 'Last 7 days', self::TD ),
            30  => __( 'Last 30 days', self::TD ),
            60  => __( 'Last 60 days', self::TD ),
            90  => __( 'Last 90 days', self::TD ),
            365 => __( 'Last 365 days', self::TD ),
            730 => __( 'Last 730 days', self::TD ),
        ];

        $group_modes = [
            'day'   => __( 'Day by day', self::TD ),
            'week'  => __( 'Week by week', self::TD ),
            'month' => __( 'Month by month', self::TD ),
        ];

        $range = isset( $_GET['range'] ) ? (int) $_GET['range'] : self::DEFAULT_RANGE;
        if ( ! array_key_exists( $range, $ranges ) ) {
            $range = self::DEFAULT_RANGE;
        }

        $group = isset( $_GET['group'] ) ? sanitize_key( $_GET['group'] ) : 'day';
        if ( ! array_key_exists( $group, $group_modes ) ) {
            $group = 'day';
        }

        // Get the same data as in the table
        switch ( $group ) {
            case 'week':
                $rows = self::get_weekly_stats( $range );
                break;
            case 'month':
                $rows = self::get_monthly_stats( $range );
                break;
            case 'day':
            default:
                $rows = self::get_daily_stats( $range );
                break;
        }

        $currency_symbol = get_option( 'shoplite_moneda_simbolo', '€' );

        // ==========================
        // Prepare CSV output
        // ==========================

        // Clear any previous output buffer (just in case)
        while ( ob_get_level() > 0 ) {
            ob_end_clean();
        }

        $filename = 'shoplite-sales-evolution-' . date( 'Ymd-His' ) . '.csv';

        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename="' . $filename . '"' );
        header( 'Pragma: no-cache' );
        header( 'Expires: 0' );

        $output = fopen( 'php://output', 'w' );

        // Column header (using ; as separator, common in many locales)
        fputcsv(
            $output,
            [
                __( 'Period', self::TD ),
                __( 'Completed orders', self::TD ),
                __( 'Period revenue', self::TD ) . ' (' . $currency_symbol . ')',
                __( 'Average order value', self::TD ) . ' (' . $currency_symbol . ')',
            ],
            ';'
        );

        if ( ! empty( $rows ) ) {
            foreach ( $rows as $row ) {

                // Period label, same as in the table
                if ( 'day' === $group ) {
                    $ts    = strtotime( $row['date'] . ' 00:00:00' );
                    $label = date_i18n( get_option( 'date_format' ), $ts );
                } elseif ( 'week' === $group ) {
                    $start_ts = strtotime( $row['start'] . ' 00:00:00' );
                    $end_ts   = strtotime( $row['end'] . ' 00:00:00' );
                    $week_no  = date_i18n( 'W', $start_ts );
                    $label    = sprintf(
                        __( 'Week %1$s (%2$s – %3$s)', self::TD ),
                        $week_no,
                        date_i18n( get_option( 'date_format' ), $start_ts ),
                        date_i18n( get_option( 'date_format' ), $end_ts )
                    );
                } else { // month
                    $start_ts = strtotime( $row['start'] . ' 00:00:00' );
                    $label    = date_i18n( 'F Y', $start_ts );
                }

                $orders = (int) $row['orders'];
                // Use dot as decimal separator in the CSV for consistency
                $rev    = number_format( (float) $row['revenue'], 2, '.', '' );
                $avg    = number_format( (float) $row['avg'], 2, '.', '' );

                fputcsv(
                    $output,
                    [
                        $label,
                        $orders,
                        $rev,
                        $avg,
                    ],
                    ';'
                );
            }
        }

        fclose( $output );
        exit;
    }

    /**
     * Monthly statistics for the last $days days.
     *
     * @param int $days
     * @return array[] Each row: [ 'start' => 'Y-m-d', 'end' => 'Y-m-d', 'orders' => int, 'revenue' => float, 'avg' => float ]
     */
    protected static function get_monthly_stats( $days ) {
        global $wpdb;

        $days = max( 1, (int) $days );

        $post_type        = 'shoplite_pedido';
        $meta_key_total   = '_shoplite_order_total';
        $meta_key_status  = '_shoplite_order_status';
        $completed_status = 'completed';

        $posts_table    = $wpdb->posts;
        $postmeta_table = $wpdb->postmeta;

        $today_ts   = current_time( 'timestamp' );
        $start_ts   = $today_ts - ( $days - 1 ) * DAY_IN_SECONDS;
        $start_date = gmdate( 'Y-m-d 00:00:00', $start_ts );

        $sql = "
            SELECT 
                DATE_FORMAT(p.post_date, '%%Y-%%m') AS period_key,
                DATE_FORMAT(MIN(p.post_date), '%%Y-%%m-01') AS period_start,
                DATE_FORMAT(MAX(p.post_date), '%%Y-%%m-%%d') AS period_end,
                COUNT(DISTINCT p.ID) AS orders_count,
                COALESCE(SUM(CAST(pm_total.meta_value AS DECIMAL(10,2))), 0) AS revenue
            FROM {$posts_table} p
            INNER JOIN {$postmeta_table} pm_total
                ON pm_total.post_id = p.ID
               AND pm_total.meta_key = %s
            INNER JOIN {$postmeta_table} pm_status
                ON pm_status.post_id = p.ID
               AND pm_status.meta_key = %s
               AND pm_status.meta_value = %s
            WHERE p.post_type = %s
              AND p.post_status = 'publish'
              AND p.post_date >= %s
            GROUP BY DATE_FORMAT(p.post_date, '%%Y-%%m')
            ORDER BY period_start DESC
        ";

        $prepared = $wpdb->prepare(
            $sql,
            $meta_key_total,
            $meta_key_status,
            $completed_status,
            $post_type,
            $start_date
        );

        $results = $wpdb->get_results( $prepared );

        $rows = [];
        if ( $results ) {
            foreach ( $results as $row ) {
                $orders  = (int) $row->orders_count;
                $revenue = (float) $row->revenue;
                $avg     = $orders > 0 ? ( $revenue / $orders ) : 0.0;

                $rows[] = [
                    'start'   => $row->period_start, // Y-m-d (first day of the month)
                    'end'     => $row->period_end,   // Y-m-d (last day with orders)
                    'orders'  => $orders,
                    'revenue' => $revenue,
                    'avg'     => $avg,
                ];
            }
        }

        return $rows;
    }
}

// Run CSV export before the admin page is rendered.
if ( is_admin() ) {
    add_action( 'admin_init', function () {

        // Make sure we are on the time-based evolution page
        if ( ! isset( $_GET['page'] ) || $_GET['page'] !== Shoplite_Pro_Evolution::MENU_SLUG ) {
            return;
        }

        // And that CSV export has been requested
        if ( isset( $_GET['shoplite_export'] ) && $_GET['shoplite_export'] === 'csv' ) {
            Shoplite_Pro_Evolution::handle_export_csv();
        }
    } );
}
