<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Shoplite Pro - Digital delivery email
 * Sends download links (HTML) after order is marked paid/completed.
 */
class Shoplite_Pro_Digital_Delivery_Email {

    const META_SENT = '_shoplite_delivery_email_sent';

    public static function init() {
        // Priority 20 so it runs after token creation (usually priority 10 in free).
        add_action( 'shoplite_order_paid', array( __CLASS__, 'maybe_send' ), 20, 1 );
    }

    public static function maybe_send( $order_id ) {

        $order_id = (int) $order_id;
        if ( $order_id <= 0 ) return;

        // Avoid duplicates.
        if ( get_post_meta( $order_id, self::META_SENT, true ) === '1' ) {
            return;
        }

        $to = (string) get_post_meta( $order_id, '_shoplite_customer_email', true );
        $to = sanitize_email( $to );
        if ( empty( $to ) ) {
            return;
        }

        $customer_name = (string) get_post_meta( $order_id, '_shoplite_customer_name', true );
        $customer_name = sanitize_text_field( $customer_name );

        // Tokens created by free module (digital-delivery.php)
        $tokens = get_post_meta( $order_id, '_shoplite_delivery_tokens', true );
        $tokens = maybe_unserialize( $tokens );

        if ( ! is_array( $tokens ) || empty( $tokens ) ) {
            // No digital items → nothing to send
            return;
        }

        // Build HTML
        $site_name = wp_specialchars_decode( get_bloginfo( 'name' ), ENT_QUOTES );
        $subject   = sprintf( __( 'Your download is ready – Order #%d', 'shoplite' ), $order_id );

        $greeting_name = $customer_name ? $customer_name : __( 'there', 'shoplite' );

        $intro = sprintf(
            __( 'Hi %s, thanks for your purchase. Here are your download link(s):', 'shoplite' ),
            esc_html( $greeting_name )
        );

        $limit_note = __( 'Each link allows up to 5 downloads and does not expire.', 'shoplite' );

        $items_html = '';
        foreach ( $tokens as $row ) {
            if ( ! is_array( $row ) ) continue;

            $product_id = isset( $row['product_id'] ) ? (int) $row['product_id'] : 0;
            $url        = isset( $row['url'] ) ? esc_url( $row['url'] ) : '';

            if ( ! $url ) continue;

            $product_title = $product_id ? get_the_title( $product_id ) : '';
            $product_title = $product_title ? $product_title : __( 'Digital product', 'shoplite' );

            $items_html .= '<li style="margin:0 0 10px 0;">'
                . '<div style="font-weight:600;">' . esc_html( $product_title ) . '</div>'
                . '<div><a href="' . $url . '">' . esc_html( $url ) . '</a></div>'
                . '</li>';
        }

        if ( $items_html === '' ) {
            return;
        }

        $html = ''
          . '<div style="font-family:Arial,Helvetica,sans-serif;font-size:14px;line-height:1.45;color:#111;">'
          . '  <p style="margin:0 0 12px 0;">' . $intro . '</p>'
          . '  <ul style="padding-left:18px;margin:0 0 12px 0;">' . $items_html . '</ul>'
          . '  <p style="margin:0 0 12px 0;"><em>' . esc_html( $limit_note ) . '</em></p>'
          . '  <hr style="border:none;border-top:1px solid #ddd;margin:16px 0;">'
          . '  <p style="margin:0;color:#555;">' . esc_html( $site_name ) . '</p>'
          . '</div>';

        // Headers: HTML email + From
        $headers = array();
        $headers[] = 'Content-Type: text/html; charset=UTF-8';

        $from_email = get_option( 'admin_email' );
        $from_name  = $site_name;
        if ( $from_email ) {
            $headers[] = 'From: ' . $from_name . ' <' . sanitize_email( $from_email ) . '>';
        }

        // Send
        $sent = wp_mail( $to, $subject, $html, $headers );

        if ( $sent ) {
            update_post_meta( $order_id, self::META_SENT, '1' );
        } else {
            // Helpful debug (doesn't break checkout)
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( '[shoplite-pro][digital-email] wp_mail FAILED for order ' . $order_id . ' to=' . $to );
            }
        }
    }
}

Shoplite_Pro_Digital_Delivery_Email::init();
