<?php
// shoplite-pro/includes/class-shoplite-pro-dashboard.php
if ( ! defined( 'ABSPATH' ) ) exit;

class Shoplite_Pro_Dashboard {

    const MENU_SLUG = 'shoplite_pro_dashboard';

    public static function init() {
        // Submenu under "Talariasoft Shoplite"
        add_action( 'admin_menu', [ __CLASS__, 'register_menu' ], 20 );
    }

    /**
     * Register the "Dashboard (Pro)" submenu under the main Shoplite menu.
     */
    public static function register_menu() {
        add_submenu_page(
            'shoplite', // parent menu slug (Talariasoft Shoplite)
            __( 'Dashboard (Pro)', 'shoplite-pro' ),
            __( 'Dashboard (Pro)', 'shoplite-pro' ),
            'manage_options',
            self::MENU_SLUG,
            [ __CLASS__, 'render_page' ]
        );
    }

    /**
     * Render the Pro Dashboard.
     */
    public static function render_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'shoplite-pro' ) );
        }

        global $wpdb;

        // ===============================
        // SELECTABLE SALES RANGE
        // ===============================
        $sales_ranges = [
            '7d' => [
                'label' => __( 'Last 7 days', 'shoplite-pro' ),
                'days'  => 7,
            ],
            '2m' => [
                'label' => __( 'Last 2 months', 'shoplite-pro' ),
                'days'  => 60,
            ],
            '6m' => [
                'label' => __( 'Last 6 months', 'shoplite-pro' ),
                'days'  => 180,
            ],
            '2y' => [
                'label' => __( 'Last 2 years', 'shoplite-pro' ),
                'days'  => 730,
            ],
        ];

        $current_range_key = isset( $_GET['shoplite_sales_range'] )
            ? sanitize_key( wp_unslash( $_GET['shoplite_sales_range'] ) )
            : '7d';

        if ( ! isset( $sales_ranges[ $current_range_key ] ) ) {
            $current_range_key = '7d';
        }

        $current_range    = $sales_ranges[ $current_range_key ];
        $sales_range_days = (int) $current_range['days'];

        // ===============================
        // 1) BASIC METRICS (BACKEND)
        // ===============================

        // Abandoned carts table (may not exist if something failed)
        $table_ac = $wpdb->prefix . 'shoplite_abandoned_carts';

        $has_ac_table = ( $wpdb->get_var(
            $wpdb->prepare( "SHOW TABLES LIKE %s", $table_ac )
        ) === $table_ac );

        $total_carts       = 0;
        $total_carts_email = 0;
        $last_carts        = [];
        $chart_days        = [];
        $chart_values      = [];

        if ( $has_ac_table ) {
            // Total tracked carts
            $total_carts = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_ac}" );

            // Total emails sent
            $total_carts_email = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$table_ac} WHERE email_sent = 1" );

            // Last 5 carts
            $last_carts = $wpdb->get_results(
                "SELECT email, cart_total, last_update, email_sent 
                 FROM {$table_ac}
                 ORDER BY last_update DESC
                 LIMIT 5",
                ARRAY_A
            );

            // Last 7 days data (for a horizontal bar chart)
            $now_ts   = current_time( 'timestamp' );
            $max_value = 0;

            for ( $i = 6; $i >= 0; $i-- ) {
                // Day i days ago (at midnight)
                $day_ts   = strtotime( '-' . $i . ' days', $now_ts );
                $day_from = gmdate( 'Y-m-d 00:00:00', $day_ts );
                $day_to   = gmdate( 'Y-m-d 23:59:59', $day_ts );

                $count_for_day = (int) $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT COUNT(*) FROM {$table_ac}
                         WHERE last_update >= %s AND last_update <= %s",
                        $day_from,
                        $day_to
                    )
                );

                $chart_days[]   = date_i18n( 'd M', $day_ts );
                $chart_values[] = $count_for_day;
                if ( $count_for_day > $max_value ) {
                    $max_value = $count_for_day;
                }
            }

        }

        // ===============================
        // 1.b) SALES METRICS (LAST N DAYS)
        // ===============================

        $orders_last_7d_total = 0.0;
        $orders_last_7d_count = 0;

        // Arrays for the sales chart (last N days)
        $orders_chart_days        = [];
        $orders_chart_totals      = [];
        $orders_chart_counts      = [];
        $orders_chart_max_total   = 0;

        // Try to read orders from CPT "shoplite_pedido".
        // Adjust these values if your core uses different ones:
        $orders_post_type   = 'shoplite_pedido';
        $meta_key_total     = '_shoplite_order_total';
        $meta_key_status    = '_shoplite_order_status';
        $completed_status   = 'completed';

        // Cutoff date according to the selected range
        $date_7d_ago = gmdate( 'Y-m-d H:i:s', time() - $sales_range_days * DAY_IN_SECONDS );

        // 1) Aggregated totals for the last N days (for the card)
        $orders_q = new WP_Query( [
            'post_type'      => $orders_post_type,
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'fields'         => 'ids',
            'date_query'     => [
                [
                    'after'     => $date_7d_ago,
                    'column'    => 'post_date_gmt',
                    'inclusive' => true,
                ],
            ],
            'meta_query'     => [
                'relation' => 'AND',
                [
                    'key'   => $meta_key_status,
                    'value' => $completed_status,
                ],
            ],
        ] );

        if ( $orders_q->have_posts() ) {
            foreach ( $orders_q->posts as $order_id ) {
                $total = get_post_meta( $order_id, $meta_key_total, true );
                $total = $total !== '' ? (float) $total : 0.0;

                if ( $total > 0 ) {
                    $orders_last_7d_total += $total;
                    $orders_last_7d_count++;
                }
            }
        }
        wp_reset_postdata();

        // 2) Daily data for the chart (last N days)
        $orders_rows = $wpdb->get_results(
            $wpdb->prepare(
                "
                SELECT 
                    DATE(p.post_date) AS order_date,
                    COALESCE(SUM(CAST(pm_total.meta_value AS DECIMAL(10,2))), 0) AS revenue,
                    COUNT(DISTINCT p.ID) AS orders_count
                FROM {$wpdb->posts} p
                INNER JOIN {$wpdb->postmeta} pm_total
                    ON pm_total.post_id = p.ID
                   AND pm_total.meta_key = %s
                INNER JOIN {$wpdb->postmeta} pm_status
                    ON pm_status.post_id = p.ID
                   AND pm_status.meta_key = %s
                   AND pm_status.meta_value = %s
                WHERE p.post_type = %s
                  AND p.post_status = 'publish'
                  AND p.post_date_gmt >= %s
                GROUP BY DATE(p.post_date)
                ORDER BY order_date ASC
                ",
                $meta_key_total,
                $meta_key_status,
                $completed_status,
                $orders_post_type,
                $date_7d_ago
            ),
            ARRAY_A
        );

        // Map by date so we can fill days without orders with 0
        $orders_by_date = [];
        foreach ( (array) $orders_rows as $row ) {
            $date_key = $row['order_date'];
            $orders_by_date[ $date_key ] = [
                'total' => (float) $row['revenue'],
                'count' => (int) $row['orders_count'],
            ];
        }

        // Use the same $now_ts as the carts block
        $now_ts = isset( $now_ts ) ? $now_ts : current_time( 'timestamp' );

        for ( $i = $sales_range_days - 1; $i >= 0; $i-- ) {

            $day_ts   = strtotime( '-' . $i . ' days', $now_ts );
            $date_key = gmdate( 'Y-m-d', $day_ts );
            $label    = date_i18n( 'd M', $day_ts );

            if ( isset( $orders_by_date[ $date_key ] ) ) {
                $day_total = $orders_by_date[ $date_key ]['total'];
                $day_count = $orders_by_date[ $date_key ]['count'];
            } else {
                $day_total = 0;
                $day_count = 0;
            }

            $orders_chart_days[]   = $label;
            $orders_chart_totals[] = $day_total;
            $orders_chart_counts[] = $day_count;

            if ( $day_total > $orders_chart_max_total ) {
                $orders_chart_max_total = $day_total;
            }
        }

        if ( $orders_chart_max_total <= 0 ) {
            $orders_chart_max_total = 1;
        }

        // Currency (assuming Shoplite has an option; adjust if it uses another key)
        $currency_symbol = get_option( 'shoplite_moneda_simbolo', '€' );

        // Avoid division by zero for bars
        $max_value = isset( $max_value ) && $max_value > 0 ? $max_value : 1;

        // Percentage of tracked emails / carts
        $rate_email = ( $total_carts > 0 )
            ? round( ( $total_carts_email / $total_carts ) * 100, 1 )
            : 0;

        // ===============================
        // 2) ABANDONED CART OPTIONS
        // ===============================
        $delay_min = (int) get_option( 'shoplite_pro_abandoned_delay_min', 30 );
        if ( $delay_min <= 0 ) {
            $delay_min = 30;
        }
        $ab_enabled = (bool) get_option( 'shoplite_pro_abandoned_enabled', 0 );

        // ===============================
        // 3) SYSTEM INFO
        // ===============================
        if ( ! function_exists( 'get_plugin_data' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $core_data = [];
        $pro_data  = [];

        // Core plugin data (shoplite/shoplite.php if that is the main file)
        $plugins = get_plugins();
        foreach ( $plugins as $file => $data ) {
            if ( strpos( $file, 'shoplite.php' ) !== false && strpos( $file, 'shoplite-pro' ) === false ) {
                $core_data = $data;
            }
            if ( strpos( $file, 'shoplite-pro.php' ) !== false ) {
                $pro_data = $data;
            }
        }

        $wp_version   = get_bloginfo( 'version' );
        $php_version  = PHP_VERSION;
        $site_url     = get_site_url();
        $timezone_str = get_option( 'timezone_string', 'UTC' );

        ?>
        <div class="wrap shoplite-pro-dashboard">
            <?php
            // Brand header: Talariasoft + Shoplite
            if ( function_exists( 'shoplite_render_brand_header' ) ) {
                shoplite_render_brand_header();
            }
            ?>

            <h1><?php esc_html_e( 'Shoplite Pro dashboard', 'shoplite-pro' ); ?></h1>
            <p style="color:#4b5563;margin-top:0;margin-bottom:14px;font-size:13px;">
                <?php esc_html_e( 'Visual summary of your lightweight store: carts, recovery emails and system status.', 'shoplite-pro' ); ?>
            </p>

            <div class="shoplite-pro-hero">
                <div class="shoplite-pro-hero-left">
                    <div class="shoplite-pro-hero-title">
                        <?php esc_html_e( 'Overview at a glance', 'shoplite-pro' ); ?>
                    </div>
                    <div class="shoplite-pro-hero-sub">
                        <?php esc_html_e( 'Monitor carts, email automations and system health from a single panel.', 'shoplite-pro' ); ?>
                    </div>
                </div>
                <div class="shoplite-pro-hero-pill">
                    <?php esc_html_e( 'Pro dashboard active', 'shoplite-pro' ); ?>
                </div>
            </div>


        <!-- =========================== -->
        <!-- SALES PERIOD SELECTOR       -->
        <!-- =========================== -->
        <form method="get" class="shoplite-pro-range-form"
              style="margin:10px 0 16px; display:flex; align-items:center; gap:8px; font-size:12px;">

            <!-- Keep user in Pro dashboard on submit -->
            <input type="hidden" name="page" value="<?php echo esc_attr( self::MENU_SLUG ); ?>" />

            <label for="shoplite_sales_range" style="font-weight:600;">
                <?php esc_html_e( 'Sales period:', 'shoplite-pro' ); ?>
            </label>

            <select id="shoplite_sales_range"
                    name="shoplite_sales_range"
                    onchange="this.form.submit();"
                    style="padding:4px 6px; border-radius:6px; border:1px solid #cbd5e1;">

                <?php foreach ( $sales_ranges as $key => $cfg ) : ?>
                    <option value="<?php echo esc_attr( $key ); ?>"
                        <?php selected( $current_range_key, $key ); ?>>
                        <?php echo esc_html( $cfg['label'] ); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </form>

        <!-- ====================== -->
        <!--  TOP STATS CARDS       -->
        <!-- ====================== -->
        <div class="shoplite-pro-row shoplite-pro-row--cards">
            <div class="shoplite-pro-card shoplite-pro-card--primary">
                <div class="shoplite-pro-card-label">
                    <?php esc_html_e( 'Tracked carts', 'shoplite-pro' ); ?>
                </div>
                <div class="shoplite-pro-card-value">
                    <?php echo esc_html( number_format_i18n( $total_carts ) ); ?>
                </div>
                <div class="shoplite-pro-card-sub">
                    <?php
                    printf(
                        /* translators: %s = table name */
                        esc_html__( 'From table: %s', 'shoplite-pro' ),
                        esc_html( $has_ac_table ? $table_ac : __( 'not found', 'shoplite-pro' ) )
                    );
                    ?>
                </div>
            </div>

            <!-- NEW CARD: Completed sales (last 7 days) -->
            <div class="shoplite-pro-card">
                <div class="shoplite-pro-card-label">
                    <?php esc_html_e( 'Completed sales (last 7 days)', 'shoplite-pro' ); ?>
                </div>
                <div class="shoplite-pro-card-value">
                    <?php echo esc_html( number_format_i18n( $orders_last_7d_total, 2 ) ); ?>
                    <span class="shoplite-pro-card-unit">
                        <?php echo esc_html( $currency_symbol ); ?>
                    </span>
                </div>
                <div class="shoplite-pro-card-sub">
                    <?php
                    printf(
                        esc_html__( '%s completed orders', 'shoplite-pro' ),
                        esc_html( number_format_i18n( $orders_last_7d_count ) )
                    );
                    ?>
                </div>
            </div>

            <div class="shoplite-pro-card">
                <div class="shoplite-pro-card-label">
                    <?php esc_html_e( 'Recovery emails sent', 'shoplite-pro' ); ?>
                </div>
                <div class="shoplite-pro-card-value">
                    <?php echo esc_html( number_format_i18n( $total_carts_email ) ); ?>
                </div>
                <div class="shoplite-pro-card-sub">
                    <?php
                    printf(
                        esc_html__( '%s %% of tracked carts', 'shoplite-pro' ),
                        esc_html( $rate_email )
                    );
                    ?>
                </div>
            </div>

            <div class="shoplite-pro-card">
                <div class="shoplite-pro-card-label">
                    <?php esc_html_e( 'Abandoned cart delay', 'shoplite-pro' ); ?>
                </div>
                <div class="shoplite-pro-card-value">
                    <?php echo esc_html( $delay_min ); ?>
                    <span class="shoplite-pro-card-unit">
                        <?php esc_html_e( 'min', 'shoplite-pro' ); ?>
                    </span>
                </div>
                <div class="shoplite-pro-card-sub">
                    <?php
                    echo $ab_enabled
                        ? esc_html__( 'Automatic emails ACTIVE', 'shoplite-pro' )
                        : esc_html__( 'Automatic emails disabled', 'shoplite-pro' );
                    ?>
                </div>
            </div>

            <div class="shoplite-pro-card">
                <div class="shoplite-pro-card-label">
                    <?php esc_html_e( 'Quick status', 'shoplite-pro' ); ?>
                </div>
                <div class="shoplite-pro-card-value shoplite-pro-card-value--small">
                    <?php esc_html_e( 'OK', 'shoplite-pro' ); ?>
                </div>
                <ul class="shoplite-pro-status-list">
                    <li><?php echo esc_html( 'WP ' . $wp_version ); ?></li>
                    <li><?php echo esc_html( 'PHP ' . $php_version ); ?></li>
                    <?php if ( ! empty( $core_data['Version'] ) ) : ?>
                        <li><?php echo esc_html( 'Shoplite ' . $core_data['Version'] ); ?></li>
                    <?php endif; ?>
                    <?php if ( ! empty( $pro_data['Version'] ) ) : ?>
                        <li><?php echo esc_html( 'Shoplite Pro ' . $pro_data['Version'] ); ?></li>
                    <?php endif; ?>
                </ul>
            </div>
        </div>

        <!-- ====================== -->
        <!--  MAIN CONTENT ROW      -->
        <!-- ====================== -->
        <div class="shoplite-pro-row">
            <!-- LEFT: last 7 days activity -->
            <div class="shoplite-pro-col shoplite-pro-col--2">
                <div class="shoplite-pro-panel">
                    <div class="shoplite-pro-panel-header">
                        <h2><?php esc_html_e( 'Cart activity (last 7 days)', 'shoplite-pro' ); ?></h2>
                        <span class="shoplite-pro-badge">
                            <?php esc_html_e( 'Quick view', 'shoplite-pro' ); ?>
                        </span>
                    </div>
                    <div class="shoplite-pro-panel-body">
                        <?php if ( $has_ac_table && ! empty( $chart_days ) ) : ?>
                            <div class="shoplite-pro-chart">
                                <?php foreach ( $chart_days as $idx => $label ) :
                                    $value   = isset( $chart_values[ $idx ] ) ? (int) $chart_values[ $idx ] : 0;
                                    $percent = $max_value > 0 ? ( $value / $max_value ) * 100 : 0;
                                    ?>
                                    <div class="shoplite-pro-chart-row">
                                        <div class="shoplite-pro-chart-label">
                                            <?php echo esc_html( $label ); ?>
                                        </div>
                                        <div class="shoplite-pro-chart-bar-wrap">
                                            <div class="shoplite-pro-chart-bar"
                                                 style="width: <?php echo esc_attr( max( 5, $percent ) ); ?>%;">
                                            </div>
                                        </div>
                                        <div class="shoplite-pro-chart-value">
                                            <?php echo esc_html( $value ); ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php else : ?>
                            <p style="color:#777;">
                                <?php esc_html_e( 'There is not enough abandoned cart data yet to display activity.', 'shoplite-pro' ); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                </div>

                <div class="shoplite-pro-panel">
                    <div class="shoplite-pro-panel-header">
                        <h2><?php esc_html_e( 'Latest detected carts', 'shoplite-pro' ); ?></h2>
                    </div>
                    <div class="shoplite-pro-panel-body">
                        <?php if ( $has_ac_table && ! empty( $last_carts ) ) : ?>
                            <table class="shoplite-pro-table">
                                <thead>
                                <tr>
                                    <th><?php esc_html_e( 'Email', 'shoplite-pro' ); ?></th>
                                    <th><?php esc_html_e( 'Cart total', 'shoplite-pro' ); ?></th>
                                    <th><?php esc_html_e( 'Last activity', 'shoplite-pro' ); ?></th>
                                    <th><?php esc_html_e( 'Email sent', 'shoplite-pro' ); ?></th>
                                </tr>
                                </thead>
                                <tbody>
                                <?php foreach ( $last_carts as $row ) : ?>
                                    <tr>
                                        <td>
                                            <?php
                                            echo $row['email']
                                                ? esc_html( $row['email'] )
                                                : '<span style="color:#999;">' . esc_html__( 'no email', 'shoplite-pro' ) . '</span>';
                                            ?>
                                        </td>
                                        <td>
                                            <?php
                                            $total = isset( $row['cart_total'] ) ? (float) $row['cart_total'] : 0;
                                            echo esc_html( number_format_i18n( $total, 2 ) );
                                            ?>
                                        </td>
                                        <td>
                                            <?php
                                            $ts = $row['last_update'] ? strtotime( $row['last_update'] ) : 0;
                                            echo $ts
                                                ? esc_html( date_i18n( 'd/m/Y H:i', $ts ) )
                                                : esc_html( '-' );
                                            ?>
                                        </td>
                                        <td>
                                            <?php if ( (int) $row['email_sent'] === 1 ) : ?>
                                                <span class="shoplite-pro-pill shoplite-pro-pill--success">
                                                    <?php esc_html_e( 'Yes', 'shoplite-pro' ); ?>
                                                </span>
                                            <?php else : ?>
                                                <span class="shoplite-pro-pill shoplite-pro-pill--muted">
                                                    <?php esc_html_e( 'No', 'shoplite-pro' ); ?>
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php else : ?>
                            <p style="color:#777;">
                                <?php esc_html_e( 'There are no carts stored in the database yet.', 'shoplite-pro' ); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- RIGHT: system info + quick links -->
            <div class="shoplite-pro-col shoplite-pro-col--1">
                <div class="shoplite-pro-panel">
                    <div class="shoplite-pro-panel-header">
                        <h2><?php esc_html_e( 'System status', 'shoplite-pro' ); ?></h2>
                    </div>
                    <div class="shoplite-pro-panel-body">
                        <ul class="shoplite-pro-system-list">
                            <li>
                                <span><?php esc_html_e( 'Site:', 'shoplite-pro' ); ?></span>
                                <code><?php echo esc_html( $site_url ); ?></code>
                            </li>
                            <li>
                                <span><?php esc_html_e( 'Timezone:', 'shoplite-pro' ); ?></span>
                                <code><?php echo esc_html( $timezone_str ); ?></code>
                            </li>
                            <li>
                                <span><?php esc_html_e( 'WP Cron:', 'shoplite-pro' ); ?></span>
                                <code><?php esc_html_e( 'Using WordPress cron', 'shoplite-pro' ); ?></code>
                            </li>
                            <li>
                                <span><?php esc_html_e( 'Abandoned carts table:', 'shoplite-pro' ); ?></span>
                                <?php if ( $has_ac_table ) : ?>
                                    <span class="shoplite-pro-pill shoplite-pro-pill--success">
                                        <?php esc_html_e( 'OK', 'shoplite-pro' ); ?>
                                    </span>
                                <?php else : ?>
                                    <span class="shoplite-pro-pill shoplite-pro-pill--danger">
                                        <?php esc_html_e( 'Not found', 'shoplite-pro' ); ?>
                                    </span>
                                <?php endif; ?>
                            </li>
                        </ul>
                    </div>
                </div>

                <div class="shoplite-pro-panel">
                    <div class="shoplite-pro-panel-header">
                        <h2><?php esc_html_e( 'Quick links', 'shoplite-pro' ); ?></h2>
                    </div>
                    <div class="shoplite-pro-panel-body shoplite-pro-quick-links">
                        <a class="shoplite-pro-button-link" href="<?php echo esc_url( admin_url( 'edit.php?post_type=shoplite_producto' ) ); ?>">
                            <?php esc_html_e( 'View products', 'shoplite-pro' ); ?>
                        </a>
                        <a class="shoplite-pro-button-link" href="<?php echo esc_url( admin_url( 'admin.php?page=shoplite-appearance' ) ); ?>">
                            <?php esc_html_e( 'Shoplite settings', 'shoplite-pro' ); ?>
                        </a>
                        <a class="shoplite-pro-button-link" href="<?php echo esc_url( admin_url( 'options-general.php?page=shoplite_pro_abandoned_carts' ) ); ?>">
                            <?php esc_html_e( 'Abandoned carts (Pro)', 'shoplite-pro' ); ?>
                        </a>
                        <a class="shoplite-pro-button-link" href="<?php echo esc_url( admin_url( 'edit.php?post_type=shoplite_coupon' ) ); ?>">
                            <?php esc_html_e( 'Coupons (Pro)', 'shoplite-pro' ); ?>
                        </a>
                    </div>
                </div>
            </div>
        </div>

    </div><!-- .shoplite-pro-dashboard -->




                    <style>
            .shoplite-pro-dashboard {
                max-width: 1200px;
                font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            }

            .shoplite-pro-dashboard h1 {
                font-weight: 650;
                letter-spacing: -0.02em;
            }

            .shoplite-pro-dashboard > p {
                font-size: 13px;
            }

            /* Hero header */
            .shoplite-pro-hero {
                margin-bottom: 18px;
                padding: 14px 16px;
                border-radius: 18px;
                background: radial-gradient(circle at top left, #eff6ff 0, #e0ecff 35%, #f9fafb 100%);
                border: 1px solid rgba(148, 163, 184, 0.45);
                display: flex;
                align-items: center;
                justify-content: space-between;
                gap: 12px;
            }

            .shoplite-pro-hero-left {
                display: flex;
                flex-direction: column;
                gap: 4px;
            }

            .shoplite-pro-hero-title {
                font-size: 15px;
                font-weight: 600;
                color: #0f172a;
            }

            .shoplite-pro-hero-sub {
                font-size: 12px;
                color: #4b5563;
            }

            .shoplite-pro-hero-pill {
                padding: 4px 10px;
                border-radius: 999px;
                font-size: 11px;
                background: rgba(37, 99, 235, 0.07);
                color: #1d4ed8;
                border: 1px solid rgba(37, 99, 235, 0.25);
                white-space: nowrap;
            }

            .shoplite-pro-row {
                display: flex;
                gap: 16px;
                margin-bottom: 20px;
                flex-wrap: wrap;
            }

            .shoplite-pro-row--cards .shoplite-pro-card {
                flex: 1 1 220px;
            }

            .shoplite-pro-col {
                display: flex;
                flex-direction: column;
                gap: 16px;
            }

            .shoplite-pro-col--2 {
                flex: 2 1 0;
            }

            .shoplite-pro-col--1 {
                flex: 1 1 0;
            }

            .shoplite-pro-card {
                background: #ffffff;
                border-radius: 18px;
                padding: 16px 18px;
                box-shadow: 0 12px 32px rgba(15, 23, 42, 0.08);
                border: 1px solid rgba(148, 163, 184, 0.4);
                display: flex;
                flex-direction: column;
                justify-content: space-between;
                position: relative;
                overflow: hidden;
            }

            .shoplite-pro-card--primary {
                background: radial-gradient(circle at top left, #2563eb 0, #1d4ed8 36%, #020617 100%);
                color: #e5e7eb;
                border-color: transparent;
            }

            .shoplite-pro-card--primary::before {
                content: "";
                position: absolute;
                inset: -40px;
                background: radial-gradient(circle at top, rgba(239, 246, 255, 0.2), transparent 60%);
                opacity: 0.9;
                pointer-events: none;
            }

            .shoplite-pro-card--primary > * {
                position: relative;
                z-index: 1;
            }

            .shoplite-pro-card--primary .shoplite-pro-card-label {
                color: #dbeafe;
            }

            .shoplite-pro-card--primary .shoplite-pro-card-value {
                color: #f9fafb;
            }

            .shoplite-pro-card--primary .shoplite-pro-card-sub {
                color: #bfdbfe;
            }

            .shoplite-pro-card-label {
                font-size: 11px;
                letter-spacing: 0.08em;
                text-transform: uppercase;
                color: #6b7280;
                margin-bottom: 10px;
            }

            .shoplite-pro-card-value {
                font-size: 26px;
                font-weight: 700;
                color: #0f172a;
                display: flex;
                align-items: baseline;
                gap: 4px;
            }

            .shoplite-pro-card-value--small {
                font-size: 22px;
            }

            .shoplite-pro-card-unit {
                font-size: 11px;
                color: #6b7280;
            }

            .shoplite-pro-card-sub {
                margin-top: 6px;
                font-size: 12px;
                color: #6b7280;
            }

            .shoplite-pro-status-list {
                list-style: none;
                padding: 0;
                margin: 8px 0 0 0;
                font-size: 12px;
                color: #4b5563;
            }

            .shoplite-pro-status-list li {
                margin-bottom: 2px;
            }

            .shoplite-pro-panel {
                background: #ffffff;
                border-radius: 20px;
                padding: 16px 18px 18px;
                box-shadow: 0 14px 40px rgba(15, 23, 42, 0.1);
                border: 1px solid rgba(148, 163, 184, 0.45);
            }

            .shoplite-pro-panel-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 12px;
                gap: 8px;
            }

            .shoplite-pro-panel-header h2 {
                font-size: 15px;
                margin: 0;
                color: #0f172a;
                letter-spacing: -0.01em;
            }

            .shoplite-pro-badge {
                font-size: 11px;
                padding: 3px 8px;
                border-radius: 999px;
                background: rgba(37, 99, 235, 0.08);
                color: #2563eb;
                border: 1px solid rgba(37, 99, 235, 0.3);
            }

            .shoplite-pro-panel-body {
                font-size: 13px;
            }

            /* Horizontal bar chart */
            .shoplite-pro-chart {
                display: flex;
                flex-direction: column;
                gap: 6px;
            }

            .shoplite-pro-chart-row {
                display: grid;
                grid-template-columns: 70px 1fr 40px;
                align-items: center;
                gap: 8px;
            }

            .shoplite-pro-chart-label {
                font-size: 12px;
                color: #6b7280;
            }

            .shoplite-pro-chart-bar-wrap {
                background: #e5e7eb;
                border-radius: 999px;
                overflow: hidden;
                height: 8px;
            }

            .shoplite-pro-chart-bar {
                height: 100%;
                border-radius: inherit;
                background: linear-gradient(90deg, #22c55e, #16a34a);
                box-shadow: 0 0 0.5px rgba(22, 163, 74, 0.9);
            }

            .shoplite-pro-chart-value {
                font-size: 12px;
                text-align: right;
                color: #111827;
                font-weight: 500;
            }

            .shoplite-pro-table {
                width: 100%;
                border-collapse: collapse;
                font-size: 12px;
            }

            .shoplite-pro-table th,
            .shoplite-pro-table td {
                padding: 6px 8px;
                border-bottom: 1px solid #e5e7eb;
                text-align: left;
            }

            .shoplite-pro-table th {
                font-weight: 600;
                background: #f9fafb;
                color: #4b5563;
            }

            .shoplite-pro-pill {
                display: inline-flex;
                align-items: center;
                justify-content: center;
                font-size: 11px;
                padding: 2px 8px;
                border-radius: 999px;
                border: 1px solid transparent;
                white-space: nowrap;
            }

            .shoplite-pro-pill--success {
                background: rgba(22, 163, 74, 0.08);
                color: #15803d;
                border-color: rgba(22, 163, 74, 0.3);
            }

            .shoplite-pro-pill--muted {
                background: rgba(148, 163, 184, 0.1);
                color: #4b5563;
                border-color: rgba(148, 163, 184, 0.4);
            }

            .shoplite-pro-pill--danger {
                background: rgba(220, 38, 38, 0.08);
                color: #b91c1c;
                border-color: rgba(220, 38, 38, 0.4);
            }

            .shoplite-pro-system-list {
                list-style: none;
                padding: 0;
                margin: 0;
            }

            .shoplite-pro-system-list li {
                display: flex;
                justify-content: space-between;
                align-items: center;
                gap: 8px;
                margin-bottom: 6px;
            }

            .shoplite-pro-system-list span {
                font-size: 12px;
                color: #4b5563;
            }

            .shoplite-pro-system-list code {
                font-size: 11px;
                background: #f3f4f6;
                padding: 2px 6px;
                border-radius: 4px;
                color: #111827;
            }

            .shoplite-pro-quick-links {
                display: flex;
                flex-direction: column;
                gap: 8px;
            }

            .shoplite-pro-button-link {
                display: inline-flex;
                align-items: center;
                justify-content: center;
                padding: 7px 11px;
                border-radius: 999px;
                border: 1px solid rgba(148, 163, 184, 0.9);
                background: linear-gradient(135deg, #f9fafb, #eef2ff);
                font-size: 12px;
                color: #111827;
                text-decoration: none;
                transition: all 0.15s ease;
            }

            .shoplite-pro-button-link:hover {
                border-color: #2563eb;
                background: linear-gradient(135deg, #eff6ff, #dbeafe);
                color: #1d4ed8;
                box-shadow: 0 6px 18px rgba(37, 99, 235, 0.2);
                transform: translateY(-1px);
            }

            @media (max-width: 960px) {
                .shoplite-pro-row {
                    flex-direction: column;
                }

                .shoplite-pro-hero {
                    flex-direction: column;
                    align-items: flex-start;
                }
            }
        </style>

        <?php
    }
}
