<?php
// shoplite-pro/includes/class-shoplite-pro-coupons.php
if ( ! defined( 'ABSPATH' ) ) exit;

class Shoplite_Pro_Coupons {

    const POST_TYPE = 'shoplite_coupon';

    // Meta keys
    const META_CODE        = '_shoplite_coupon_code';
    const META_TYPE        = '_shoplite_coupon_type';
    const META_AMOUNT      = '_shoplite_coupon_amount';
    const META_MIN_TOTAL   = '_shoplite_coupon_min_total';
    const META_EXPIRES     = '_shoplite_coupon_expires';
    const META_USAGE_LIMIT = '_shoplite_coupon_usage_limit';
    const META_PRODUCTS    = '_shoplite_coupon_products';

    public static function init() {
        add_action( 'init', [ __CLASS__, 'register_cpt' ] );

        add_action( 'add_meta_boxes', [ __CLASS__, 'register_metabox' ] );
        add_action( 'save_post_' . self::POST_TYPE, [ __CLASS__, 'save_meta' ], 10, 2 );

        // Columnas en el listado
        add_filter( 'manage_' . self::POST_TYPE . '_posts_columns', [ __CLASS__, 'columns' ] );
        add_action( 'manage_' . self::POST_TYPE . '_posts_custom_column', [ __CLASS__, 'column_content' ], 10, 2 );

        // Recolocar el submenú
        add_action( 'admin_menu', [ __CLASS__, 'reorder_submenu' ], 999 );

        // Aplicar cupones al carrito (frontend)
        add_filter( 'shoplite_cart_totals', [ __CLASS__, 'apply_coupon' ], 10, 2 );
    }

    /**
     * Registrar el Custom Post Type "Cupón"
     */
    public static function register_cpt() {
        $labels = [
            'name'               => __( 'Coupons', 'shoplite-pro' ),
            'singular_name'      => __( 'Coupon', 'shoplite-pro' ),
            'add_new'            => __( 'Add New', 'shoplite-pro' ),
            'add_new_item'       => __( 'Add New Coupon', 'shoplite-pro' ),
            'edit_item'          => __( 'Edit Coupon', 'shoplite-pro' ),
            'new_item'           => __( 'New Coupon', 'shoplite-pro' ),
            'view_item'          => __( 'View Coupon', 'shoplite-pro' ),
            'search_items'       => __( 'Search Coupons', 'shoplite-pro' ),
            'not_found'          => __( 'No coupons found', 'shoplite-pro' ),
            'not_found_in_trash' => __( 'No coupons in the trash', 'shoplite-pro' ),
            'menu_name'          => __( 'Coupons (Pro)', 'shoplite-pro' ),
        ];

        $args = [
            'labels'             => $labels,
            'public'             => false,
            'show_ui'            => true,
            'show_in_menu'       => 'shoplite', // debajo de Talariasoft Shoplite
            'capability_type'    => 'post',
            'map_meta_cap'       => true,
            'hierarchical'       => false,
            'supports'           => [ 'title' ],
            'has_archive'        => false,
            'menu_position'      => 60,
        ];

        register_post_type( self::POST_TYPE, $args );
    }

    /**
     * Registrar metabox de datos del cupón
     */
    public static function register_metabox() {
        add_meta_box(
            'shoplite_coupon_data',
            __( 'Coupon data', 'shoplite-pro' ),
            [ __CLASS__, 'render_metabox' ],
            self::POST_TYPE,
            'normal',
            'high'
        );
    }

    /**
     * Pintar campos del cupón en el metabox
     */
    public static function render_metabox( $post ) {
        wp_nonce_field( 'shoplite_coupon_save', 'shoplite_coupon_nonce' );

        $code        = get_post_meta( $post->ID, self::META_CODE, true );
        $type        = get_post_meta( $post->ID, self::META_TYPE, true ) ?: 'percent';
        $amount      = get_post_meta( $post->ID, self::META_AMOUNT, true );
        $min_total   = get_post_meta( $post->ID, self::META_MIN_TOTAL, true );
        $expires     = get_post_meta( $post->ID, self::META_EXPIRES, true );
        $usage_limit = get_post_meta( $post->ID, self::META_USAGE_LIMIT, true );

        // Productos a los que aplica el cupón
        $selected_products = get_post_meta( $post->ID, self::META_PRODUCTS, true );
        if ( ! is_array( $selected_products ) ) {
            $selected_products = [];
        }

        // Obtener productos publicados del CPT shoplite_producto
        $productos = get_posts( [
            'post_type'      => 'shoplite_producto',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'title',
            'order'          => 'ASC',
        ] );
        ?>
        <table class="form-table">
            <tr>
                <th>
                    <label for="shoplite_coupon_code">
                        <?php esc_html_e( 'Coupon code', 'shoplite-pro' ); ?>
                    </label>
                </th>
                <td>
                    <input
                        type="text"
                        id="shoplite_coupon_code"
                        name="shoplite_coupon_code"
                        value="<?php echo esc_attr( $code ); ?>"
                        class="regular-text"
                    />
                    <p class="description">
                        <?php esc_html_e( 'The code the customer will enter at checkout (e.g. BLACKFRIDAY20).', 'shoplite-pro' ); ?>
                    </p>
                </td>
            </tr>

            <tr>
                <th>
                    <label for="shoplite_coupon_type">
                        <?php esc_html_e( 'Discount type', 'shoplite-pro' ); ?>
                    </label>
                </th>
                <td>
                    <select id="shoplite_coupon_type" name="shoplite_coupon_type">
                        <option value="percent" <?php selected( $type, 'percent' ); ?>>
                            <?php esc_html_e( 'Percentage (%)', 'shoplite-pro' ); ?>
                        </option>
                        <option value="fixed_cart" <?php selected( $type, 'fixed_cart' ); ?>>
                            <?php esc_html_e( 'Fixed cart discount', 'shoplite-pro' ); ?>
                        </option>
                    </select>
                </td>
            </tr>

            <tr>
                <th>
                    <label for="shoplite_coupon_amount">
                        <?php esc_html_e( 'Discount amount', 'shoplite-pro' ); ?>
                    </label>
                </th>
                <td>
                    <input
                        type="number"
                        step="0.01"
                        min="0"
                        id="shoplite_coupon_amount"
                        name="shoplite_coupon_amount"
                        value="<?php echo esc_attr( $amount ); ?>"
                    />
                    <p class="description">
                        <?php esc_html_e( 'If it is a percentage, use 10 for a 10% discount. If it is a fixed amount, use the store currency.', 'shoplite-pro' ); ?>
                    </p>
                </td>
            </tr>

            <tr>
                <th>
                    <label for="shoplite_coupon_min_total">
                        <?php esc_html_e( 'Minimum cart amount', 'shoplite-pro' ); ?>
                    </label>
                </th>
                <td>
                    <input
                        type="number"
                        step="0.01"
                        min="0"
                        id="shoplite_coupon_min_total"
                        name="shoplite_coupon_min_total"
                        value="<?php echo esc_attr( $min_total ); ?>"
                    />
                    <p class="description">
                        <?php esc_html_e( 'Optional. The coupon will only be valid if the cart subtotal reaches this amount.', 'shoplite-pro' ); ?>
                    </p>
                </td>
            </tr>

            <tr>
                <th>
                    <label for="shoplite_coupon_expires">
                        <?php esc_html_e( 'Expiration date', 'shoplite-pro' ); ?>
                    </label>
                </th>
                <td>
                    <input
                        type="date"
                        id="shoplite_coupon_expires"
                        name="shoplite_coupon_expires"
                        value="<?php echo esc_attr( $expires ); ?>"
                    />
                    <p class="description">
                        <?php esc_html_e( 'Optional. The coupon will stop being valid after this date.', 'shoplite-pro' ); ?>
                    </p>
                </td>
            </tr>

            <tr>
                <th>
                    <label for="shoplite_coupon_usage_limit">
                        <?php esc_html_e( 'Usage limit', 'shoplite-pro' ); ?>
                    </label>
                </th>
                <td>
                    <input
                        type="number"
                        min="0"
                        id="shoplite_coupon_usage_limit"
                        name="shoplite_coupon_usage_limit"
                        value="<?php echo esc_attr( $usage_limit ); ?>"
                    />
                    <p class="description">
                        <?php esc_html_e( 'Optional. Maximum number of times this coupon can be used. 0 = no limit.', 'shoplite-pro' ); ?>
                    </p>
                </td>
            </tr>

            <!-- Selector de productos -->
            <tr>
                <th>
                    <label for="shoplite_coupon_products">
                        <?php esc_html_e( 'Apply only to these products', 'shoplite-pro' ); ?>
                    </label>
                </th>
                <td>
                    <select
                        id="shoplite_coupon_products"
                        name="shoplite_coupon_products[]"
                        multiple
                        style="width: 100%; max-width: 420px; height: 160px;"
                    >
                        <?php foreach ( $productos as $p ) : ?>
                            <option value="<?php echo esc_attr( $p->ID ); ?>"
                                <?php selected( in_array( $p->ID, $selected_products, true ) ); ?>>
                                <?php echo esc_html( $p->post_title . ' (ID: ' . $p->ID . ')' ); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <p class="description">
                        <?php esc_html_e( 'Optional. If you select one or more products, the coupon will only apply to those products. If you select none, the coupon applies to the whole cart subtotal.', 'shoplite-pro' ); ?>
                    </p>
                </td>
            </tr>

        </table>
        <?php
    }

    /**
     * Guardar metadatos del cupón
     */
    public static function save_meta( $post_id, $post ) {
        if (
            ! isset( $_POST['shoplite_coupon_nonce'] ) ||
            ! wp_verify_nonce( $_POST['shoplite_coupon_nonce'], 'shoplite_coupon_save' )
        ) {
            return;
        }

        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Código
        $code = isset( $_POST['shoplite_coupon_code'] )
            ? sanitize_text_field( $_POST['shoplite_coupon_code'] )
            : '';
        $code = strtoupper( $code ); // Normalizamos en mayúsculas
        update_post_meta( $post_id, self::META_CODE, $code );

        // Tipo
        $type = isset( $_POST['shoplite_coupon_type'] )
            ? sanitize_text_field( $_POST['shoplite_coupon_type'] )
            : 'percent';

        if ( ! in_array( $type, [ 'percent', 'fixed_cart' ], true ) ) {
            $type = 'percent';
        }
        update_post_meta( $post_id, self::META_TYPE, $type );

        // Cantidad
        $amount = isset( $_POST['shoplite_coupon_amount'] )
            ? floatval( $_POST['shoplite_coupon_amount'] )
            : 0;
        if ( $amount < 0 ) {
            $amount = 0;
        }
        update_post_meta( $post_id, self::META_AMOUNT, $amount );

        // Importe mínimo
        $min_total = isset( $_POST['shoplite_coupon_min_total'] )
            ? floatval( $_POST['shoplite_coupon_min_total'] )
            : 0;
        if ( $min_total < 0 ) {
            $min_total = 0;
        }
        update_post_meta( $post_id, self::META_MIN_TOTAL, $min_total );

        // Fecha caducidad
        $expires = isset( $_POST['shoplite_coupon_expires'] )
            ? sanitize_text_field( $_POST['shoplite_coupon_expires'] )
            : '';
        update_post_meta( $post_id, self::META_EXPIRES, $expires );

        // Límite de usos
        $usage_limit = isset( $_POST['shoplite_coupon_usage_limit'] )
            ? intval( $_POST['shoplite_coupon_usage_limit'] )
            : 0;
        if ( $usage_limit < 0 ) {
            $usage_limit = 0;
        }
        update_post_meta( $post_id, self::META_USAGE_LIMIT, $usage_limit );

        // Productos asociados al cupón
        if ( isset( $_POST['shoplite_coupon_products'] ) && is_array( $_POST['shoplite_coupon_products'] ) ) {
            $product_ids = array_map( 'intval', $_POST['shoplite_coupon_products'] );
            $product_ids = array_filter( $product_ids ); // elimina ceros o nulls
            update_post_meta( $post_id, self::META_PRODUCTS, $product_ids );
        } else {
            update_post_meta( $post_id, self::META_PRODUCTS, [] );
        }
    }

    /**
     * Columnas en el listado de cupones
     */
    public static function columns( $cols ) {
        $new        = [];
        $new['cb']  = $cols['cb'];

        $new['title']   = __( 'Internal name', 'shoplite-pro' );
        $new['code']    = __( 'Code', 'shoplite-pro' );
        $new['type']    = __( 'Type', 'shoplite-pro' );
        $new['amount']  = __( 'Discount', 'shoplite-pro' );
        $new['scope']   = __( 'Scope', 'shoplite-pro' );
        $new['expires'] = __( 'Expires', 'shoplite-pro' );

        return $new;
    }

    public static function column_content( $column, $post_id ) {
        switch ( $column ) {
            case 'code':
                $code = get_post_meta( $post_id, self::META_CODE, true );
                echo $code
                    ? esc_html( $code )
                    : esc_html_x( '—', 'placeholder when coupon code is empty', 'shoplite-pro' );
                break;

            case 'type':
                $type = get_post_meta( $post_id, self::META_TYPE, true ) ?: 'percent';
                echo esc_html(
                    $type === 'percent'
                        ? __( '%', 'shoplite-pro' )
                        : __( 'Fixed cart', 'shoplite-pro' )
                );
                break;

            case 'amount':
                $type   = get_post_meta( $post_id, self::META_TYPE, true ) ?: 'percent';
                $amount = get_post_meta( $post_id, self::META_AMOUNT, true );
                if ( $amount === '' || $amount === null ) {
                    echo esc_html_x( '—', 'placeholder when coupon amount is empty', 'shoplite-pro' );
                } else {
                    if ( $type === 'percent' ) {
                        echo esc_html( $amount ) . '%';
                    } else {
                        echo esc_html( number_format_i18n( (float) $amount, 2 ) );
                    }
                }
                break;

            case 'scope':
                $prods = get_post_meta( $post_id, self::META_PRODUCTS, true );
                if ( ! is_array( $prods ) || empty( $prods ) ) {
                    esc_html_e( 'Whole cart', 'shoplite-pro' );
                } else {
                    $count = count( $prods );
                    printf(
                        esc_html( _n( '%d product', '%d products', $count, 'shoplite-pro' ) ),
                        $count
                    );
                }
                break;

            case 'expires':
                $expires = get_post_meta( $post_id, self::META_EXPIRES, true );
                echo $expires
                    ? esc_html( $expires )
                    : esc_html__( 'No expiration', 'shoplite-pro' );
                break;
        }
    }

    /**
     * Helper para obtener un cupón por código
     */
    public static function get_coupon_by_code( $code ) {
        $code = strtoupper( trim( $code ) );
        if ( $code === '' ) {
            return null;
        }

        $q = new WP_Query( [
            'post_type'      => self::POST_TYPE,
            'posts_per_page' => 1,
            'post_status'    => 'publish',
            'meta_query'     => [
                [
                    'key'   => self::META_CODE,
                    'value' => $code,
                ],
            ],
        ] );

        if ( ! $q->have_posts() ) {
            return null;
        }

        $post = $q->posts[0];

        return $post->ID;
    }

    /**
     * Reordenar el submenú de Talariasoft Shoplite
     * para que "Cupones (Pro)" vaya después de "Shoplite - Productos".
     */
    public static function reorder_submenu() {
        global $submenu;

        if ( ! isset( $submenu['shoplite'] ) ) {
            return;
        }

        $items   = $submenu['shoplite'];
        $coupons = [];
        $others  = [];

        foreach ( $items as $item ) {
            $slug = isset( $item[2] ) ? $item[2] : '';

            // Nuestro CPT de cupones
            if ( $slug === 'edit.php?post_type=' . self::POST_TYPE ) {
                $coupons[] = $item;
            } else {
                $others[] = $item;
            }
        }

        if ( empty( $coupons ) ) {
            return;
        }

        $new = [];
        foreach ( $others as $item ) {
            $new[] = $item;

            $slug = isset( $item[2] ) ? $item[2] : '';

            // Después del menú de productos insertamos Cupones
            if (
                $slug === 'edit.php?post_type=shoplite_producto'
                || $slug === 'edit.php?post_type=shoplite_product'
            ) {
                foreach ( $coupons as $c ) {
                    $new[] = $c;
                }
            }
        }

        $submenu['shoplite'] = $new;
    }

    /**
     * Aplica el cupón (si existe y es válido) a los totales del carrito.
     *
     * @param array $totals ['subtotal','shipping','discount','total']
     * @param array $cart   Carrito actual (array de líneas)
     * @return array Totales ajustados
     */
    public static function apply_coupon( array $totals, $cart ) {
        // Aseguramos sesión y reseteamos contexto previo
        if ( ! session_id() ) {
            @session_start();
        }
        unset( $_SESSION['shoplite_coupon_applied'] );

        // Código actual guardado en sesión (desde el checkout)
        $code = isset( $_SESSION['shoplite_coupon_code'] )
            ? trim( $_SESSION['shoplite_coupon_code'] )
            : '';

        if ( $code === '' ) {
            // No hay cupón aplicado
            return $totals;
        }

        $code_up = strtoupper( $code );

        // Buscar cupón (CPT) por meta _shoplite_coupon_code
        $q = new WP_Query( [
            'post_type'      => self::POST_TYPE,
            'posts_per_page' => 1,
            'post_status'    => 'publish',
            'meta_query'     => [
                [
                    'key'   => self::META_CODE,
                    'value' => $code_up,
                ],
            ],
        ] );

        if ( ! $q->have_posts() ) {
            // Código inválido
            return $totals;
        }

        $coupon_post = $q->posts[0];
        $coupon_id   = $coupon_post->ID;

        // Leer metadatos del cupón
        $type      = get_post_meta( $coupon_id, self::META_TYPE, true ) ?: 'percent';
        $amount    = (float) get_post_meta( $coupon_id, self::META_AMOUNT, true );
        $min_total = get_post_meta( $coupon_id, self::META_MIN_TOTAL, true );
        $expires   = get_post_meta( $coupon_id, self::META_EXPIRES, true );
        $products  = get_post_meta( $coupon_id, self::META_PRODUCTS, true );
        if ( ! is_array( $products ) ) {
            $products = [];
        }

        // Normalizar tipo: aceptamos 'fixed_cart' como fijo
        if ( $type === 'fixed_cart' ) {
            $type = 'fixed';
        }

        $subtotal = isset( $totals['subtotal'] ) ? (float) $totals['subtotal'] : 0.0;
        $shipping = isset( $totals['shipping'] ) ? (float) $totals['shipping'] : 0.0;

        if ( $subtotal <= 0 ) {
            return $totals;
        }


        // ================================
        // Subtotal elegible (productos)
        // ================================
        $eligible_subtotal = 0.0;
        $applied_products  = []; // nombres de productos a los que se aplica el cupón

        if ( empty( $products ) ) {
            // Cupón global: afecta a todo el carrito
            $eligible_subtotal = $subtotal;
        } else {
            // Cupón limitado a ciertos productos
            if ( ! is_array( $cart ) ) {
                $cart = [];
            }

            foreach ( $cart as $line_key => $item ) {
                // Determinar product ID
                $pid = 0;
                if ( isset( $item['id'] ) ) {
                    $pid = (int) $item['id'];
                } elseif ( is_string( $line_key ) && preg_match( '/^tbp_(\d+)/', $line_key, $m ) ) {
                    $pid = (int) $m[1];
                }

                if ( $pid <= 0 || ! in_array( $pid, $products, true ) ) {
                    continue; // este producto no está en el cupón
                }

                $price = isset( $item['precio'] )
                    ? (float) $item['precio']
                    : ( isset( $item['price'] ) ? (float) $item['price'] : 0.0 );

                $qty = isset( $item['cantidad'] )
                    ? max( 1, (int) $item['cantidad'] )
                    : ( isset( $item['qty'] ) ? max( 1, (int) $item['qty'] ) : 1 );

                $eligible_subtotal += $price * $qty;

                // Guardar nombre para mostrar en el carrito
                $name = isset( $item['nombre'] ) ? $item['nombre'] : get_the_title( $pid );
                if ( $name ) {
                    $applied_products[ $pid ] = $name;
                }
            }
        }

        // Si no hay nada elegible, no aplicar cupón
        if ( $eligible_subtotal <= 0 ) {
            return $totals;
        }

        // Validar caducidad (si hay fecha, formato YYYY-MM-DD)
        if ( ! empty( $expires ) ) {
            $now_ts     = current_time( 'timestamp' );
            $expires_ts = strtotime( $expires . ' 23:59:59' );

            if ( $expires_ts && $now_ts > $expires_ts ) {
                // Cupón caducado
                return $totals;
            }
        }

        // Validar mínimo de carrito (si está definido)
        // Se compara contra el subtotal elegible (no contra todo el carrito)
        if ( $min_total !== '' && $min_total !== null ) {
            $min_total_f = (float) $min_total;
            if ( $min_total_f > 0 && $eligible_subtotal < $min_total_f ) {
                // No alcanza el mínimo
                return $totals;
            }
        }

        // Calcular descuento
        $discount = 0.0;

        if ( $type === 'percent' ) {
            $percent  = max( 0.0, $amount );
            $discount = round( $eligible_subtotal * ( $percent / 100 ), 2 );
        } elseif ( $type === 'fixed' ) {
            $discount = max( 0.0, $amount );
        }

        if ( $discount <= 0 ) {
            return $totals;
        }

        // No permitir que el descuento supere lo elegible ni el subtotal general
        if ( $discount > $eligible_subtotal ) {
            $discount = $eligible_subtotal;
        }
        if ( $discount > $subtotal ) {
            $discount = $subtotal;
        }

        // Si ya hubiera algún otro descuento, lo sumamos
        $prev_discount      = isset( $totals['discount'] ) ? (float) $totals['discount'] : 0.0;
        $totals['discount'] = $prev_discount + $discount;
        $totals['subtotal'] = $subtotal; // mostramos el subtotal tal cual
        $totals['total']    = max( 0, $subtotal + $shipping - $totals['discount'] );

        // Guardar contexto para mostrar en el carrito
        $_SESSION['shoplite_coupon_applied'] = [
            'code'        => $code_up,
            'global'      => empty( $products ),
            'product_ids' => empty( $products ) ? [] : array_values( $products ),
            'products'    => array_values( $applied_products ),
            'discount'    => $discount,
        ];

        // Normalizamos el código en sesión (mayúsculas)
        $_SESSION['shoplite_coupon_code'] = $code_up;

        return $totals;
    }

    /**
     * Aplica un cupón a un total simple (por ejemplo, para PayPal/Stripe),
     * devolviendo total nuevo + descuento + código aplicado.
     */
    public static function apply_coupon_to_total( float $total, ?string $force_code = null ): array {
        // Resultado por defecto (sin cupón)
        $result = [
            'total'    => $total,
            'discount' => 0.0,
            'code'     => null,
        ];

        if ( $total <= 0 ) {
            return $result;
        }

        if ( ! session_id() ) {
            @session_start();
        }

        // 1) Determinar código
        if ( $force_code !== null ) {
            $code = trim( (string) $force_code );
        } else {
            $code = isset( $_SESSION['shoplite_coupon_code'] )
                ? trim( (string) $_SESSION['shoplite_coupon_code'] )
                : '';
        }

        if ( $code === '' ) {
            return $result;
        }

        // Normalizar igual que al guardar
        $code = strtoupper( $code );

        // 2) Buscar cupón por código
        $coupon_posts = get_posts( [
            'post_type'      => self::POST_TYPE,
            'post_status'    => 'publish',
            'posts_per_page' => 1,
            'meta_query'     => [
                [
                    'key'   => self::META_CODE, // _shoplite_coupon_code
                    'value' => $code,
                ],
            ],
        ] );

        if ( empty( $coupon_posts ) ) {
            return $result; // Código no válido
        }

        $coupon_id = $coupon_posts[0]->ID;

        // Leer metadatos usando constantes
        $tipo      = get_post_meta( $coupon_id, self::META_TYPE, true ) ?: 'percent';
        $cantidad  = (float) get_post_meta( $coupon_id, self::META_AMOUNT, true );
        $min_cart  = (float) get_post_meta( $coupon_id, self::META_MIN_TOTAL, true );
        $fecha_cad = get_post_meta( $coupon_id, self::META_EXPIRES, true );
        $products  = get_post_meta( $coupon_id, self::META_PRODUCTS, true );
        if ( ! is_array( $products ) ) {
            $products = [];
        }

        // Normalizar tipo: aceptar 'fixed_cart' como fijo
        if ( $tipo === 'fixed_cart' ) {
            $tipo = 'fixed';
        }

        if ( $cantidad <= 0 ) {
            return $result;
        }

        // ================================
        // Subtotal elegible según productos
        // ================================
        $eligible_subtotal = 0.0;

        // Intentamos reconstruir el subtotal elegible desde el carrito en sesión
        $cart = isset( $_SESSION['shoplite_cart'] )
            ? $_SESSION['shoplite_cart']
            : ( isset( $_SESSION['carrito'] ) ? $_SESSION['carrito'] : [] );

        if ( ! is_array( $cart ) ) {
            $cart = [];
        }

        if ( empty( $products ) ) {
            // Cupón para todo el carrito → elegible = subtotal recibido
            $eligible_subtotal = $total;
        } else {
            foreach ( $cart as $line_key => $item ) {
                // Determinar product ID
                $pid = 0;
                if ( isset( $item['id'] ) ) {
                    $pid = (int) $item['id'];
                } elseif ( is_string( $line_key ) && preg_match( '/^tbp_(\d+)/', $line_key, $m ) ) {
                    $pid = (int) $m[1];
                }

                if ( $pid <= 0 || ! in_array( $pid, $products, true ) ) {
                    continue;
                }

                $price = isset( $item['precio'] )
                    ? (float) $item['precio']
                    : ( isset( $item['price'] ) ? (float) $item['price'] : 0.0 );

                $qty = isset( $item['cantidad'] )
                    ? max( 1, (int) $item['cantidad'] )
                    : ( isset( $item['qty'] ) ? max( 1, (int) $item['qty'] ) : 1 );

                $eligible_subtotal += $price * $qty;
            }
        }

        if ( $eligible_subtotal <= 0 ) {
            // Nada elegible → no aplicar
            return $result;
        }

        // Mínimo de carrito: lo comparamos con el subtotal elegible
        if ( $min_cart > 0 && $eligible_subtotal < $min_cart ) {
            return $result;
        }

        // Caducidad
        if ( $fecha_cad ) {
            $ts_cad = strtotime( $fecha_cad . ' 23:59:59' );
            if ( $ts_cad && $ts_cad < time() ) {
                return $result;
            }
        }

        // 3) Calcular descuento SOLO sobre el subtotal elegible
        if ( $tipo === 'fixed' ) {
            $discount = min( $cantidad, $eligible_subtotal );
        } else {
            // percent
            $discount = round( $eligible_subtotal * ( $cantidad / 100 ), 2 );
        }

        if ( $discount <= 0 ) {
            return $result;
        }

        // Nunca descontar más que el total completo
        if ( $discount > $total ) {
            $discount = $total;
        }

        $new_total = max( 0, $total - $discount );

        // Guardar en sesión por si hace falta en otros puntos
        $_SESSION['shoplite_coupon_total']    = $new_total;
        $_SESSION['shoplite_coupon_discount'] = $discount;
        $_SESSION['shoplite_coupon_code_ok']  = $code;

        $result['total']    = $new_total;
        $result['discount'] = $discount;
        $result['code']     = $code;

        return $result;
    }

}

