<?php
// shoplite-pro/includes/class-shoplite-pro-abandoned-engine.php
if ( ! defined( 'ABSPATH' ) ) exit;

class Shoplite_Pro_Abandoned_Engine {

    // Hook del cron para carritos abandonados
    const CRON_HOOK = 'shoplite_pro_abandoned_carts_cron';

    // Opciones internas (no hace falta exponerlas)
    const OPTION_LAST_RUN      = 'shoplite_pro_ac_last_cron_run';
    const TRANSIENT_RUN_LOCK   = 'shoplite_pro_ac_run_lock';

    public static function init() {

        // 1) Intervalo de cron “cada 5 minutos”
        add_filter( 'cron_schedules', [ __CLASS__, 'register_cron_interval' ] );

        // 2) Programar (o desprogramar) cron según opción
        // init se ejecuta también cuando WordPress carga vía wp-cron.php (cron real)
        add_action( 'init', [ __CLASS__, 'maybe_schedule_cron' ] );

        // 3) Lógica del cron (envío de emails)
        add_action( self::CRON_HOOK, [ __CLASS__, 'cron_send_emails' ] );

        // 4) Tracking de actividad del carrito (frontend)
        add_action( 'template_redirect', [ __CLASS__, 'track_activity' ], 20 );

        // 5) Fallback “self-heal”:
        // Si WP-Cron falla en un hosting de cliente, aprovechamos tráfico real para disparar el envío,
        // con lock para no ejecutar en cada visita.
        add_action( 'template_redirect', [ __CLASS__, 'maybe_run_fallback_sender' ], 25 );
    }

    /* ============================================================
     * Helpers
     * ============================================================ */

    private static function log( $msg ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG && function_exists( 'error_log' ) ) {
            error_log( '[shoplite][AC] ' . $msg );
        }
    }

    private static function is_enabled(): bool {
        return (bool) get_option( Shoplite_Pro_Abandoned_Carts::OPTION_ENABLED, 0 );
    }

    /**
     * Devuelve email válido desde user_id. Si no existe o no es válido, devuelve ''.
     */
    private static function get_user_email( int $user_id ): string {
        if ( $user_id <= 0 ) {
            return '';
        }
        $u = get_userdata( $user_id );
        if ( ! $u || empty( $u->user_email ) ) {
            return '';
        }
        $email = sanitize_email( (string) $u->user_email );
        if ( ! $email || ! is_email( $email ) ) {
            return '';
        }
        return $email;
    }

    private static function get_user_name( int $user_id ): string {
        if ( $user_id <= 0 ) {
            return '';
        }
        $u = get_userdata( $user_id );
        if ( ! $u ) {
            return '';
        }
        $name = '';
        if ( ! empty( $u->display_name ) ) {
            $name = (string) $u->display_name;
        } elseif ( ! empty( $u->user_login ) ) {
            $name = (string) $u->user_login;
        }
        return sanitize_text_field( $name );
    }

    private static function mark_last_run(): void {
        update_option( self::OPTION_LAST_RUN, time(), false );
    }

    private static function get_last_run(): int {
        return (int) get_option( self::OPTION_LAST_RUN, 0 );
    }

    /**
     * Lock para evitar que el fallback ejecute en cada visita.
     */
    private static function acquire_lock( int $seconds ): bool {
        if ( get_transient( self::TRANSIENT_RUN_LOCK ) ) {
            return false;
        }
        set_transient( self::TRANSIENT_RUN_LOCK, 1, $seconds );
        return true;
    }

    /* ============================================================
     * Cron: intervalo + programación
     * ============================================================ */

    /**
     * Registra un intervalo personalizado de cron: cada 5 minutos.
     */
    public static function register_cron_interval( $schedules ) {
        if ( ! isset( $schedules['shoplite_five_min'] ) ) {
            $schedules['shoplite_five_min'] = [
                'interval' => 5 * 60,
                'display'  => __( 'Cada 5 minutos (Shoplite)', 'shoplite-pro' ),
            ];
        }
        return $schedules;
    }

    /**
     * Programa el evento de cron si la función de carritos abandonados está activa.
     * Si está desactivada y el cron existe, lo desprograma.
     */
    public static function maybe_schedule_cron() {

        if ( ! self::is_enabled() ) {
            // Si está apagado, desprogramar para no gastar recursos
            $timestamp = wp_next_scheduled( self::CRON_HOOK );
            while ( $timestamp ) {
                wp_unschedule_event( $timestamp, self::CRON_HOOK );
                $timestamp = wp_next_scheduled( self::CRON_HOOK );
            }
            return;
        }

        if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
            // Nota: aunque el intervalo sea de 5 minutos, el envío real se rige por OPTION_DELAY_MIN.
            wp_schedule_event( time() + 5 * 60, 'shoplite_five_min', self::CRON_HOOK );
        }
    }

    /* ============================================================
     * Tracking de carritos en wp*_shoplite_abandoned_carts
     * ============================================================ */

    /**
 * Registra la actividad del carrito cuando ya conocemos el email del comprador.
 *
 * Nota:
 * - Ya NO dependemos de usuarios logueados.
 * - La fuente de verdad es $_SESSION['shoplite_customer_email'] (capturado en checkout).
 * - Guardamos el email en la tabla para poder enviar el recordatorio al comprador real.
 */
public static function track_activity() {
    if ( is_admin() ) {
        return;
    }

    // Función apagada → no registramos nada
    if ( ! self::is_enabled() ) {
        return;
    }

    // Asegurar sesión (solo para agrupar “sesión”; el envío NO depende de sesión)
    if ( function_exists( 'shoplite_session_start' ) ) {
        shoplite_session_start();
    } elseif ( ! session_id() ) {
        @session_start();
    }

    // Necesitamos email real del comprador (si no existe aún, no trackeamos)
    $customer_email = '';
    if ( ! empty( $_SESSION['shoplite_customer_email'] ) ) {
        $customer_email = sanitize_email( (string) $_SESSION['shoplite_customer_email'] );
    }
    if ( ! $customer_email || ! is_email( $customer_email ) ) {
        return;
    }

    // (Opcional) nombre si existe
    // $customer_name = '';
    // if ( ! empty( $_SESSION['shoplite_customer_name'] ) ) {
    //     $customer_name = sanitize_text_field( (string) $_SESSION['shoplite_customer_name'] );
    // }

    if ( empty( $_SESSION['shoplite_cart'] ) && empty( $_SESSION['carrito'] ) ) {
        return;
    }

    $cart = ! empty( $_SESSION['shoplite_cart'] )
        ? (array) $_SESSION['shoplite_cart']
        : (array) $_SESSION['carrito'];

    if ( empty( $cart ) ) {
        return;
    }

    // Calcular total del carrito
    $total = 0.0;
    foreach ( $cart as $item ) {
        $precio = isset( $item['precio'] )
            ? (float) $item['precio']
            : ( isset( $item['price'] ) ? (float) $item['price'] : 0 );
        $cant   = isset( $item['cantidad'] )
            ? (int) $item['cantidad']
            : ( isset( $item['qty'] ) ? (int) $item['qty'] : 1 );

        if ( $cant < 1 ) {
            $cant = 1;
        }
        $total += $precio * $cant;
    }

    if ( $total <= 0 ) {
        return;
    }

    // ID de sesión (si no existe, no registramos)
    $session_id = session_id();
    if ( ! $session_id ) {
        return;
    }

    global $wpdb;
    $table = $wpdb->prefix . 'shoplite_abandoned_carts';

    $cart_json = wp_json_encode( $cart );

    // Buscar fila existente para esta sesión (sin email enviado)
    $row = $wpdb->get_row(
        $wpdb->prepare(
            "SELECT id, email_sent FROM $table
             WHERE session_id = %s
             ORDER BY id DESC
             LIMIT 1",
            $session_id
        )
    );

    $data = [
        'session_id'  => $session_id,
        'user_id'     => 0,                 // ya no dependemos de WP users
        'email'       => $customer_email,    // comprador real
        'cart_json'   => $cart_json,
        'cart_total'  => $total,
        'last_update' => current_time( 'mysql' ),
    ];

    if ( $row && ! empty( $row->id ) && (int) $row->email_sent === 0 ) {
        $wpdb->update(
            $table,
            $data,
            [ 'id' => (int) $row->id ],
            [ '%s','%d','%s','%s','%f','%s' ],
            [ '%d' ]
        );
        self::log( sprintf( 'track update id=%d session=%s email=%s total=%.2f', (int) $row->id, $session_id, $customer_email, $total ) );
    } else {
        $wpdb->insert(
            $table,
            array_merge( $data, [ 'email_sent' => 0 ] ),
            [ '%s','%d','%s','%s','%f','%s','%d' ]
        );
        self::log( sprintf( 'track insert id=%d session=%s email=%s total=%.2f', (int) $wpdb->insert_id, $session_id, $customer_email, $total ) );
    }
}


    /* ============================================================
     * Fallback: si WP-Cron no funciona bien en el hosting del cliente
     * ============================================================ */

    /**
     * Si detectamos que el cron no se ha ejecutado en un tiempo y hay tráfico,
     * intentamos ejecutar el envío con un lock para no hacerlo en cada visita.
     *
     * Esto hace que funcione “en cualquier instalación” aunque WP-Cron sea irregular.
     */
    public static function maybe_run_fallback_sender() {
        if ( is_admin() ) {
            return;
        }

        if ( ! self::is_enabled() ) {
            return;
        }

        // Solo en frontend y evitando AJAX, REST, etc.
        if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
            return;
        }
        if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
            return;
        }

        // Si el cron se ejecutó hace poco, no hacemos nada.
        $last = self::get_last_run();
        $now  = time();

        // Umbral: si no ha corrido en 15 min, intentamos “self-heal”.
        if ( $last && ( $now - $last ) < ( 15 * MINUTE_IN_SECONDS ) ) {
            return;
        }

// Lock para no ejecutar esto en cada visita
if ( ! self::acquire_lock( 5 * MINUTE_IN_SECONDS ) ) {
    return;
}

self::log( 'fallback runner firing cron_send_emails()' );
self::cron_send_emails();
}

/* ============================================================
 * Cron: enviar emails
 * ============================================================ */

/**
 * Tarea programada: envía emails de recuperación de carrito abandonado.
 *
 * - NO depende de usuarios logueados.
 * - NO hay fallback a admin_email.
 * - El destinatario se lee de la columna "email" (capturada en checkout).
 * - No depende de columnas extra (portable).
 */
public static function cron_send_emails() {
    if ( ! self::is_enabled() ) {
        return;
    }

    // Marcar “latido” para diagnóstico/healthcheck
    self::mark_last_run();

    global $wpdb;
    $table = $wpdb->prefix . 'shoplite_abandoned_carts';

    $delay_min = (int) get_option( Shoplite_Pro_Abandoned_Carts::OPTION_DELAY_MIN, 30 );
    if ( $delay_min < 5 ) {
        $delay_min = 5;
    }

    $cutoff = gmdate( 'Y-m-d H:i:s', time() - $delay_min * 60 );

    $rows = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT id, email, cart_total, last_update
             FROM $table
             WHERE email_sent = 0
               AND cart_total > 0
               AND email IS NOT NULL
               AND email <> ''
               AND last_update <= %s
             ORDER BY id ASC
             LIMIT 20",
            $cutoff
        )
    );

    if ( empty( $rows ) ) {
        return;
    }

    // Plantillas de asunto y cuerpo
    $subject_tpl = get_option(
        Shoplite_Pro_Abandoned_Carts::OPTION_EMAIL_SUBJECT,
        __( '¿Olvidaste completar tu compra?', 'shoplite-pro' )
    );

    $body_tpl = get_option(
        Shoplite_Pro_Abandoned_Carts::OPTION_EMAIL_BODY,
        __( "Hola {{nombre}},\n\nVimos que dejaste productos en tu carrito y no llegaste a finalizar la compra.\nSi aún te interesan, puedes volver a tu carrito aquí:\n\n{{cart_link}}\n\n¡Gracias!", 'shoplite-pro' )
    );

    // URL del carrito (como en tu plugin core)
    $cart_url = get_option( 'shoplite_cart_page_url' );
    if ( ! $cart_url ) {
        $cart_url = home_url( '/carrito-de-compra/' );
    }

    foreach ( $rows as $row ) {

        $email = sanitize_email( (string) ( $row->email ?? '' ) );
        if ( ! $email || ! is_email( $email ) ) {
            self::log( sprintf( 'skip id=%d reason=invalid_email', (int) $row->id ) );
            continue;
        }

        // Portable: no dependemos de nombre en DB
        $name = __( 'customer', 'shoplite-pro' );

        // Enlace para volver al carrito (incluimos ID del registro)
        $link = add_query_arg(
            [ 'slac' => (int) $row->id ],
            $cart_url
        );

        $replacements = [
            '{{nombre}}'    => $name,
            '{{name}}'      => $name,
            '{{cart_link}}' => $link,
        ];

        $subject = strtr( (string) $subject_tpl, $replacements );
        $body    = strtr( (string) $body_tpl,    $replacements );

        // Email como HTML sencillo
        $headers   = [ 'Content-Type: text/html; charset=UTF-8' ];
        $body_html = nl2br( esc_html( $body ) );

        $sent = wp_mail( $email, $subject, $body_html, $headers );

        if ( $sent ) {
            $wpdb->update(
                $table,
                [
                    'email_sent'    => 1,
                    'email_sent_at' => current_time( 'mysql' ),
                ],
                [ 'id' => (int) $row->id ],
                [ '%d','%s' ],
                [ '%d' ]
            );
            self::log( sprintf( 'email id=%d to=%s result=OK', (int) $row->id, $email ) );
        } else {
            self::log( sprintf( 'email id=%d to=%s result=FAIL', (int) $row->id, $email ) );
        }
    }
}
}